# -*- coding: utf-8 -*-
"""
OCR优化器 - 按照用户要求的优先级顺序进行图片识别
"""

import os
from pathlib import Path
from typing import Optional, Tuple
from logger import logger

try:
    import pytesseract
    from PIL import Image
    HAS_TESSERACT = True
except ImportError:
    HAS_TESSERACT = False

try:
    from lib.ai import recognize_image_with_unstructured, test_unstructured_availability, HAS_UNSTRUCTURED
except ImportError:
    HAS_UNSTRUCTURED = False
    test_unstructured_availability = lambda: False
    recognize_image_with_unstructured = lambda *args, **kwargs: ""

try:
    from lib.ai import recognize_image_with_moonshot
except ImportError:
    recognize_image_with_moonshot = lambda *args, **kwargs: ""


class OCROptimizer:
    """OCR优化器 - 按照用户要求的优先级顺序进行图片识别
    
    在线模式（ChatGPT/DeepSeek等文本模型）的优先级：
    1. 多模态AI (Claude/GPT-4V/Gemini等) - 如果配置了
    2. Moonshot API (文件提取) - 如果没有配置多模态AI
    3. 用户配置的Tesseract OCR
    4. Unstructured 方法
    
    注意：ChatGPT/DeepSeek等文本模型不用于图片识别
    """
    
    def __init__(self):
        self.methods_order = [
            "multimodal_ai",  # 1. 多模态AI (优先)
            "moonshot_api",   # 2. Moonshot API (备用)
            "tesseract",      # 3. 用户配置的Tesseract
            "unstructured",   # 4. Unstructured
        ]
    
    def recognize_image_with_priority(self, image_path: str, max_chars: int = 5000, 
                                    api_key: str = '', is_local: bool = False,
                                    image_recognition_method: str = "tesseract",
                                    online_model: str = None, api_base: str = None, model_name: str = None) -> Tuple[str, str]:
        """
        按照优先级顺序识别图片
        
        Args:
            image_path: 图片路径
            max_chars: 最大字符数
            api_key: API密钥
            is_local: 是否本地模型
            image_recognition_method: 用户配置的识别方法
            online_model: 在线模型类型 (chatgpt/deepseek/None)
            
        Returns:
            Tuple[str, str]: (识别结果文本, 识别方法)
        """
        
        logger.info(f"🔍 开始按优先级顺序识别图片: {os.path.basename(image_path)}")
        logger.info(f"🎯 模型类型: {online_model}, 本地模型: {is_local}")
        
        # 🚀 根据模型类型使用不同的识别优先级
        if is_local:
            logger.info("🏠 本地模型识别优先级: 多模态AI -> 用户自定义API -> Tesseract -> Unstructured")
        else:
            logger.info("🌙 在线模型识别优先级: 多模态AI -> Moonshot API -> Tesseract -> Unstructured")
            if online_model in ['chatgpt', 'deepseek']:
                logger.info(f"💡 注意：{online_model}为文本模型，不用于图片识别，将使用多模态AI或Moonshot")
        
        # 1. 在线模式（包括ChatGPT/DeepSeek）：优先尝试多模态AI
        if not is_local:
            logger.info("🤖 在线模式：优先尝试使用多模态AI服务进行图片识别")
            text, method = self._try_multimodal_ai(image_path)
            if text:
                return text, method
            
            # 如果多模态AI失败或未配置，尝试Moonshot API作为备用
            if api_key:
                logger.info("🌙 多模态AI未配置或失败，尝试使用Moonshot API作为备用")
                text, method = self._try_moonshot_api(image_path, api_key, max_chars)
                if text:
                    return text, method
            else:
                logger.info("⚠️  未提供API密钥，无法使用Moonshot API")
        
        # 2. 本地模式：优先使用多模态AI服务
        if is_local:
            logger.info("🏠 本地模式：优先尝试使用多模态AI服务进行图片识别")
            text, method = self._try_multimodal_ai(image_path)
            if text:
                return text, method
            
            # 如果多模态AI失败，尝试用户自定义API服务（作为备用）
            logger.info("🏠 本地模式：多模态AI失败，尝试使用用户自定义API服务")
            text, method = self._try_local_model_recognition(image_path, api_key, api_base, model_name, max_chars)
            if text:
                return text, method
        
        # 3. 使用用户配置的Tesseract OCR
        text, method = self._try_tesseract(image_path, image_recognition_method)
        if text:
            return text, method
        
        # 4. 如果Tesseract失败，尝试Unstructured
        text, method = self._try_unstructured(image_path, max_chars)
        if text:
            return text, method
        
        # 5. 所有方法都失败，直接跳过
        logger.warning(f"❌ 图片 {os.path.basename(image_path)} 所有识别方法均失败")
        logger.info("🔄 所有识别方法均失败，将跳过此文件")
        return "", "识别失败"
    
    def _try_tesseract(self, image_path: str, image_recognition_method: str) -> Tuple[str, str]:
        """尝试使用Tesseract OCR"""
        if not HAS_TESSERACT:
            logger.info("⚠️  pytesseract未安装，跳过Tesseract方法")
            return "", ""
        
        if image_recognition_method not in ["tesseract", "both"]:
            logger.info("🔧 用户未配置Tesseract方法，跳过")
            return "", ""
        
        processed_image_path = None
        try:
            logger.info("🎯 优先使用Tesseract OCR（用户配置）")
            
            # 🚀 优化：使用增强版图片预处理提高识别率
            try:
                from lib.ai import preprocess_image_for_ocr
                processed_image_path = preprocess_image_for_ocr(image_path)
                logger.info(f"图片预处理完成: {image_path} -> {processed_image_path}")
            except Exception as preprocess_e:
                logger.warning(f"图片预处理失败，使用原始图片: {str(preprocess_e)}")
                processed_image_path = image_path
            
            # 使用预处理后的图片进行OCR
            img = Image.open(processed_image_path)
            tesseract_text = pytesseract.image_to_string(img, lang='chi_sim+eng')
            img.close()
            
            if tesseract_text and len(tesseract_text.strip()) > 5:
                logger.info(f"✅ Tesseract识别成功，内容长度: {len(tesseract_text)} 字符")
                logger.info(f"【Tesseract识别内容】 {os.path.basename(image_path)}:\n{tesseract_text}")
                return tesseract_text, "Tesseract OCR (用户配置)"
            else:
                logger.info("⚠️  Tesseract识别结果为空或过短")
                return "", ""
                
        except Exception as e:
            logger.warning(f"❌ Tesseract识别失败: {str(e)}")
            return "", ""
        finally:
            # 🚀 优化：清理临时预处理文件
            if processed_image_path and processed_image_path != image_path:
                try:
                    if os.path.exists(processed_image_path):
                        os.remove(processed_image_path)
                        logger.debug(f"成功删除临时预处理文件: {processed_image_path}")
                except Exception as cleanup_e:
                    logger.warning(f"清理临时预处理文件失败: {str(cleanup_e)}")
    
    def _try_unstructured(self, image_path: str, max_chars: int) -> Tuple[str, str]:
        """尝试使用Unstructured"""
        if not HAS_UNSTRUCTURED or not test_unstructured_availability():
            logger.info("⚠️  Unstructured未安装或不可用，跳过")
            return "", ""
        
        try:
            logger.info("🔄 Tesseract失败，尝试使用Unstructured")
            unstructured_text = recognize_image_with_unstructured(image_path, max_chars)
            
            if unstructured_text and len(unstructured_text.strip()) > 10:
                logger.info(f"✅ Unstructured识别成功，内容长度: {len(unstructured_text)} 字符")
                logger.info(f"【Unstructured识别内容】 {os.path.basename(image_path)}:\n{unstructured_text}")
                return unstructured_text, "Unstructured (备用方案)"
            else:
                logger.info("⚠️  Unstructured识别结果为空或过短")
                return "", ""
                
        except Exception as e:
            logger.warning(f"❌ Unstructured识别失败: {str(e)}")
            return "", ""
    
    def _try_multimodal_ai(self, image_path: str) -> Tuple[str, str]:
        """尝试使用多模态AI"""
        try:
            from lib.multimodal_ai import multimodal_service
            
            # 检查多模态AI服务是否可用
            if not multimodal_service.is_available():
                logger.warning("❌ 多模态AI服务未配置或不可用，尝试重新加载配置")
                
                # 尝试重新加载配置
                multimodal_service.reload_config()
                
                # 重新检查是否可用
                if not multimodal_service.is_available():
                    logger.warning("❌ 重新加载后多模态AI服务仍不可用")
                
                # 详细检查配置信息
                logger.info("🔍 多模态AI配置检查:")
                
                # 检查依赖
                try:
                    import requests, base64
                    from PIL import Image
                    logger.info("  ✅ 基本依赖: requests, base64, PIL 已安装")
                except ImportError as dep_e:
                    logger.warning(f"  ❌ 基本依赖缺失: {str(dep_e)}")
                
                # 检查API密钥配置
                logger.info(f"  - API密钥字典: {'已配置' if multimodal_service.api_keys else '未配置'}")
                if multimodal_service.api_keys:
                    for key, value in multimodal_service.api_keys.items():
                        logger.info(f"    - {key}: {'已配置' if value else '未配置'}")
                    
                    # 特别检查 multimodal 提供商
                    if 'multimodal' in multimodal_service.api_keys:
                        has_api_base = bool(multimodal_service.api_base)
                        has_model = bool(multimodal_service.model_name)
                        logger.info(f"  - multimodal 提供商完整性检查:")
                        logger.info(f"    - API密钥: ✅")
                        logger.info(f"    - API地址: {'✅' if has_api_base else '❌'}")
                        logger.info(f"    - 模型名称: {'✅' if has_model else '❌'}")
                        logger.info(f"    - 整体可用性: {'✅' if has_api_base and has_model else '❌'}")
                else:
                    logger.warning("  ❌ 没有配置任何API密钥")
                
                # 检查API配置
                logger.info(f"  - API地址: {multimodal_service.api_base or '未配置'}")
                logger.info(f"  - 模型名称: {multimodal_service.model_name or '未配置'}")
                
                # 检查配置文件
                try:
                    config_file = os.path.join(os.path.expanduser('~'), '.fileneatai', 'multimodal_settings.json')
                    logger.info(f"  - 配置文件: {config_file}")
                    logger.info(f"  - 配置文件存在: {'是' if os.path.exists(config_file) else '否'}")
                    
                    if os.path.exists(config_file):
                        try:
                            import json
                            with open(config_file, 'r', encoding='utf-8') as f:
                                settings = json.load(f)
                            multimodal_config = settings.get('multimodal', {})
                            logger.info(f"  - 多模态配置启用: {'是' if multimodal_config.get('enabled') else '否'}")
                            logger.info(f"  - 配置文件中的API密钥: {'已配置' if multimodal_config.get('api_key') else '未配置'}")
                            logger.info(f"  - 配置文件中的API地址: {multimodal_config.get('api_base', '未配置')}")
                            logger.info(f"  - 配置文件中的模型: {multimodal_config.get('model', '未配置')}")
                        except Exception as config_e:
                            logger.warning(f"  ❌ 读取配置文件失败: {str(config_e)}")
                except Exception as file_e:
                    logger.warning(f"  ❌ 检查配置文件失败: {str(file_e)}")
                
                return "", ""
            
            logger.info("🤖 尝试使用多模态AI进行图片识别")
            success, content, cost, provider = multimodal_service.recognize_image_smart(
                image_path=image_path,
                prompt="请详细识别这张图片中的所有文字内容，包括中文和英文。"
            )
            
            if success and content and len(content.strip()) > 10:
                logger.info(f"✅ 多模态AI识别成功，服务商: {provider}，成本: ${cost:.4f}")
                logger.info(f"【多模态AI识别内容】 {os.path.basename(image_path)}:\n{content}")
                return content, f"多模态AI ({provider})"
            else:
                # 详细显示失败信息，就像用户自定义API服务那样
                logger.error(f"❌ 多模态AI识别失败: {content}")
                logger.error(f"  - 成功状态: {success}")
                logger.error(f"  - 内容长度: {len(content) if content else 0}")
                logger.error(f"  - 服务商: {provider}")
                logger.error(f"  - 成本: ${cost:.4f}")
                if content and not success:
                    # 如果有错误内容，显示完整错误信息
                    logger.error(f"  - 详细错误: {content}")
                return "", ""
                
        except ImportError as e:
            logger.error(f"❌ 多模态AI服务导入失败: {str(e)}")
            return "", ""
        except Exception as e:
            logger.error(f"❌ 多模态AI识别过程中出现异常: {str(e)}")
            logger.error(f"  - 异常类型: {type(e).__name__}")
            import traceback
            logger.error(f"  - 详细堆栈: {traceback.format_exc()}")
            return "", ""
    
    def _try_local_model_recognition(self, image_path: str, api_key: str, api_base: str, model_name: str, max_chars: int) -> Tuple[str, str]:
        """尝试使用本地模式的用户自定义API服务进行图片识别"""
        if not api_key or not api_base or not model_name:
            logger.info("⚠️  本地模式缺少API配置信息，跳过自定义API服务识别")
            return "", ""
        
        try:
            from lib.ai import recognize_image_with_online_model
            logger.info(f"🏠 本地模式：使用用户自定义API服务识别图片")
            logger.info(f"🔧 API服务: {api_base}, 模型: {model_name}")
            
            result_text = recognize_image_with_online_model(
                image_path=image_path,
                api_key=api_key,
                api_base=api_base,
                model_name=model_name,
                max_chars=max_chars
            )
            
            if result_text and len(result_text.strip()) > 10:
                logger.info(f"✅ 本地自定义API服务识别成功，内容长度: {len(result_text)} 字符")
                logger.info(f"📄 识别结果：\n{result_text}")
                return result_text, f"本地自定义API服务 ({model_name})"
            else:
                logger.info(f"⚠️  本地自定义API服务识别结果为空或过短")
                return "", ""
                
        except Exception as e:
            logger.warning(f"❌ 本地自定义API服务识别失败: {str(e)}")
            return "", ""
    
    # ❌ 已移除 _try_online_model 方法
    # ChatGPT/DeepSeek等文本模型不适合图片识别，已改为使用多模态AI或Moonshot API
    
    def _try_moonshot_api(self, image_path: str, api_key: str, max_chars: int) -> Tuple[str, str]:
        """尝试使用Moonshot API - 第一优先级文字提取"""
        
        # 🚀 优先使用传入的API密钥，如果没有则从配置文件中获取Moonshot专用密钥
        moonshot_api_key = api_key  # 优先使用传入的密钥
        
        # 如果传入的密钥为空，尝试从多模态设置中获取专门的Moonshot密钥
        if not moonshot_api_key:
            moonshot_api_key = self._get_moonshot_api_key()
            
        if not moonshot_api_key:
            logger.info("⚠️  未提供API密钥且未配置Moonshot专用API密钥，跳过Moonshot API")
            logger.info("💡 提示：请传入API密钥或在多模态设置中配置Moonshot API密钥")
            return "", ""
        
        try:
            logger.info("🌙 【第一优先级】使用 Moonshot API 提取图片文字")
            logger.info(f"🔑 使用的API密钥: [已配置]")
            
            moonshot_text = recognize_image_with_moonshot(image_path, moonshot_api_key, max_chars)
            
            if moonshot_text and len(moonshot_text.strip()) > 10:
                logger.info(f"✅ Moonshot API 文字提取成功，内容长度: {len(moonshot_text)} 字符")
                logger.info(f"📄 Moonshot API 提取的文字内容：\n{moonshot_text}")
                return moonshot_text, "Moonshot API (第一优先级)"
            else:
                logger.info("⚠️  Moonshot API 未提取到有效文字内容")
                return "", ""
                
        except Exception as e:
            error_msg = str(e)
            if "401" in error_msg or "Invalid Authentication" in error_msg:
                logger.error(f"❌ Moonshot API 认证失败，请检查API密钥是否正确: {error_msg}")
                logger.info("💡 解决方案：请检查传入的API密钥是否为有效的Moonshot API密钥")
            else:
                logger.warning(f"❌ Moonshot API 文字提取失败: {error_msg}")
            return "", ""
    
    def _get_moonshot_api_key(self) -> str:
        """获取Moonshot专用API密钥"""
        try:
            import json
            
            # 从多模态设置文件中读取Moonshot密钥
            config_file = os.path.join(os.path.expanduser('~'), '.fileneatai', 'multimodal_settings.json')
            if os.path.exists(config_file):
                with open(config_file, 'r', encoding='utf-8') as f:
                    settings = json.load(f)
                
                # 检查是否配置了Moonshot API
                multimodal_config = settings.get('multimodal', {})
                if multimodal_config.get('enabled', False):
                    api_base = multimodal_config.get('api_base', '').strip()
                    api_key = multimodal_config.get('api_key', '').strip()
                    
                    # 判断是否是Moonshot API配置
                    if 'moonshot.cn' in api_base.lower() and api_key:
                        logger.info("✅ 找到Moonshot专用API密钥配置")
                        return api_key
                    elif api_key and not api_base:
                        # 如果只配置了密钥没有配置地址，假设是Moonshot
                        logger.info("✅ 使用多模态设置中的API密钥作为Moonshot密钥")
                        return api_key
            
            logger.info("⚠️  未找到Moonshot专用API密钥配置")
            return ""
            
        except Exception as e:
            logger.warning(f"读取Moonshot API密钥配置失败: {str(e)}")
            return ""


# 全局实例
ocr_optimizer = OCROptimizer()
