"""
系统环境检查模块
在程序启动时检测系统版本和依赖项，特别针对 Windows 7/8 用户
"""
import platform
import sys
import os
import ctypes
from pathlib import Path


class SystemChecker:
    """系统环境检查器"""
    
    def __init__(self):
        self.os_version = None
        self.is_win7 = False
        self.is_win8 = False
        self.is_old_windows = False
        self.ucrt_installed = False
        self.vcruntime_installed = False
        self._detect_system()
    
    def _detect_system(self):
        """检测系统版本"""
        try:
            self.os_version = platform.version()
            release = platform.release()
            
            # Windows 版本判断
            # Windows 7 = 6.1, Windows 8 = 6.2, Windows 8.1 = 6.3, Windows 10 = 10.0
            version_info = sys.getwindowsversion()
            major = version_info.major
            minor = version_info.minor
            
            if major == 6 and minor == 1:
                self.is_win7 = True
                self.is_old_windows = True
            elif major == 6 and minor in [2, 3]:
                self.is_win8 = True
                self.is_old_windows = True
            
        except Exception as e:
            print(f"检测系统版本失败: {e}")
    
    def _check_ucrt(self):
        """检查 Universal CRT 是否安装"""
        try:
            # 尝试加载 UCRT 的关键 DLL
            ucrt_dlls = [
                "api-ms-win-crt-runtime-l1-1-0.dll",
                "api-ms-win-crt-stdio-l1-1-0.dll",
                "api-ms-win-core-path-l1-1-0.dll",
            ]
            
            for dll_name in ucrt_dlls:
                try:
                    dll = ctypes.CDLL(dll_name)
                    if dll:
                        self.ucrt_installed = True
                        return True
                except OSError:
                    continue
            
            return False
        except Exception as e:
            print(f"检查 UCRT 失败: {e}")
            return False
    
    def _check_vcruntime(self):
        """检查 Visual C++ Runtime 是否安装"""
        try:
            # 检查 vcruntime140.dll
            vcruntime_dlls = [
                "vcruntime140.dll",
                "vcruntime140_1.dll",
                "msvcp140.dll",
            ]
            
            # 先检查当前目录（打包的 DLL）
            for dll_name in vcruntime_dlls:
                if os.path.exists(dll_name):
                    self.vcruntime_installed = True
                    return True
            
            # 再检查系统目录
            for dll_name in vcruntime_dlls:
                try:
                    dll = ctypes.CDLL(dll_name)
                    if dll:
                        self.vcruntime_installed = True
                        return True
                except OSError:
                    continue
            
            return False
        except Exception as e:
            print(f"检查 VC++ Runtime 失败: {e}")
            return False
    
    def check_all(self):
        """执行完整检查"""
        self._check_ucrt()
        self._check_vcruntime()
        
        return {
            "os_version": self.os_version,
            "is_win7": self.is_win7,
            "is_win8": self.is_win8,
            "is_old_windows": self.is_old_windows,
            "ucrt_installed": self.ucrt_installed,
            "vcruntime_installed": self.vcruntime_installed,
            "needs_action": self.is_old_windows and not self.ucrt_installed,
        }
    
    def get_system_name(self):
        """获取系统名称"""
        if self.is_win7:
            return "Windows 7"
        elif self.is_win8:
            return "Windows 8/8.1"
        else:
            return f"Windows {platform.release()}"
    
    def get_fix_guide_path(self):
        """获取修复指南路径"""
        base_paths = [
            "运行环境问题完全解决方案.txt",
            "Windows7-8用户必读.txt",
            "../运行环境问题完全解决方案.txt",
            "../Windows7-8用户必读.txt",
        ]
        
        for path in base_paths:
            if os.path.exists(path):
                return os.path.abspath(path)
        
        return None
    
    def get_fix_script_path(self):
        """获取一键修复脚本路径"""
        base_paths = [
            "一键修复运行环境.bat",
            "../一键修复运行环境.bat",
        ]
        
        for path in base_paths:
            if os.path.exists(path):
                return os.path.abspath(path)
        
        return None


def check_system_on_startup():
    """
    程序启动时的系统检查
    返回: (需要显示警告, 检查结果字典)
    """
    checker = SystemChecker()
    result = checker.check_all()
    
    # 如果是 Windows 7/8 且缺少 UCRT，需要警告
    if result["needs_action"]:
        return True, result, checker
    
    return False, result, checker


def get_warning_message(checker):
    """
    生成警告信息
    """
    system_name = checker.get_system_name()
    
    message = f"""检测到您正在使用 {system_name} 系统

为了确保程序正常运行，您需要安装必要的系统组件：
• Visual C++ 运行时库
• Windows 更新补丁 KB2999226

请选择以下操作：

1. 【推荐】点击"立即修复"，自动安装必要组件（需要管理员权限）
2. 点击"查看指南"，手动按步骤操作
3. 点击"忽略"继续运行（可能会遇到错误）

注意：安装完成后需要重启程序才能生效。"""
    
    return message


def get_kb_download_url():
    """获取 KB2999226 下载地址"""
    return "https://www.microsoft.com/zh-cn/download/details.aspx?id=49984"


def open_fix_guide(checker):
    """打开修复指南"""
    guide_path = checker.get_fix_guide_path()
    if guide_path:
        try:
            os.startfile(guide_path)
            return True
        except Exception as e:
            print(f"打开指南失败: {e}")
            return False
    return False


def run_fix_script(checker):
    """运行一键修复脚本"""
    script_path = checker.get_fix_script_path()
    if script_path:
        try:
            # 以管理员权限运行
            ctypes.windll.shell32.ShellExecuteW(
                None, 
                "runas",  # 请求管理员权限
                script_path, 
                None, 
                None, 
                1  # SW_SHOWNORMAL
            )
            return True
        except Exception as e:
            print(f"运行修复脚本失败: {e}")
            return False
    return False


if __name__ == "__main__":
    # 测试代码
    checker = SystemChecker()
    result = checker.check_all()
    
    print("=" * 60)
    print("系统环境检查")
    print("=" * 60)
    print(f"系统版本: {checker.get_system_name()}")
    print(f"是否 Windows 7: {result['is_win7']}")
    print(f"是否 Windows 8: {result['is_win8']}")
    print(f"是否旧版 Windows: {result['is_old_windows']}")
    print(f"UCRT 已安装: {result['ucrt_installed']}")
    print(f"VC++ Runtime 已安装: {result['vcruntime_installed']}")
    print(f"需要用户操作: {result['needs_action']}")
    print()
    
    if result['needs_action']:
        print("⚠️ 警告：检测到系统缺少必要组件")
        print(get_warning_message(checker))
    else:
        print("✅ 系统环境正常")

