"""
Windows 7/8 系统警告对话框
"""
from PySide6.QtCore import Qt
from PySide6.QtWidgets import QDialog, QVBoxLayout, QHBoxLayout, QLabel, QPushButton, QTextBrowser
from PySide6.QtGui import QIcon, QPixmap


class SystemWarningDialog(QDialog):
    """系统环境警告对话框"""
    
    def __init__(self, system_name, message, parent=None):
        super().__init__(parent)
        self.result_action = None  # 用户选择的操作：'fix', 'guide', 'ignore'
        self.system_name = system_name
        self.message = message
        self.setupUI()
    
    def setupUI(self):
        self.setWindowTitle("系统环境检查")
        self.setModal(True)
        self.setMinimumWidth(500)
        self.setMinimumHeight(400)
        
        # 设置窗口样式
        self.setStyleSheet("""
            QDialog {
                background-color: #f5f5f5;
            }
            QLabel#titleLabel {
                color: #d84315;
                font-size: 16px;
                font-weight: bold;
                padding: 10px;
            }
            QTextBrowser {
                background-color: white;
                border: 1px solid #ddd;
                border-radius: 4px;
                padding: 10px;
                font-size: 13px;
                color: #333;
            }
            QPushButton {
                padding: 8px 20px;
                font-size: 13px;
                border-radius: 4px;
                min-width: 100px;
            }
            QPushButton#fixButton {
                background-color: #4CAF50;
                color: white;
                border: none;
                font-weight: bold;
            }
            QPushButton#fixButton:hover {
                background-color: #45a049;
            }
            QPushButton#guideButton {
                background-color: #2196F3;
                color: white;
                border: none;
            }
            QPushButton#guideButton:hover {
                background-color: #0b7dda;
            }
            QPushButton#ignoreButton {
                background-color: #757575;
                color: white;
                border: none;
            }
            QPushButton#ignoreButton:hover {
                background-color: #616161;
            }
        """)
        
        # 主布局
        layout = QVBoxLayout(self)
        layout.setSpacing(15)
        layout.setContentsMargins(20, 20, 20, 20)
        
        # 标题
        title_label = QLabel("⚠️ 检测到旧版 Windows 系统")
        title_label.setObjectName("titleLabel")
        title_label.setAlignment(Qt.AlignCenter)
        layout.addWidget(title_label)
        
        # 消息内容（使用 QTextBrowser 支持富文本）
        message_browser = QTextBrowser()
        message_browser.setOpenExternalLinks(False)
        message_browser.setHtml(self._format_message())
        message_browser.setMaximumHeight(250)
        layout.addWidget(message_browser)
        
        # 按钮区域
        button_layout = QHBoxLayout()
        button_layout.setSpacing(10)
        
        # 立即修复按钮
        fix_button = QPushButton("🔧 立即修复（推荐）")
        fix_button.setObjectName("fixButton")
        fix_button.clicked.connect(self._on_fix_clicked)
        button_layout.addWidget(fix_button)
        
        # 查看指南按钮
        guide_button = QPushButton("📖 查看指南")
        guide_button.setObjectName("guideButton")
        guide_button.clicked.connect(self._on_guide_clicked)
        button_layout.addWidget(guide_button)
        
        # 忽略按钮
        ignore_button = QPushButton("忽略继续")
        ignore_button.setObjectName("ignoreButton")
        ignore_button.clicked.connect(self._on_ignore_clicked)
        button_layout.addWidget(ignore_button)
        
        layout.addLayout(button_layout)
        
        # 底部说明
        note_label = QLabel("提示：修复完成后请重启程序")
        note_label.setStyleSheet("color: #666; font-size: 11px;")
        note_label.setAlignment(Qt.AlignCenter)
        layout.addWidget(note_label)
    
    def _format_message(self):
        """格式化消息为 HTML"""
        html = f"""
        <div style="font-family: 'Microsoft YaHei', Arial; line-height: 1.6;">
            <p style="font-size: 14px; margin-bottom: 10px;">
                <strong>系统版本：</strong>{self.system_name}
            </p>
            
            <p style="margin-bottom: 10px;">
                为了确保程序正常运行，需要安装以下系统组件：
            </p>
            
            <ul style="margin-left: 20px; margin-bottom: 15px;">
                <li><strong>Visual C++ 运行时库</strong></li>
                <li><strong>Windows 更新补丁 KB2999226</strong></li>
            </ul>
            
            <p style="background-color: #fff3cd; padding: 10px; border-left: 4px solid #ffc107; margin-bottom: 10px;">
                <strong>⚠️ 重要：</strong>如果不安装这些组件，程序可能无法启动或运行时出错。
            </p>
            
            <p style="margin-bottom: 10px;">
                <strong>推荐操作：</strong>
            </p>
            <ol style="margin-left: 20px;">
                <li>点击 <strong>"立即修复"</strong> 按钮，自动安装（需要管理员权限）</li>
                <li>或点击 <strong>"查看指南"</strong>，按步骤手动安装</li>
            </ol>
        </div>
        """
        return html
    
    def _on_fix_clicked(self):
        """点击立即修复"""
        self.result_action = 'fix'
        self.accept()
    
    def _on_guide_clicked(self):
        """点击查看指南"""
        self.result_action = 'guide'
        self.accept()
    
    def _on_ignore_clicked(self):
        """点击忽略"""
        self.result_action = 'ignore'
        self.accept()
    
    def get_user_action(self):
        """获取用户选择的操作"""
        return self.result_action


# 简化版对话框（如果需要）
class SimpleSystemWarningDialog(QDialog):
    """简化版系统警告对话框"""
    
    def __init__(self, message, parent=None):
        super().__init__(parent)
        self.result_action = None
        self.message = message
        self.setupUI()
    
    def setupUI(self):
        self.setWindowTitle("系统环境警告")
        self.setModal(True)
        self.resize(450, 250)
        
        layout = QVBoxLayout(self)
        layout.setSpacing(20)
        layout.setContentsMargins(20, 20, 20, 20)
        
        # 消息
        message_label = QLabel(self.message)
        message_label.setWordWrap(True)
        message_label.setStyleSheet("font-size: 13px; color: #333;")
        layout.addWidget(message_label)
        
        # 按钮
        button_layout = QHBoxLayout()
        
        fix_button = QPushButton("立即修复")
        fix_button.clicked.connect(lambda: self._set_action('fix'))
        button_layout.addWidget(fix_button)
        
        guide_button = QPushButton("查看指南")
        guide_button.clicked.connect(lambda: self._set_action('guide'))
        button_layout.addWidget(guide_button)
        
        ignore_button = QPushButton("忽略")
        ignore_button.clicked.connect(lambda: self._set_action('ignore'))
        button_layout.addWidget(ignore_button)
        
        layout.addLayout(button_layout)
    
    def _set_action(self, action):
        self.result_action = action
        self.accept()
    
    def get_user_action(self):
        return self.result_action


if __name__ == "__main__":
    import sys
    from PySide6.QtWidgets import QApplication
    
    app = QApplication(sys.argv)
    
    test_message = """检测到您正在使用 Windows 7 系统

为了确保程序正常运行，您需要安装必要的系统组件"""
    
    dialog = SystemWarningDialog("Windows 7 SP1", test_message)
    if dialog.exec_() == QDialog.Accepted:
        print(f"用户选择: {dialog.get_user_action()}")
    
    sys.exit()

