# -*- coding: utf-8 -*-
"""
视频处理模块 - 关键帧提取和内容分析
"""

import os
import sys
import json
import tempfile
import subprocess
import shutil
from pathlib import Path
from typing import List, Optional, Tuple, Dict
from logger import logger


class FFmpegManager:
    """FFmpeg管理器 - 负责下载和管理ffmpeg"""
    
    def __init__(self):
        self.config_dir = self._get_config_dir()
        self.ffmpeg_dir = os.path.join(self.config_dir, 'ffmpeg')
        self.ffmpeg_path = self._get_ffmpeg_path()
        
    def _get_config_dir(self) -> str:
        """获取配置目录"""
        config_dir = os.path.join(os.path.expanduser('~'), '.fileneatai')
        os.makedirs(config_dir, exist_ok=True)
        return config_dir
    
    def _get_ffmpeg_path(self) -> str:
        """获取ffmpeg可执行文件路径"""
        if sys.platform == 'win32':
            return os.path.join(self.ffmpeg_dir, 'bin', 'ffmpeg.exe')
        else:
            return os.path.join(self.ffmpeg_dir, 'bin', 'ffmpeg')
    
    def is_installed(self) -> bool:
        """检查ffmpeg是否已安装"""
        # 先检查本地安装
        if os.path.exists(self.ffmpeg_path):
            return True
        
        # 检查系统PATH中是否有ffmpeg
        try:
            result = subprocess.run(
                ['ffmpeg', '-version'],
                capture_output=True,
                timeout=5,
                encoding='utf-8',
                errors='ignore'
            )
            return result.returncode == 0
        except:
            return False
    
    def get_ffmpeg_executable(self) -> Optional[str]:
        """获取可用的ffmpeg可执行文件路径"""
        # 优先使用本地安装的
        if os.path.exists(self.ffmpeg_path):
            return self.ffmpeg_path
        
        # 尝试使用系统PATH中的 - 获取完整路径
        try:
            # 使用shutil.which获取系统PATH中ffmpeg的完整路径
            system_ffmpeg = shutil.which('ffmpeg')
            if system_ffmpeg:
                # 验证是否能运行
                result = subprocess.run(
                    [system_ffmpeg, '-version'],
                    capture_output=True,
                    timeout=5,
                    encoding='utf-8',
                    errors='ignore'
                )
                if result.returncode == 0:
                    return system_ffmpeg  # 返回完整路径
        except:
            pass
        
        return None
    
    def download_ffmpeg(self, progress_callback=None) -> Tuple[bool, str]:
        """
        下载ffmpeg和ffprobe
        
        Args:
            progress_callback: 进度回调函数 callback(percent, message)
            
        Returns:
            (success, message)
        """
        try:
            import requests
            
            os.makedirs(self.ffmpeg_dir, exist_ok=True)
            
            # 根据平台选择下载链接
            if sys.platform == 'win32':
                # Windows 64位版本 - 使用CDN直接下载exe文件
                files_to_download = [
                    {
                        'url': "https://longstore.fileneatapi.top/filezen/ffmpeg.exe",
                        'filename': "ffmpeg.exe",
                        'description': "ffmpeg主程序"
                    },
                    {
                        'url': "https://longstore.fileneatapi.top/filezen/ffprobe.exe",
                        'filename': "ffprobe.exe",
                        'description': "ffprobe元数据工具"
                    }
                ]
            elif sys.platform == 'darwin':
                # macOS
                if progress_callback:
                    progress_callback(0, "macOS请使用 brew install ffmpeg 安装")
                return False, "macOS请使用 Homebrew 安装: brew install ffmpeg"
            else:
                # Linux
                if progress_callback:
                    progress_callback(0, "Linux请使用包管理器安装")
                return False, "Linux请使用包管理器安装: sudo apt install ffmpeg"
            
            # 创建bin目录
            bin_dir = os.path.join(self.ffmpeg_dir, 'bin')
            os.makedirs(bin_dir, exist_ok=True)
            
            # 下载所有文件
            total_files = len(files_to_download)
            for index, file_info in enumerate(files_to_download):
                download_url = file_info['url']
                filename = file_info['filename']
                description = file_info['description']
                download_path = os.path.join(bin_dir, filename)
                
                # 计算当前文件的进度范围
                base_progress = int((index / total_files) * 90)
                file_progress_range = int(90 / total_files)
                
                # 下载文件
                if progress_callback:
                    progress_callback(base_progress + 5, f"开始下载{description}...")
                
                logger.info(f"开始下载{description}: {download_url}")
                
                response = requests.get(download_url, stream=True, timeout=300)
                total_size = int(response.headers.get('content-length', 0))
                
                downloaded_size = 0
                with open(download_path, 'wb') as f:
                    for chunk in response.iter_content(chunk_size=8192):
                        if chunk:
                            f.write(chunk)
                            downloaded_size += len(chunk)
                            if total_size > 0 and progress_callback:
                                file_percent = int((downloaded_size / total_size) * file_progress_range)
                                percent = base_progress + 5 + file_percent
                                progress_callback(percent, f"下载{description}... {downloaded_size/1024/1024:.1f}MB")
                
                # 验证文件是否下载成功
                if not os.path.exists(download_path) or os.path.getsize(download_path) == 0:
                    raise Exception(f"下载的{description}无效或为空")
                
                logger.info(f"{description}下载成功: {download_path}")
            
            if progress_callback:
                progress_callback(95, "验证安装...")
            
            # 验证所有文件都已正确安装
            ffmpeg_path = os.path.join(bin_dir, 'ffmpeg.exe')
            ffprobe_path = os.path.join(bin_dir, 'ffprobe.exe')
            
            if not os.path.exists(ffmpeg_path) or not os.path.exists(ffprobe_path):
                raise Exception("部分文件下载失败")
            
            if progress_callback:
                progress_callback(100, "安装完成！")
            
            logger.info(f"ffmpeg和ffprobe安装成功")
            logger.info(f"  - ffmpeg: {ffmpeg_path}")
            logger.info(f"  - ffprobe: {ffprobe_path}")
            return True, "ffmpeg和ffprobe安装成功！"
            
        except Exception as e:
            error_msg = f"下载失败: {str(e)}"
            logger.error(error_msg)
            if progress_callback:
                progress_callback(0, error_msg)
            return False, error_msg


class VideoFrameExtractor:
    """视频关键帧提取器"""
    
    def __init__(self, ffmpeg_manager: FFmpegManager = None):
        self.ffmpeg_manager = ffmpeg_manager or FFmpegManager()
        self.temp_dir = tempfile.mkdtemp(prefix='fileneat_video_')
        
    def __del__(self):
        """清理临时目录"""
        try:
            if os.path.exists(self.temp_dir):
                shutil.rmtree(self.temp_dir)
        except:
            pass
    
    def extract_key_frames(self, video_path: str, num_frames: int = 3,
                          method: str = 'uniform') -> List[str]:
        """
        提取视频关键帧
        
        Args:
            video_path: 视频文件路径
            num_frames: 提取帧数（最多5帧）
            method: 提取方法 ('uniform' 均匀采样)
            
        Returns:
            关键帧图片路径列表
        """
        # 限制最多5帧
        num_frames = min(num_frames, 5)
        
        ffmpeg_path = self.ffmpeg_manager.get_ffmpeg_executable()
        if not ffmpeg_path:
            logger.error("ffmpeg未安装，无法提取视频关键帧")
            return []
        
        try:
            # 1. 获取视频时长
            duration = self._get_video_duration(video_path, ffmpeg_path)
            if duration <= 0:
                logger.error(f"无法获取视频时长: {video_path}")
                return []
            
            # 2. 计算采样时间点
            if method == 'uniform':
                # 均匀采样：避开开头和结尾各10%
                start_offset = duration * 0.1
                end_offset = duration * 0.9
                effective_duration = end_offset - start_offset
                
                if num_frames == 1:
                    timestamps = [duration * 0.5]  # 中间帧
                else:
                    interval = effective_duration / (num_frames - 1)
                    timestamps = [start_offset + i * interval for i in range(num_frames)]
            else:
                timestamps = [duration * 0.5]  # 默认只取中间帧
            
            # 3. 提取帧
            frame_paths = []
            for i, timestamp in enumerate(timestamps):
                output_path = os.path.join(
                    self.temp_dir,
                    f"frame_{i:02d}.jpg"
                )
                
                # 使用ffmpeg提取单帧
                cmd = [
                    ffmpeg_path,
                    '-ss', str(timestamp),
                    '-i', video_path,
                    '-vframes', '1',
                    '-q:v', '2',  # 高质量
                    '-y',  # 覆盖已存在的文件
                    output_path
                ]
                
                result = subprocess.run(
                    cmd,
                    capture_output=True,
                    timeout=30,
                    encoding='utf-8',
                    errors='ignore'
                )
                
                if result.returncode == 0 and os.path.exists(output_path):
                    frame_paths.append(output_path)
                    logger.info(f"成功提取关键帧 {i+1}/{num_frames}: {timestamp:.2f}s")
                else:
                    logger.warning(f"提取关键帧失败: timestamp={timestamp:.2f}s")
            
            return frame_paths
            
        except Exception as e:
            logger.error(f"提取视频关键帧失败: {str(e)}")
            return []
    
    def _get_video_duration(self, video_path: str, ffmpeg_path: str) -> float:
        """获取视频时长（秒）"""
        try:
            cmd = [
                ffmpeg_path,
                '-i', video_path
            ]
            
            result = subprocess.run(
                cmd,
                capture_output=True,
                timeout=10,
                encoding='utf-8',
                errors='ignore'
            )
            
            # 从ffmpeg输出中解析时长
            output = result.stderr
            
            # 匹配 Duration: 00:01:23.45 格式
            import re
            match = re.search(r'Duration: (\d+):(\d+):(\d+\.\d+)', output)
            if match:
                hours = int(match.group(1))
                minutes = int(match.group(2))
                seconds = float(match.group(3))
                duration = hours * 3600 + minutes * 60 + seconds
                return duration
            
            return 0.0
            
        except Exception as e:
            logger.error(f"获取视频时长失败: {str(e)}")
            return 0.0


class VideoMetadataExtractor:
    """视频元数据提取器"""
    
    def __init__(self, ffmpeg_manager: FFmpegManager = None):
        self.ffmpeg_manager = ffmpeg_manager or FFmpegManager()
    
    def extract_metadata(self, video_path: str) -> Dict:
        """提取视频元数据"""
        ffmpeg_path = self.ffmpeg_manager.get_ffmpeg_executable()
        if not ffmpeg_path:
            return self._get_basic_metadata(video_path)
        
        try:
            # 使用ffprobe获取详细信息
            ffprobe_path = ffmpeg_path.replace('ffmpeg', 'ffprobe')
            
            cmd = [
                ffprobe_path,
                '-v', 'quiet',
                '-print_format', 'json',
                '-show_format',
                '-show_streams',
                video_path
            ]
            
            result = subprocess.run(
                cmd,
                capture_output=True,
                timeout=10,
                encoding='utf-8',
                errors='ignore'
            )
            
            if result.returncode == 0:
                data = json.loads(result.stdout)
                return self._parse_ffprobe_data(data, video_path)
            
        except Exception as e:
            logger.warning(f"使用ffprobe提取元数据失败: {str(e)}")
        
        return self._get_basic_metadata(video_path)
    
    def _parse_ffprobe_data(self, data: dict, video_path: str) -> Dict:
        """解析ffprobe数据"""
        format_info = data.get('format', {})
        video_stream = None
        
        # 找到视频流
        for stream in data.get('streams', []):
            if stream.get('codec_type') == 'video':
                video_stream = stream
                break
        
        metadata = {
            'filename': os.path.basename(video_path),
            'file_size': int(format_info.get('size', 0)),
            'duration': float(format_info.get('duration', 0)),
            'bitrate': int(format_info.get('bit_rate', 0)),
            'format': format_info.get('format_name', 'unknown'),
        }
        
        if video_stream:
            metadata.update({
                'width': video_stream.get('width', 0),
                'height': video_stream.get('height', 0),
                'codec': video_stream.get('codec_name', 'unknown'),
                'fps': self._parse_fps(video_stream.get('r_frame_rate', '0/1')),
                'resolution': f"{video_stream.get('width', 0)}x{video_stream.get('height', 0)}"
            })
        
        return metadata
    
    def _parse_fps(self, fps_str: str) -> float:
        """解析帧率字符串"""
        try:
            if '/' in fps_str:
                num, den = fps_str.split('/')
                return float(num) / float(den)
            return float(fps_str)
        except:
            return 0.0
    
    def _get_basic_metadata(self, video_path: str) -> Dict:
        """获取基本元数据（不依赖ffmpeg）"""
        stat = os.stat(video_path)
        return {
            'filename': os.path.basename(video_path),
            'file_size': stat.st_size,
            'duration': 0,
            'resolution': 'unknown',
            'format': Path(video_path).suffix.lower()
        }


class VideoContentAnalyzer:
    """视频内容分析器"""
    
    def __init__(self):
        self.ffmpeg_manager = FFmpegManager()
        self.frame_extractor = VideoFrameExtractor(self.ffmpeg_manager)
        self.metadata_extractor = VideoMetadataExtractor(self.ffmpeg_manager)
    
    def analyze_video(self, video_path: str, num_frames: int = 3) -> Dict:
        """
        完整的视频分析流程
        
        Args:
            video_path: 视频文件路径
            num_frames: 提取关键帧数量（最多5个）
            
        Returns:
            分析结果字典
        """
        try:
            logger.info(f"开始分析视频: {video_path}")
            
            # 1. 提取元数据
            metadata = self.metadata_extractor.extract_metadata(video_path)
            logger.info(f"视频元数据: {metadata}")
            
            # 2. 提取关键帧
            num_frames = min(num_frames, 5)  # 确保不超过5帧
            key_frames = self.frame_extractor.extract_key_frames(
                video_path,
                num_frames=num_frames,
                method='uniform'
            )
            
            logger.info(f"成功提取 {len(key_frames)} 个关键帧")
            
            # 3. 使用多模态AI分析关键帧
            frame_analyses = []
            if key_frames:
                from lib.multimodal_ai import multimodal_service
                
                # 检查多模态AI是否可用
                if multimodal_service.is_available():
                    for i, frame_path in enumerate(key_frames):
                        logger.info(f"分析关键帧 {i+1}/{len(key_frames)}")
                        
                        success, content, cost, provider = \
                            multimodal_service.recognize_image_smart(
                                frame_path,
                                prompt=self._get_video_frame_analysis_prompt()
                            )
                        
                        if success:
                            frame_analyses.append({
                                'frame_index': i,
                                'content': content,
                                'cost': cost,
                                'provider': provider
                            })
                            logger.info(f"关键帧 {i+1} 分析成功")
                            logger.info(f"【视频关键帧识别内容】 帧{i+1}:\n{content}")
                        else:
                            logger.warning(f"关键帧 {i+1} 分析失败: {content}")
                else:
                    logger.warning("多模态AI服务不可用，跳过关键帧内容分析")
            
            # 4. 综合分析结果
            return {
                'metadata': metadata,
                'key_frames': key_frames,
                'frame_analyses': frame_analyses,
                'summary': self._generate_summary(metadata, frame_analyses)
            }
            
        except Exception as e:
            logger.error(f"视频分析失败: {str(e)}")
            return {
                'metadata': {'filename': os.path.basename(video_path)},
                'key_frames': [],
                'frame_analyses': [],
                'summary': f"视频分析失败: {str(e)}"
            }
    
    def _get_video_frame_analysis_prompt(self) -> str:
        """获取视频帧分析提示词 - 优化用于分类"""
        return """请详细分析这个视频画面，重点描述：

1. 【内容主题】这是什么类型的视频？（如：工作会议、教学课程、生活记录、游戏解说、产品演示、新闻报道、娱乐节目等）

2. 【场景描述】画面中有什么？（人物、环境、物体、文字等关键元素）

3. 【用途判断】这个视频可能的用途或目的是什么？（学习、工作、娱乐、记录、分享等）

4. 【关键特征】有助于分类的特殊标识（如品牌logo、软件界面、特定场景等）

请用清晰、具体的语言描述，帮助准确理解视频内容（150字以内）。
重点是识别视频的主题和类型，而不只是描述画面。"""
    
    def _generate_summary(self, metadata: Dict, 
                         frame_analyses: List[Dict]) -> str:
        """生成视频内容摘要 - 优化用于AI分类"""
        filename = metadata.get('filename', 'unknown')
        duration = metadata.get('duration', 0)
        resolution = metadata.get('resolution', 'unknown')
        file_size = metadata.get('file_size', 0)
        
        # 🚀 优化：从文件名中提取更多信息
        filename_analysis = self._analyze_filename(filename)
        
        # 如果有关键帧分析内容，以内容为主
        if frame_analyses and len(frame_analyses) > 0:
            # 合并所有帧的分析内容
            contents = [fa['content'] for fa in frame_analyses]
            content_description = ' '.join(contents)
            
            # 🔥 新格式：突出内容，提供更多分类线索
            summary = f"""【视频内容】
{content_description}

【文件信息】
原始文件名: {filename}
{filename_analysis}
视频时长: {self._format_duration(duration)}
文件大小: {self._format_file_size(file_size)}

【分类指导】
这是一个视频文件，请根据上述内容描述来理解视频的主题和用途，然后进行精准分类。
不要将其简单归类为"视频"或"多媒体"，而应该根据视频的实际内容主题进行分类。
比如教学视频可以归到"教育与培训"，会议录像可以归到"工作会议"，生活记录可以归到"个人生活"等。"""
            
        else:
            # 如果没有内容分析（多模态AI不可用），提供基础信息并强调基于文件名分类
            summary = f"""【视频文件分类建议】

原始文件名: {filename}
{filename_analysis}
视频时长: {self._format_duration(duration)}
文件大小: {self._format_file_size(file_size)}

【智能分类提示】
由于多模态AI未启用，无法识别视频内容，请根据以下信息进行智能分类：

1. **文件名分析**：{filename_analysis if filename_analysis != '文件名特征: 无明显特征' else '文件名较为简单，可能是随手拍摄或临时记录'}

2. **时长判断**：
   - 超短视频(< 30秒): 可能是快拍、片段、GIF式内容
   - 短视频(30秒-3分钟): 可能是社交媒体内容、精彩片段、展示视频
   - 中等时长(3-15分钟): 可能是教程、演示、短片、生活记录
   - 长视频(15-60分钟): 可能是课程、会议、讲座、完整作品
   - 超长视频(> 60分钟): 可能是电影、长篇讲座、完整会议

3. **文件大小参考**：
   - 小文件: 可能是低分辨率、简短内容
   - 大文件: 可能是高清内容、重要录制

【分类建议】
- 如果文件名包含日期但无其他信息，建议归类到"个人记录/按日期分类"
- 如果文件名包含关键词，优先使用关键词分类（如"会议"→工作会议，"教程"→教育培训）
- 如果文件名无明显特征且时长较短，建议归类到"生活记录/短视频"
- 避免使用"未分类"、"视频文件"等模糊分类

【重要】请创建具体、有意义的分类，帮助用户更好地管理视频文件。"""
        
        return summary
    
    def _analyze_filename(self, filename: str) -> str:
        """分析文件名，提取有用的分类信息"""
        import re
        
        features = []
        
        # 去除扩展名
        name_without_ext = filename.rsplit('.', 1)[0] if '.' in filename else filename
        
        # 1. 检测日期模式
        date_patterns = [
            (r'\d{4}[-_年]\d{1,2}[-_月]\d{1,2}', '包含完整日期'),
            (r'\d{1,2}月\d{1,2}日', '包含月日信息'),
            (r'\d{8}', '包含8位数字日期'),
            (r'\d{4}\d{2}\d{2}', '包含年月日格式'),
        ]
        
        for pattern, desc in date_patterns:
            if re.search(pattern, name_without_ext):
                features.append(desc)
                break
        
        # 2. 检测常见关键词
        keywords = {
            '工作': ['会议', '项目', '汇报', '演示', '培训', 'meeting', 'presentation'],
            '教育': ['教程', '课程', '讲座', '学习', 'tutorial', 'course', 'lesson'],
            '生活': ['旅游', '旅行', '聚会', '生日', '纪念', 'vlog', 'travel', 'party'],
            '娱乐': ['电影', '视频', '搞笑', '游戏', 'movie', 'game', 'funny'],
            '技术': ['编程', '开发', '技术', '代码', 'coding', 'dev', 'tech'],
            '产品': ['产品', '演示', 'demo', 'product', '展示'],
        }
        
        for category, words in keywords.items():
            for word in words:
                if word.lower() in name_without_ext.lower():
                    features.append(f'可能是{category}相关内容（包含关键词"{word}"）')
                    break
        
        # 3. 检测序号或版本号
        if re.search(r'[vV]\d+|版本\d+|第\d+[章节期]', name_without_ext):
            features.append('包含版本或序号信息')
        
        # 4. 检测人名或地点（中文）
        if re.search(r'[\u4e00-\u9fa5]{2,4}(?:的|在|与)', name_without_ext):
            features.append('包含人名或地点信息')
        
        if features:
            return '文件名特征: ' + '，'.join(features)
        else:
            return '文件名特征: 无明显特征'
    
    def _format_file_size(self, bytes_size: int) -> str:
        """格式化文件大小"""
        if bytes_size < 1024 * 1024:  # < 1MB
            return f"{bytes_size / 1024:.1f} KB"
        elif bytes_size < 1024 * 1024 * 1024:  # < 1GB
            return f"{bytes_size / (1024 * 1024):.1f} MB"
        else:
            return f"{bytes_size / (1024 * 1024 * 1024):.2f} GB"
    
    def _format_duration(self, seconds: float) -> str:
        """格式化时长"""
        if seconds < 60:
            return f"{seconds:.0f}秒"
        elif seconds < 3600:
            minutes = seconds / 60
            return f"{minutes:.1f}分钟"
        else:
            hours = seconds / 3600
            return f"{hours:.1f}小时"


# 创建全局实例
ffmpeg_manager = FFmpegManager()
video_analyzer = VideoContentAnalyzer()

