"""
FileNeatAI 崩溃日志处理模块
用于捕获和记录程序崩溃信息，便于问题排查
"""

import sys
import os
import traceback
import platform
import datetime
import threading
import faulthandler
import atexit
from pathlib import Path

# 全局变量存储崩溃日志路径
_crash_log_path = None
_is_initialized = False


def get_crash_log_path():
    """获取崩溃日志文件路径"""
    global _crash_log_path
    
    if _crash_log_path:
        return _crash_log_path
    
    # 确定日志目录
    if getattr(sys, 'frozen', False):
        # 打包后的exe
        base_dir = os.path.dirname(sys.executable)
    else:
        # 开发环境
        base_dir = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
    
    # 创建 logs 目录
    log_dir = os.path.join(base_dir, 'logs')
    try:
        if not os.path.exists(log_dir):
            os.makedirs(log_dir)
    except:
        # 如果无法创建，使用临时目录
        import tempfile
        log_dir = tempfile.gettempdir()
    
    _crash_log_path = os.path.join(log_dir, 'crash.log')
    return _crash_log_path


def get_system_info():
    """收集系统信息"""
    info = []
    info.append(f"操作系统: {platform.system()} {platform.release()} ({platform.version()})")
    info.append(f"架构: {platform.machine()}")
    info.append(f"Python版本: {platform.python_version()}")
    info.append(f"Python实现: {platform.python_implementation()}")
    
    # 尝试获取内存信息
    try:
        import psutil
        mem = psutil.virtual_memory()
        info.append(f"内存: 总计 {mem.total / (1024**3):.1f}GB, 可用 {mem.available / (1024**3):.1f}GB, 使用率 {mem.percent}%")
        
        # 当前进程信息
        process = psutil.Process()
        mem_info = process.memory_info()
        info.append(f"进程内存: RSS {mem_info.rss / (1024**2):.1f}MB, VMS {mem_info.vms / (1024**2):.1f}MB")
    except:
        pass
    
    # Qt/PySide6 版本
    try:
        from PySide6.QtCore import qVersion
        info.append(f"Qt版本: {qVersion()}")
    except:
        pass
    
    try:
        import PySide6
        info.append(f"PySide6版本: {PySide6.__version__}")
    except:
        pass
    
    # 当前工作目录
    info.append(f"当前目录: {os.getcwd()}")
    
    # 可执行文件路径
    if getattr(sys, 'frozen', False):
        info.append(f"可执行文件: {sys.executable}")
    
    return "\n".join(info)


def get_thread_info():
    """获取当前线程信息"""
    info = []
    current = threading.current_thread()
    info.append(f"当前线程: {current.name} (ID: {current.ident})")
    
    # 列出所有活动线程
    info.append(f"活动线程数: {threading.active_count()}")
    for thread in threading.enumerate():
        info.append(f"  - {thread.name} (daemon: {thread.daemon}, alive: {thread.is_alive()})")
    
    return "\n".join(info)


def write_crash_log(error_type, error_value, error_tb, context=""):
    """写入崩溃日志"""
    crash_log_path = get_crash_log_path()
    
    timestamp = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    separator = "=" * 80
    
    log_content = []
    log_content.append(f"\n{separator}")
    log_content.append(f"崩溃时间: {timestamp}")
    log_content.append(separator)
    
    # 上下文信息
    if context:
        log_content.append(f"\n【上下文】")
        log_content.append(context)
    
    # 错误信息
    log_content.append(f"\n【错误类型】")
    log_content.append(f"{error_type.__name__ if error_type else 'Unknown'}: {error_value}")
    
    # 完整堆栈追踪
    log_content.append(f"\n【堆栈追踪】")
    if error_tb:
        tb_lines = traceback.format_exception(error_type, error_value, error_tb)
        log_content.append("".join(tb_lines))
    
    # 系统信息
    log_content.append(f"\n【系统信息】")
    log_content.append(get_system_info())
    
    # 线程信息
    log_content.append(f"\n【线程信息】")
    log_content.append(get_thread_info())
    
    # 所有线程的堆栈（用于诊断死锁等问题）
    log_content.append(f"\n【所有线程堆栈】")
    try:
        for thread_id, frame in sys._current_frames().items():
            log_content.append(f"\n--- 线程 {thread_id} ---")
            log_content.append("".join(traceback.format_stack(frame)))
    except:
        log_content.append("无法获取线程堆栈")
    
    log_content.append(f"\n{separator}\n")
    
    # 写入文件
    try:
        with open(crash_log_path, 'a', encoding='utf-8') as f:
            f.write("\n".join(log_content))
        # 安全打印
        if sys.stdout is not None:
            try:
                print(f"崩溃日志已保存到: {crash_log_path}")
            except:
                pass
    except Exception as e:
        # 安全打印错误信息
        if sys.stderr is not None:
            try:
                print(f"无法写入崩溃日志: {e}", file=sys.stderr)
                print("\n".join(log_content), file=sys.stderr)
            except:
                pass


def global_exception_handler(exctype, value, tb):
    """全局异常处理器"""
    # 忽略 KeyboardInterrupt
    if issubclass(exctype, KeyboardInterrupt):
        sys.__excepthook__(exctype, value, tb)
        return
    
    # 写入崩溃日志
    write_crash_log(exctype, value, tb, context="全局未捕获异常")
    
    # 尝试记录到 logger
    try:
        from logger import logger
        error_msg = ''.join(traceback.format_exception(exctype, value, tb))
        logger.critical(f"未捕获的全局异常:\n{error_msg}")
    except:
        pass
    
    # 调用默认处理器
    sys.__excepthook__(exctype, value, tb)


def thread_exception_handler(args):
    """线程异常处理器 (Python 3.8+)"""
    exctype = args.exc_type
    value = args.exc_value
    tb = args.exc_traceback
    thread = args.thread
    
    context = f"线程 '{thread.name}' (ID: {thread.ident}) 中发生异常"
    write_crash_log(exctype, value, tb, context=context)
    
    # 尝试记录到 logger
    try:
        from logger import logger
        error_msg = ''.join(traceback.format_exception(exctype, value, tb))
        logger.critical(f"线程异常 [{thread.name}]:\n{error_msg}")
    except:
        pass


def qt_message_handler(mode, context, message):
    """Qt 消息处理器"""
    from PySide6.QtCore import QtMsgType
    
    # 根据消息类型处理
    if mode == QtMsgType.QtFatalMsg:
        # 致命错误，写入崩溃日志
        write_crash_log(
            RuntimeError, 
            RuntimeError(f"Qt Fatal: {message}"), 
            None,
            context=f"Qt致命错误\n文件: {context.file}\n行: {context.line}\n函数: {context.function}"
        )
    elif mode == QtMsgType.QtCriticalMsg:
        # 严重错误
        try:
            from logger import logger
            logger.critical(f"Qt Critical: {message} (at {context.file}:{context.line})")
        except:
            pass
    elif mode == QtMsgType.QtWarningMsg:
        # 警告（只记录重要的）
        if "QThread" in message or "segfault" in message.lower() or "crash" in message.lower():
            try:
                from logger import logger
                logger.warning(f"Qt Warning: {message}")
            except:
                pass


def on_exit():
    """程序退出时的处理"""
    crash_log_path = get_crash_log_path()
    
    # 记录正常退出
    try:
        timestamp = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        with open(crash_log_path, 'a', encoding='utf-8') as f:
            f.write(f"\n[{timestamp}] 程序正常退出\n")
    except:
        pass


def init_crash_handler():
    """初始化崩溃处理系统"""
    global _is_initialized
    
    if _is_initialized:
        return
    
    # 启用 faulthandler（捕获 C 级别的崩溃）
    crash_log_path = get_crash_log_path()
    crash_log_file = None
    
    try:
        # 将 faulthandler 输出重定向到崩溃日志文件
        crash_log_file = open(crash_log_path, 'a', encoding='utf-8')
        faulthandler.enable(file=crash_log_file)
    except Exception:
        # 如果无法写入文件，尝试使用 stderr（如果可用）
        try:
            if sys.stderr is not None:
                faulthandler.enable(file=sys.stderr)
        except Exception:
            pass  # 在无控制台模式下，stderr 为 None，跳过 faulthandler
    
    # 设置全局异常处理器
    sys.excepthook = global_exception_handler
    
    # 设置线程异常处理器 (Python 3.8+)
    if hasattr(threading, 'excepthook'):
        threading.excepthook = thread_exception_handler
    
    # 设置 Qt 消息处理器
    try:
        from PySide6.QtCore import qInstallMessageHandler
        qInstallMessageHandler(qt_message_handler)
    except Exception:
        pass
    
    # 注册退出处理
    atexit.register(on_exit)
    
    # 记录启动信息
    try:
        timestamp = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        with open(crash_log_path, 'a', encoding='utf-8') as f:
            f.write(f"\n{'='*80}\n")
            f.write(f"[{timestamp}] 程序启动\n")
            f.write(f"{'='*80}\n")
            f.write(f"{get_system_info()}\n")
    except Exception:
        pass
    
    _is_initialized = True
    
    # 安全打印（在无控制台模式下不会报错）
    try:
        if sys.stdout is not None:
            print(f"崩溃日志系统已初始化，日志文件: {crash_log_path}")
    except Exception:
        pass


def log_operation(operation_name, details=""):
    """记录当前操作（用于崩溃时知道在做什么）"""
    try:
        from logger import logger
        logger.info(f"[操作] {operation_name}: {details}")
    except:
        pass


# 方便的装饰器，用于包装可能崩溃的函数
def crash_protected(func):
    """装饰器：保护函数不会因异常而崩溃程序"""
    def wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except Exception as e:
            error_tb = sys.exc_info()[2]
            context = f"在函数 '{func.__name__}' 中发生异常"
            write_crash_log(type(e), e, error_tb, context=context)
            raise  # 重新抛出异常
    return wrapper


# 用于在关键位置记录状态
class CrashContext:
    """上下文管理器：记录关键操作"""
    _current_context = []
    
    def __init__(self, description):
        self.description = description
    
    def __enter__(self):
        CrashContext._current_context.append(self.description)
        return self
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        CrashContext._current_context.pop()
        if exc_type is not None:
            # 发生异常时记录上下文
            context = " -> ".join(CrashContext._current_context + [self.description])
            write_crash_log(exc_type, exc_val, exc_tb, context=f"操作链: {context}")
        return False  # 不抑制异常
    
    @classmethod
    def get_current_context(cls):
        return " -> ".join(cls._current_context) if cls._current_context else "无"

