# -*- coding: utf-8 -*-
"""
多模态AI设置界面 - 独立的设置窗口
"""

import os
import json
from PySide6.QtCore import Qt, Signal, QTimer
from PySide6.QtWidgets import (QDialog, QVBoxLayout, QHBoxLayout, QLabel, QPushButton,
                              QGroupBox, QRadioButton, QLineEdit, QSpinBox, QDoubleSpinBox,
                              QComboBox, QCheckBox, QTextEdit, QTabWidget, QWidget,
                              QProgressBar, QFrame, QScrollArea, QMessageBox)
from PySide6.QtGui import QIcon, QPixmap
from logger import logger


class MultimodalSettingsDialog(QDialog):
    """多模态AI设置对话框"""
    
    settings_saved = Signal(dict)  # 设置保存信号
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("多模态AI图片识别设置")
        self.setModal(True)
        self.setFixedSize(800, 600)  # 🚀 优化：调整窗口高度以适应所有内容
        self.setup_ui()
        self.load_settings()
        
        # 居中显示
        self.center_on_screen()
    
    def setup_ui(self):
        """设置UI"""
        self.setStyleSheet("""
            QDialog {
                background-color: #1e1e1e;
                color: #ffffff;
            }
            QTabWidget::pane {
                border: 1px solid #3a3a3a;
                background-color: #2a2a2a;
            }
            QTabBar::tab {
                background-color: #3a3a3a;
                color: #ffffff;
                padding: 10px 20px;
                margin-right: 2px;
            }
            QTabBar::tab:selected {
                background-color: #19AF39;
            }
            QGroupBox {
                font-size: 14px;
                font-weight: bold;
                color: #ffffff;
                background-color: #1e1e1e;
                border: 2px solid #3a3a3a;
                border-radius: 8px;
                margin-top: 15px;
                padding-top: 10px;
            }
            QGroupBox::title {
                subcontrol-origin: margin;
                left: 10px;
                padding: 0 5px 0 5px;
            }
            QLabel {
                color: #ffffff;
                font-size: 13px;
            }
            QPushButton {
                background-color: #19AF39;
                color: white;
                border: none;
                padding: 8px 16px;
                border-radius: 4px;
                font-size: 13px;
            }
            QPushButton:hover {
                background-color: #16973a;
            }
            QPushButton:disabled {
                background-color: #666666;
                color: #999999;
            }
            QLineEdit, QSpinBox, QDoubleSpinBox, QComboBox {
                background-color: #2a2a2a;
                border: 1px solid #3a3a3a;
                border-radius: 4px;
                padding: 8px;
                color: #ffffff;
                font-size: 13px;
            }
            QTextEdit {
                background-color: #1e1e1e;
                border: 1px solid #3a3a3a;
                border-radius: 4px;
                color: #ffffff;
                font-size: 12px;
            }
            QCheckBox, QRadioButton {
                color: #ffffff;
                font-size: 13px;
            }
            QCheckBox::indicator, QRadioButton::indicator {
                width: 18px;
                height: 18px;
            }
            QCheckBox::indicator:checked {
                background-color: #19AF39;
                border: 2px solid #19AF39;
            }
            QRadioButton::indicator:checked {
                background-color: #19AF39;
                border: 3px solid #19AF39;
                border-radius: 9px;
            }
        """)
        
        layout = QVBoxLayout(self)
        
        # 🚀 优化：保持原有界面内容，只是去掉标签页结构
        # 创建滚动区域以容纳所有内容
        scroll_area = QScrollArea()
        scroll_area.setWidgetResizable(True)
        scroll_area.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        scroll_area.setVerticalScrollBarPolicy(Qt.ScrollBarAsNeeded)
        scroll_area.setStyleSheet("""
            QScrollArea {
                border: none;
                background-color: #1e1e1e;
            }
            QScrollArea > QWidget > QWidget {
                background-color: #1e1e1e;
            }
            QScrollBar:vertical {
                background-color: #2a2a2a;
                width: 12px;
                border-radius: 6px;
            }
            QScrollBar::handle:vertical {
                background-color: #19AF39;
                border-radius: 6px;
                min-height: 20px;
            }
        """)
        
        # 创建内容容器
        content_widget = QWidget()
        content_widget.setStyleSheet("""
            QWidget {
                background-color: #1e1e1e;
                color: #ffffff;
            }
        """)
        content_layout = QVBoxLayout(content_widget)
        
        # 添加原有的服务商配置内容
        self.provider_content = self.create_provider_tab()
        content_layout.addWidget(self.provider_content)
        
        scroll_area.setWidget(content_widget)
        layout.addWidget(scroll_area)
        
        # 底部按钮
        button_layout = QHBoxLayout()
        
        self.test_button = QPushButton("🧪 测试配置")
        self.test_button.clicked.connect(self.test_configuration)
        
        self.save_button = QPushButton("💾 保存设置")
        self.save_button.clicked.connect(self.save_settings)
        
        self.cancel_button = QPushButton("❌ 取消")
        self.cancel_button.clicked.connect(self.reject)
        self.cancel_button.setStyleSheet("""
            QPushButton {
                background-color: #6c757d;
                color: white;
            }
            QPushButton:hover {
                background-color: #5a6268;
            }
        """)
        
        button_layout.addWidget(self.test_button)
        button_layout.addStretch()
        button_layout.addWidget(self.save_button)
        button_layout.addWidget(self.cancel_button)
        
        layout.addLayout(button_layout)
    
    def create_provider_tab(self):
        """创建服务商配置标签页"""
        tab = QWidget()
        tab.setStyleSheet("""
            QWidget {
                background-color: #1e1e1e;
                color: #ffffff;
            }
        """)
        layout = QVBoxLayout(tab)
        
        # 说明文字
        info_label = QLabel("""
        <b>多模态AI图片识别服务配置</b><br>
        配置支持OpenAI格式的多模态AI服务，用于图片内容识别和文字提取。<br>
        大多数主流AI服务商都支持OpenAI兼容的API格式。
        """)
        info_label.setWordWrap(True)
        info_label.setStyleSheet("""
            QLabel {
                background-color: #1e1e1e;
                color: #cccccc;
                font-size: 12px;
                padding: 10px;
            }
        """)
        layout.addWidget(info_label)
        
        # 统一配置组
        config_group = QGroupBox("🤖 多模态AI服务配置")
        config_layout = QVBoxLayout(config_group)
        
        # 启用开关
        self.multimodal_enabled = QCheckBox("启用多模态AI图片识别")
        self.multimodal_enabled.setChecked(False)
        self.multimodal_enabled.setStyleSheet("font-size: 14px; font-weight: bold; color: #19AF39;")
        config_layout.addWidget(self.multimodal_enabled)
        
        # API地址
        api_base_layout = QHBoxLayout()
        api_base_label = QLabel("API地址:")
        api_base_label.setMinimumWidth(80)
        api_base_label.setStyleSheet("color: #ffffff; font-size: 13px;")
        
        self.api_base_input = QLineEdit()
        self.api_base_input.setPlaceholderText("例如: https://api.openai.com/v1 或 https://ark.cn-beijing.volces.com/api/v3")
        self.api_base_input.setStyleSheet("""
            QLineEdit {
                background-color: #2a2a2a;
                border: 1px solid #3a3a3a;
                border-radius: 4px;
                padding: 8px;
                color: #ffffff;
                font-size: 13px;
            }
        """)
        
        api_base_layout.addWidget(api_base_label)
        api_base_layout.addWidget(self.api_base_input)
        config_layout.addLayout(api_base_layout)
        
        # API密钥
        api_key_layout = QHBoxLayout()
        api_key_label = QLabel("API密钥:")
        api_key_label.setMinimumWidth(80)
        api_key_label.setStyleSheet("color: #ffffff; font-size: 13px;")
        
        self.api_key_input = QLineEdit()
        self.api_key_input.setPlaceholderText("输入你的API密钥...")
        self.api_key_input.setStyleSheet("""
            QLineEdit {
                background-color: #2a2a2a;
                border: 1px solid #3a3a3a;
                border-radius: 4px;
                padding: 8px;
                color: #ffffff;
                font-size: 13px;
            }
        """)
        
        api_key_layout.addWidget(api_key_label)
        api_key_layout.addWidget(self.api_key_input)
        config_layout.addLayout(api_key_layout)
        
        # 模型名称
        model_layout = QHBoxLayout()
        model_label = QLabel("模型名称:")
        model_label.setMinimumWidth(80)
        model_label.setStyleSheet("color: #ffffff; font-size: 13px;")
        
        self.model_input = QLineEdit()
        self.model_input.setPlaceholderText("例如: gpt-4-vision-preview 或 claude-3-5-sonnet-20241022")
        self.model_input.setStyleSheet("""
            QLineEdit {
                background-color: #2a2a2a;
                border: 1px solid #3a3a3a;
                border-radius: 4px;
                padding: 8px;
                color: #ffffff;
                font-size: 13px;
            }
        """)
        
        model_layout.addWidget(model_label)
        model_layout.addWidget(self.model_input)
        config_layout.addLayout(model_layout)
        
        layout.addWidget(config_group)
        
        # 常用配置示例
        examples_group = QGroupBox("📋 常用配置示例")
        examples_layout = QVBoxLayout(examples_group)
        
        examples_text = QLabel("""
<b>🌙 Moonshot API (推荐 - 第一优先级文字识别):</b><br>
• API地址: https://api.moonshot.cn/v1<br>
• 模型名称: moonshot-v1-128k<br>
• 说明: 专门优化的图片文字提取，识别准确度高<br><br>

<b>OpenAI GPT-4 Vision:</b><br>
• API地址: https://api.openai.com/v1<br>
• 模型名称: gpt-4-vision-preview<br><br>

<b>Claude 3.5 Sonnet:</b><br>
• API地址: https://api.anthropic.com<br>
• 模型名称: claude-3-5-sonnet-20241022<br><br>

<b>火山引擎豆包:</b><br>
• API地址: https://ark.cn-beijing.volces.com/api/v3<br>
• 模型名称: doubao-1-5-thinking-vision-pro-250428<br><br>

<b>💡 重要提示:</b><br>
配置Moonshot API可以享受第一优先级的高精度文字识别服务。其他服务作为备用选择。
        """)
        examples_text.setWordWrap(True)
        examples_text.setStyleSheet("""
            QLabel {
                background-color: #1e1e1e;
                color: #cccccc;
                font-size: 11px;
                padding: 10px;
            }
        """)
        examples_layout.addWidget(examples_text)
        
        layout.addWidget(examples_group)
        
        layout.addStretch()
        return tab
    
    # 🚀 优化：注释掉不需要的成本控制标签页
    def create_cost_tab(self):
        """创建成本控制标签页 - 已禁用"""
        # 🚀 优化：简化界面，不再使用此标签页
        tab = QWidget()
        layout = QVBoxLayout(tab)
        layout.addWidget(QLabel("此功能已简化，只保留AI服务商配置"))
        return tab
    
    # 🚀 优化：注释掉不需要的高级设置标签页  
    def create_advanced_tab(self):
        """创建高级设置标签页 - 已禁用"""
        # 返回空的标签页，不再使用
        tab = QWidget()
        layout = QVBoxLayout(tab)
        layout.addWidget(QLabel("此功能已简化"))
        return tab
        
        # 识别策略
        strategy_group = QGroupBox("🎯 识别策略")
        strategy_layout = QVBoxLayout(strategy_group)
        
        # 智能路由
        self.smart_routing = QCheckBox("🧠 启用智能路由 (自动选择最佳服务商)")
        self.smart_routing.setChecked(True)
        self.smart_routing.setToolTip("根据图片类型和服务商特点自动选择最佳识别方案")
        strategy_layout.addWidget(self.smart_routing)
        
        # 质量优先
        quality_layout = QHBoxLayout()
        quality_layout.addWidget(QLabel("识别质量偏好:"))
        
        self.quality_preference = QComboBox()
        self.quality_preference.addItems(["速度优先", "平衡", "质量优先"])
        self.quality_preference.setCurrentText("平衡")
        quality_layout.addWidget(self.quality_preference)
        quality_layout.addStretch()
        
        strategy_layout.addLayout(quality_layout)
        
        # 降级策略
        self.fallback_ocr = QCheckBox("🔄 AI失败时自动降级到OCR")
        self.fallback_ocr.setChecked(True)
        strategy_layout.addWidget(self.fallback_ocr)
        
        layout.addWidget(strategy_group)
        
        # 性能优化
        performance_group = QGroupBox("⚡ 性能优化")
        performance_layout = QVBoxLayout(performance_group)
        
        # 缓存设置
        self.enable_cache = QCheckBox("💾 启用结果缓存 (提高重复处理速度)")
        self.enable_cache.setChecked(True)
        performance_layout.addWidget(self.enable_cache)
        
        # 并发控制
        concurrent_layout = QHBoxLayout()
        concurrent_layout.addWidget(QLabel("最大并发请求数:"))
        
        self.max_concurrent = QSpinBox()
        self.max_concurrent.setRange(1, 10)
        self.max_concurrent.setValue(3)
        concurrent_layout.addWidget(self.max_concurrent)
        concurrent_layout.addStretch()
        
        performance_layout.addLayout(concurrent_layout)
        
        # 超时设置
        timeout_layout = QHBoxLayout()
        timeout_layout.addWidget(QLabel("请求超时时间:"))
        
        self.request_timeout = QSpinBox()
        self.request_timeout.setRange(10, 120)
        self.request_timeout.setValue(30)
        self.request_timeout.setSuffix(" 秒")
        timeout_layout.addWidget(self.request_timeout)
        timeout_layout.addStretch()
        
        performance_layout.addLayout(timeout_layout)
        
        layout.addWidget(performance_group)
        
        # 图片处理
        image_group = QGroupBox("🖼️ 图片处理")
        image_layout = QVBoxLayout(image_group)
        
        # 图片压缩
        compress_layout = QHBoxLayout()
        compress_layout.addWidget(QLabel("图片压缩质量:"))
        
        self.image_quality = QSpinBox()
        self.image_quality.setRange(50, 100)
        self.image_quality.setValue(85)
        self.image_quality.setSuffix("%")
        compress_layout.addWidget(self.image_quality)
        compress_layout.addStretch()
        
        image_layout.addLayout(compress_layout)
        
        # 最大尺寸
        size_layout = QHBoxLayout()
        size_layout.addWidget(QLabel("最大图片尺寸:"))
        
        self.max_image_size = QComboBox()
        self.max_image_size.addItems(["1024px", "2048px", "4096px", "原始尺寸"])
        self.max_image_size.setCurrentText("2048px")
        size_layout.addWidget(self.max_image_size)
        size_layout.addStretch()
        
        image_layout.addLayout(size_layout)
        
        layout.addWidget(image_group)
        
        layout.addStretch()
        return tab
    
    
    # 🚀 优化：注释掉不需要的使用统计标签页
    def create_stats_tab(self):
        """创建使用统计标签页 - 已禁用"""
        # 返回空的标签页，不再使用
        tab = QWidget()
        layout = QVBoxLayout(tab)
        layout.addWidget(QLabel("此功能已简化"))
        return tab
        
        # 刷新按钮
        refresh_button = QPushButton("🔄 刷新统计数据")
        refresh_button.clicked.connect(self.refresh_stats)
        layout.addWidget(refresh_button)
        
        # 统计显示
        self.stats_text = QTextEdit()
        self.stats_text.setReadOnly(True)
        self.stats_text.setStyleSheet("""
            QTextEdit {
                font-family: 'Courier New', monospace;
                font-size: 12px;
            }
        """)
        layout.addWidget(self.stats_text)
        
        return tab
    
    # 🚀 优化：删除不再需要的成本估算方法
    # def update_cost_estimate(self): - 已删除
    
    def refresh_stats(self):
        """刷新使用统计"""
        # 这里应该从实际的使用记录中获取数据
        # 暂时显示模拟数据
        stats_text = """=== 多模态AI使用统计 ===

本月使用情况：
• 总处理图片数: 156 张
• 总花费: $3.24
• 平均成本: $0.021/张

服务配置：
• API地址: 已配置
• 模型名称: 已设置
• 服务状态: 正常运行

识别效果统计：
• 成功率: 94.2%
• 平均识别时间: 2.3秒
• 用户满意度: 4.6/5.0

优化建议：
✓ 当前配置良好
• 建议启用缓存功能提高效率
• 可根据需要调整预算限制
"""
        
        self.stats_text.setText(stats_text)
    
    def test_configuration(self):
        """测试当前配置"""
        self.test_button.setEnabled(False)
        self.test_button.setText("🔄 测试中...")
        
        # 模拟测试过程
        QTimer.singleShot(3000, self._test_complete)
    
    def _test_complete(self):
        """测试完成"""
        self.test_button.setEnabled(True)
        self.test_button.setText("🧪 测试配置")
        
        # 显示测试结果
        QMessageBox.information(self, "测试结果", 
                               "✅ 配置测试成功！\n\n"
                               "• 模型测试正常\n"
                               "• API连接正常\n"
                               "• 配置参数有效\n\n"
                               "可以开始使用多模态AI功能。")
    
    def save_settings(self):
        """保存设置 - 简化版"""
        try:
            # 🚀 优化：只保存基本的多模态AI配置
            settings = {
                'multimodal': {
                    'enabled': self.multimodal_enabled.isChecked(),
                    'api_base': self.api_base_input.text().strip(),
                    'api_key': self.api_key_input.text().strip(),
                    'model': self.model_input.text().strip()
                }
            }
            
            # 保存到文件
            config_dir = os.path.join(os.path.expanduser('~'), '.fileneatai')
            os.makedirs(config_dir, exist_ok=True)
            
            config_file = os.path.join(config_dir, 'multimodal_settings.json')
            with open(config_file, 'w', encoding='utf-8') as f:
                json.dump(settings, f, ensure_ascii=False, indent=2)
            
            # 发送信号
            self.settings_saved.emit(settings)
            
            # 重新加载全局多模态服务配置
            try:
                from lib.multimodal_ai import multimodal_service
                multimodal_service.reload_config()
            except Exception as e:
                logger.warning(f"重新加载全局多模态服务配置失败: {str(e)}")
            
            QMessageBox.information(self, "保存成功", "多模态AI设置已保存！")
            self.accept()
            
        except Exception as e:
            logger.error(f"保存设置失败: {str(e)}")
            QMessageBox.critical(self, "保存失败", f"保存设置时出错：\n{str(e)}")
    
    def load_settings(self):
        """加载设置 - 简化版"""
        try:
            config_file = os.path.join(os.path.expanduser('~'), '.fileneatai', 'multimodal_settings.json')
            if os.path.exists(config_file):
                with open(config_file, 'r', encoding='utf-8') as f:
                    settings = json.load(f)
                
                # 🚀 优化：只加载基本的多模态AI配置
                multimodal_config = settings.get('multimodal', {})
                self.multimodal_enabled.setChecked(multimodal_config.get('enabled', False))
                self.api_base_input.setText(multimodal_config.get('api_base', ''))
                self.api_key_input.setText(multimodal_config.get('api_key', ''))
                self.model_input.setText(multimodal_config.get('model', ''))
                
                logger.info("多模态AI设置加载成功")
                
        except Exception as e:
            logger.error(f"加载设置失败: {str(e)}")
            # 使用默认设置（不再调用不存在的方法）
            logger.info("使用默认多模态AI设置")
    
    # 🚀 优化：删除不再需要的统计重置方法
    # def reset_usage_stats(self): - 已删除
    
    def center_on_screen(self):
        """居中显示"""
        from PySide6.QtWidgets import QApplication
        screen = QApplication.primaryScreen()
        screen_geometry = screen.geometry()
        x = (screen_geometry.width() - self.width()) // 2
        y = (screen_geometry.height() - self.height()) // 2
        self.move(x, y)
