"""
🔧 API 调用管理器 - 使用成熟的并发方案解决多线程网络请求问题

问题根源：
- httpx/httpcore 的连接池在 Windows 多线程环境下有线程安全问题
- 多个线程共享同一个连接池时会发生竞争

解决方案（成熟的 Python 并发方案）：
- 使用 **信号量（Semaphore）** 控制并发数，而非全局锁
- 每个请求使用 **独立的客户端实例**，避免共享连接池
- 使用 **requests 库** 作为备选方案（更稳定）
- 适当的超时和重试机制

🔧 2026-01-22 修复：
- 降低并发数从 3 到 2，减少资源竞争
- 添加更完善的异常处理
- 添加资源清理机制
"""

import threading
import time
import gc
import sys
from typing import Optional, Callable, Any, Dict
from logger import logger
import traceback

# 🔧 使用信号量控制并发数（比全局锁更高效）
# 🔧 2026-01-22 修改：降低到最多 2 个并发，提高稳定性
_API_SEMAPHORE = threading.BoundedSemaphore(2)

# 🔧 API 调用计数器 - 用于监控和调试
_api_call_count = 0
_api_call_count_lock = threading.Lock()

# 🔧 API 调用超时设置
API_CALL_TIMEOUT = 60  # 单次 API 调用最大超时时间（秒）
API_CONNECT_TIMEOUT = 10  # 连接超时时间（秒）


def get_api_call_count() -> int:
    """获取 API 调用总次数"""
    with _api_call_count_lock:
        return _api_call_count


def _increment_api_call_count():
    """增加 API 调用计数"""
    global _api_call_count
    with _api_call_count_lock:
        _api_call_count += 1


def _create_isolated_httpx_client(timeout: float = API_CALL_TIMEOUT):
    """
    🔧 创建完全隔离的 httpx 客户端
    
    关键配置：
    - 禁用 HTTP/2（避免多路复用问题）
    - 不保持连接（每次请求都是独立的）
    - 禁用连接池复用
    
    🔧 2026-01-22 修复：添加更严格的隔离和错误处理
    """
    import httpx
    
    try:
        # 🔧 创建独立的 Transport，不共享任何连接
        transport = httpx.HTTPTransport(
            retries=0,  # 不在 transport 层重试
            http2=False,  # 🔧 禁用 HTTP/2，避免多路复用带来的线程安全问题
        )
        
        client = httpx.Client(
            transport=transport,
            timeout=httpx.Timeout(timeout, connect=API_CONNECT_TIMEOUT),
            limits=httpx.Limits(
                max_connections=1,  # 🔧 每个客户端只用一个连接
                max_keepalive_connections=0  # 🔧 不保持连接
            )
        )
        return client
    except Exception as e:
        logger.error(f"❌ 创建 httpx 客户端失败: {str(e)}")
        raise


def _safe_close_client(client):
    """
    🔧 安全关闭客户端并清理资源
    """
    try:
        if client is not None:
            client.close()
    except Exception as e:
        logger.debug(f"关闭客户端时出现异常（可忽略）: {str(e)}")
    finally:
        # 🔧 强制垃圾回收，释放资源
        try:
            gc.collect()
        except:
            pass


def create_safe_openai_client(api_key: str, api_base: str, timeout: float = API_CALL_TIMEOUT):
    """
    🔧 创建安全的 OpenAI 客户端
    
    使用完全隔离的 HTTP 客户端，避免连接池共享问题
    """
    import httpx
    from openai import OpenAI
    
    client = OpenAI(
        base_url=api_base,
        api_key=api_key,
        timeout=httpx.Timeout(timeout, connect=API_CONNECT_TIMEOUT),
        max_retries=1,
        http_client=_create_isolated_httpx_client(timeout)
    )
    
    return client


def create_safe_langchain_http_client(timeout: float = API_CALL_TIMEOUT):
    """
    🔧 创建安全的 LangChain HTTP 客户端
    """
    return _create_isolated_httpx_client(timeout)


def safe_api_call(func: Callable, *args, timeout: float = API_CALL_TIMEOUT, **kwargs) -> Any:
    """
    🔧 安全的 API 调用包装器
    
    使用信号量控制并发，允许多个请求同时进行，但限制最大并发数
    
    Args:
        func: 要调用的函数
        *args: 位置参数
        timeout: 超时时间
        **kwargs: 关键字参数
    
    Returns:
        函数返回值
    """
    _increment_api_call_count()
    
    # 🔧 使用信号量控制并发（比全局锁更高效）
    acquired = _API_SEMAPHORE.acquire(timeout=timeout + 30)
    if not acquired:
        raise TimeoutError(f"等待 API 信号量超时（{timeout + 30}秒）")
    
    try:
        logger.debug(f"🔒 获取 API 信号量，开始调用")
        result = func(*args, **kwargs)
        logger.debug(f"🔓 API 调用完成，释放信号量")
        return result
    except Exception as e:
        logger.warning(f"⚠️ API 调用出错: {str(e)}")
        raise
    finally:
        _API_SEMAPHORE.release()


class SafeOpenAIClient:
    """
    🔧 安全的 OpenAI 客户端包装器
    
    使用成熟的并发控制方案：
    1. 信号量控制并发数（最多 3 个同时进行）
    2. 每次调用创建独立的客户端实例
    3. 调用完成后立即关闭客户端
    4. 自动释放信号量
    
    这样既保证了稳定性，又允许一定程度的并发
    """
    
    def __init__(self, api_key: str, api_base: str, timeout: float = API_CALL_TIMEOUT):
        self.api_key = api_key
        self.api_base = api_base
        self.timeout = timeout
    
    def chat_completion(self, messages: list, model: str, max_tokens: int = 200, **kwargs) -> str:
        """
        🔧 安全的聊天完成调用
        
        Returns:
            响应内容字符串
        """
        def _do_call():
            client = None
            try:
                client = create_safe_openai_client(self.api_key, self.api_base, self.timeout)
                response = client.chat.completions.create(
                    model=model,
                    messages=messages,
                    max_tokens=max_tokens,
                    **kwargs
                )
                return response.choices[0].message.content
            except Exception as e:
                logger.warning(f"⚠️ chat_completion 调用失败: {str(e)}")
                raise
            finally:
                # 🔧 安全关闭客户端，释放所有连接
                _safe_close_client(client)
        
        return safe_api_call(_do_call, timeout=self.timeout)
    
    def file_upload(self, file_path: str, purpose: str = "file-extract"):
        """
        🔧 安全的文件上传调用（用于 Moonshot API）
        """
        from pathlib import Path
        
        def _do_call():
            client = None
            try:
                client = create_safe_openai_client(self.api_key, self.api_base, self.timeout)
                file_object = client.files.create(file=Path(file_path), purpose=purpose)
                return file_object
            except Exception as e:
                logger.warning(f"⚠️ file_upload 调用失败: {str(e)}")
                raise
            finally:
                _safe_close_client(client)
        
        return safe_api_call(_do_call, timeout=self.timeout)
    
    def file_content(self, file_id: str) -> str:
        """
        🔧 安全的文件内容获取调用（用于 Moonshot API）
        """
        def _do_call():
            client = None
            try:
                client = create_safe_openai_client(self.api_key, self.api_base, self.timeout)
                content = client.files.content(file_id=file_id)
                return content.text
            except Exception as e:
                logger.warning(f"⚠️ file_content 调用失败: {str(e)}")
                raise
            finally:
                _safe_close_client(client)
        
        return safe_api_call(_do_call, timeout=self.timeout)
    
    def file_delete(self, file_id: str):
        """
        🔧 安全的文件删除调用（用于 Moonshot API）
        """
        def _do_call():
            client = None
            try:
                client = create_safe_openai_client(self.api_key, self.api_base, self.timeout)
                client.files.delete(file_id=file_id)
            except Exception as e:
                logger.debug(f"文件删除异常（可忽略）: {str(e)}")
            finally:
                _safe_close_client(client)
        
        try:
            safe_api_call(_do_call, timeout=30)  # 删除操作用短超时
        except:
            pass  # 删除失败不影响主流程


# 🔧 全局 Moonshot 客户端（懒加载）
_moonshot_client: Optional[SafeOpenAIClient] = None
_moonshot_client_lock = threading.Lock()


def get_moonshot_client(api_key: str) -> SafeOpenAIClient:
    """获取 Moonshot API 客户端"""
    global _moonshot_client
    with _moonshot_client_lock:
        if _moonshot_client is None or _moonshot_client.api_key != api_key:
            _moonshot_client = SafeOpenAIClient(
                api_key=api_key,
                api_base="https://api.moonshot.cn/v1",
                timeout=API_CALL_TIMEOUT
            )
        return _moonshot_client


def safe_moonshot_recognize(image_path: str, api_key: str, max_chars: int = 5000) -> str:
    """
    🔧 安全的 Moonshot 图片识别
    
    使用信号量控制并发，允许多个请求同时进行
    """
    import os
    import json
    
    logger.info(f"🌙 开始使用Moonshot API识别图片: {os.path.basename(image_path)}")
    
    try:
        client = get_moonshot_client(api_key)
        
        # 上传文件
        file_object = client.file_upload(image_path, "file-extract")
        file_id = file_object.id
        logger.info(f"🔍 文件上传成功，文件ID: {file_id}")
        
        # 获取文件内容
        file_content = client.file_content(file_id)
        
        # 解析内容
        extracted_text = ""
        try:
            content_data = json.loads(file_content)
            extracted_text = content_data.get("content", "")
        except (json.JSONDecodeError, KeyError):
            if file_content and len(file_content.strip()) > 10:
                extracted_text = file_content.strip()
        
        # 清理文件
        client.file_delete(file_id)
        
        if extracted_text and len(extracted_text.strip()) > 5:
            logger.info(f"✅ Moonshot API 识别成功，内容长度: {len(extracted_text)} 字符")
            return extracted_text[:max_chars]
        
        # 如果没有提取到内容，使用简洁对话模式
        logger.info("⚠️ 文件提取未获得有效内容，使用简洁对话模式")
        
        messages = [
            {
                "role": "user",
                "content": f"图片文件 {os.path.basename(image_path)} 无法直接提取文字。请简要描述：1.可能的图片类型 2.建议的分类。限50字内。"
            }
        ]
        
        description = client.chat_completion(
            messages=messages,
            model="moonshot-v1-8k",
            max_tokens=100,
            temperature=0.1
        )
        
        return description[:max_chars] if description else ""
        
    except Exception as e:
        error_msg = str(e)
        if "400" in error_msg or "text extract error" in error_msg or "没有解析出内容" in error_msg:
            logger.warning(f"⚠️ Moonshot API无法解析此图片内容: {error_msg}")
        else:
            logger.error(f"❌ Moonshot API识别失败: {error_msg}")
        return ""


def safe_online_model_recognize(image_path: str, api_key: str, api_base: str, model_name: str, max_chars: int = 5000) -> str:
    """
    🔧 安全的在线模型图片识别（豆包等）
    
    使用信号量控制并发，允许多个请求同时进行
    
    🔧 2026-01-22 修复：添加更完善的异常处理和资源清理
    """
    import os
    import base64
    
    logger.info(f"🤖 开始使用在线模型识别图片: {model_name}, 图片: {os.path.basename(image_path)}")
    
    inner_client = None
    try:
        # 导入压缩函数
        from lib.ai import compress_image_for_api
        
        # 压缩图片
        compressed_image_data = compress_image_for_api(image_path, max_size=1024, quality=70)
        base64_image = base64.b64encode(compressed_image_data).decode('utf-8')
        
        logger.info(f"📦 Base64编码大小: {len(base64_image)/1024:.1f}KB")
        
        # 构建消息
        messages = [
            {
                "role": "user",
                "content": [
                    {
                        "type": "text",
                        "text": "简要描述图片：1.文字内容 2.类型 3.主题（限50字）"
                    },
                    {
                        "type": "image_url",
                        "image_url": {
                            "url": f"data:image/jpeg;base64,{base64_image}"
                        }
                    }
                ]
            }
        ]
        
        # 调用 API
        def _do_call():
            nonlocal inner_client
            inner_client = None
            try:
                inner_client = create_safe_openai_client(api_key, api_base, API_CALL_TIMEOUT)
                response = inner_client.chat.completions.create(
                    model=model_name,
                    messages=messages,
                    max_tokens=200
                )
                return response.choices[0].message.content
            except Exception as e:
                logger.warning(f"⚠️ 在线模型 API 调用异常: {str(e)}")
                raise
            finally:
                _safe_close_client(inner_client)
                inner_client = None
        
        description = safe_api_call(_do_call, timeout=API_CALL_TIMEOUT)
        
        if description:
            logger.info(f"🤖 在线模型 {model_name} 识别成功，内容长度: {len(description)} 字符")
            return description[:max_chars]
        
        return ""
        
    except MemoryError as me:
        logger.error(f"❌ 内存不足: {str(me)}")
        gc.collect()  # 尝试回收内存
        return ""
    except Exception as e:
        logger.error(f"❌ 在线模型 {model_name} 识别失败: {str(e)}")
        return ""
    finally:
        # 🔧 确保资源被清理
        _safe_close_client(inner_client)
        gc.collect()


# 🔧 导出的安全 API 调用函数
__all__ = [
    'safe_api_call',
    'create_safe_openai_client',
    'create_safe_langchain_http_client',
    'SafeOpenAIClient',
    'safe_moonshot_recognize',
    'safe_online_model_recognize',
    'get_api_call_count',
    'API_CALL_TIMEOUT',
    'API_CONNECT_TIMEOUT',
]

