# -*- coding: utf-8 -*-
"""
OCR优化器 - 按照用户要求的优先级顺序进行图片识别

🔧 2026-01-22 修复：
- 添加全面的异常捕获，防止 access violation 导致程序崩溃
- 添加超时保护机制
- 优化多线程调用的稳定性
"""

import os
import sys
import threading
from pathlib import Path
from typing import Optional, Tuple
from concurrent.futures import ThreadPoolExecutor, TimeoutError as FuturesTimeoutError
from logger import logger

# Moonshot API 调用超时时间（秒）
MOONSHOT_API_TIMEOUT = 60

# 🔧 本地模型识别超时时间（秒）
LOCAL_MODEL_TIMEOUT = 45

# 🔧 全局信号量 - 限制同时进行的 OCR 识别任务数量
_OCR_SEMAPHORE = threading.BoundedSemaphore(2)

try:
    import pytesseract
    from PIL import Image
    HAS_TESSERACT = True
except ImportError:
    HAS_TESSERACT = False

try:
    from lib.ai import recognize_image_with_moonshot
except ImportError:
    recognize_image_with_moonshot = lambda *args, **kwargs: ""


class OCROptimizer:
    """OCR优化器 - 按照用户要求的优先级顺序进行图片识别
    
    在线模式（ChatGPT/DeepSeek等文本模型）的优先级：
    VIP用户:
        1. Moonshot API (文件提取)
        2. Tesseract OCR
    
    非VIP用户:
        1. Moonshot API (文件提取) - 首选方案
        2. 豆包API - 如果Moonshot API未识别出内容
        3. Tesseract OCR
    
    本地模式的优先级：
    1. 多模态AI - 如果配置了
    2. 用户自定义API服务
    3. Tesseract OCR
    
    注意：ChatGPT/DeepSeek等文本模型不用于图片识别
    """
    
    def __init__(self):
        self.methods_order = [
            "multimodal_ai",  # 1. 多模态AI (优先)
            "moonshot_api",   # 2. Moonshot API (备用)
            "tesseract",      # 3. 用户配置的Tesseract
        ]
    
    def recognize_image_with_priority(self, image_path: str, max_chars: int = 5000, 
                                    api_key: str = '', is_local: bool = False,
                                    image_recognition_method: str = "tesseract",
                                    online_model: str = None, api_base: str = None, model_name: str = None,
                                    is_vip: bool = False, doubao_info: dict = None) -> Tuple[str, str]:
        """
        按照优先级顺序识别图片
        
        Args:
            image_path: 图片路径
            max_chars: 最大字符数
            api_key: API密钥 (VIP用户使用Moonshot API)
            is_local: 是否本地模型
            image_recognition_method: 用户配置的识别方法
            online_model: 在线模型类型 (chatgpt/deepseek/None)
            is_vip: 是否VIP用户
            doubao_info: 豆包配置信息（非VIP用户使用）
            
        Returns:
            Tuple[str, str]: (识别结果文本, 识别方法)
        """
        
        logger.info(f"🔍 开始按优先级顺序识别图片: {os.path.basename(image_path)}")
        logger.info(f"🎯 模型类型: {online_model}, 本地模型: {is_local}, VIP状态: {is_vip}")
        
        # 🚀 根据模型类型和VIP状态使用不同的识别优先级
        if is_local:
            logger.info("🏠 本地模式识别优先级: 多模态AI -> 用户自定义API -> Tesseract")
        else:
            # 在线模式：VIP和非VIP用户使用不同的策略
            if is_vip:
                logger.info("🌙 在线模式(VIP用户)识别优先级: Moonshot API -> Tesseract OCR")
            else:
                logger.info("🌙🔥 在线模式(非VIP用户)识别优先级: Moonshot API -> 豆包API -> Tesseract OCR")
                logger.info("💡 非VIP用户首先尝试Moonshot API，如果没识别出内容再使用豆包API")
            
            if online_model in ['chatgpt', 'deepseek']:
                logger.info(f"💡 注意：{online_model}为文本模型，不用于图片识别")
        
        # 在线模式的识别逻辑
        if not is_local:
            # 🚀 VIP用户：直接使用Moonshot API，不尝试多模态AI
            if is_vip:
                if api_key:
                    logger.info("🌙 VIP用户：使用专属Moonshot API进行图片识别")
                    text, method = self._try_moonshot_api(image_path, api_key, max_chars)
                    if text:
                        return text, method
                    else:
                        logger.info("⚠️  Moonshot API识别失败，将尝试Tesseract OCR")
                else:
                    logger.warning("⚠️  VIP用户：未提供Moonshot API密钥")
            
            # 🚀 非VIP用户：首先尝试Moonshot API，如果没识别出内容再用豆包API
            else:
                # 1. 先尝试Moonshot API
                if api_key:
                    logger.info("🌙 非VIP用户：首先尝试使用Moonshot API进行图片识别")
                    text, method = self._try_moonshot_api(image_path, api_key, max_chars)
                    if text:
                        logger.info("✅ Moonshot API识别成功，使用该结果")
                        return text, method
                    else:
                        logger.info("⚠️  Moonshot API未识别出内容，将尝试豆包API")
                else:
                    logger.info("⚠️  非VIP用户：未提供Moonshot API密钥，直接尝试豆包API")
                
                # 2. 如果Moonshot API没识别出东西，再尝试豆包API
                if doubao_info and doubao_info.get('key'):
                    logger.info("🔥 非VIP用户：使用豆包API进行图片识别（备用方案）")
                    text, method = self._try_doubao_api(image_path, doubao_info, max_chars)
                    if text:
                        return text, method
                    else:
                        logger.info("⚠️  豆包API识别失败，将尝试Tesseract OCR")
                else:
                    logger.warning("⚠️  非VIP用户：未配置豆包API")
        
        # 本地模式：可以尝试多模态AI（如果用户手动配置了）
        if is_local:
            logger.info("🏠 本地模式：尝试使用多模态AI服务进行图片识别")
            text, method = self._try_multimodal_ai(image_path)
            if text:
                return text, method
            
            # 如果多模态AI失败，尝试用户自定义API服务（作为备用）
            logger.info("🏠 本地模式：多模态AI失败，尝试使用用户自定义API服务")
            text, method = self._try_local_model_recognition(image_path, api_key, api_base, model_name, max_chars)
            if text:
                return text, method
        
        # 最后尝试用户配置的Tesseract OCR
        text, method = self._try_tesseract(image_path, image_recognition_method)
        if text:
            return text, method
        
        # 所有方法都失败，返回空结果（load_document会提取基本图片信息）
        logger.warning(f"❌ 图片 {os.path.basename(image_path)} 所有识别方法均失败")
        logger.info("🔄 将跳过此图片或使用基本图片信息")
        return "", "识别失败"
    
    def _try_tesseract(self, image_path: str, image_recognition_method: str) -> Tuple[str, str]:
        """尝试使用Tesseract OCR"""
        if not HAS_TESSERACT:
            logger.info("⚠️  pytesseract未安装，跳过Tesseract方法")
            return "", ""
        
        if image_recognition_method not in ["tesseract", "both"]:
            logger.info("🔧 用户未配置Tesseract方法，跳过")
            return "", ""
        
        processed_image_path = None
        try:
            logger.info("🎯 优先使用Tesseract OCR（用户配置）")
            
            # 🚀 优化：使用增强版图片预处理提高识别率
            try:
                from lib.ai import preprocess_image_for_ocr
                processed_image_path = preprocess_image_for_ocr(image_path)
                logger.info(f"图片预处理完成: {image_path} -> {processed_image_path}")
            except Exception as preprocess_e:
                logger.warning(f"图片预处理失败，使用原始图片: {str(preprocess_e)}")
                processed_image_path = image_path
            
            # 使用预处理后的图片进行OCR
            img = Image.open(processed_image_path)
            tesseract_text = pytesseract.image_to_string(img, lang='chi_sim+eng')
            img.close()
            
            if tesseract_text and len(tesseract_text.strip()) > 5:
                logger.info(f"✅ Tesseract识别成功，内容长度: {len(tesseract_text)} 字符")
                logger.info(f"【Tesseract识别内容】 {os.path.basename(image_path)}:\n{tesseract_text}")
                return tesseract_text, "Tesseract OCR (用户配置)"
            else:
                logger.info("⚠️  Tesseract识别结果为空或过短")
                return "", ""
                
        except Exception as e:
            logger.warning(f"❌ Tesseract识别失败: {str(e)}")
            return "", ""
        finally:
            # 🚀 优化：清理临时预处理文件
            if processed_image_path and processed_image_path != image_path:
                try:
                    if os.path.exists(processed_image_path):
                        os.remove(processed_image_path)
                        logger.debug(f"成功删除临时预处理文件: {processed_image_path}")
                except Exception as cleanup_e:
                    logger.warning(f"清理临时预处理文件失败: {str(cleanup_e)}")
    
    def _try_multimodal_ai(self, image_path: str) -> Tuple[str, str]:
        """
        尝试使用多模态AI
        
        🔧 修复：添加超时保护和异常处理
        """
        logger.info("🔍 开始尝试使用多模态AI服务")
        
        # 🔧 使用信号量控制并发
        acquired = _OCR_SEMAPHORE.acquire(timeout=LOCAL_MODEL_TIMEOUT + 10)
        if not acquired:
            logger.warning("⚠️  等待 OCR 信号量超时，跳过多模态AI识别")
            return "", ""
        
        try:
            from lib.multimodal_ai import multimodal_service
            
            # 🔧 使用超时保护检查可用性
            is_available = False
            try:
                with ThreadPoolExecutor(max_workers=1) as executor:
                    future = executor.submit(multimodal_service.is_available)
                    is_available = future.result(timeout=20)  # 20秒超时
            except FuturesTimeoutError:
                logger.warning("⚠️  多模态AI可用性检查超时")
                return "", ""
            except Exception as check_e:
                logger.warning(f"⚠️  多模态AI可用性检查异常: {str(check_e)}")
                return "", ""
            
            logger.info(f"📊 多模态AI服务可用性检查结果: {is_available}")
            
            if not is_available:
                logger.warning("❌ 多模态AI服务未配置或不可用，尝试重新加载配置")
                
                # 尝试重新加载配置
                reload_result = multimodal_service.reload_config()
                logger.info(f"🔄 配置重新加载结果: {reload_result}")
                
                # 重新检查是否可用
                is_available_after_reload = multimodal_service.is_available()
                logger.info(f"📊 重新加载后多模态AI服务可用性: {is_available_after_reload}")
                
                if not is_available_after_reload:
                    logger.warning("❌ 重新加载后多模态AI服务仍不可用")
                
                # 详细检查配置信息
                logger.info("🔍 多模态AI配置检查:")
                
                # 检查依赖
                try:
                    import requests, base64
                    from PIL import Image
                    logger.info("  ✅ 基本依赖: requests, base64, PIL 已安装")
                except ImportError as dep_e:
                    logger.warning(f"  ❌ 基本依赖缺失: {str(dep_e)}")
                
                # 检查API密钥配置
                logger.info(f"  - API密钥字典: {'已配置' if multimodal_service.api_keys else '未配置'}")
                if multimodal_service.api_keys:
                    for key, value in multimodal_service.api_keys.items():
                        logger.info(f"    - {key}: {'已配置' if value else '未配置'}")
                    
                    # 特别检查 multimodal 提供商
                    if 'multimodal' in multimodal_service.api_keys:
                        has_api_base = bool(multimodal_service.api_base)
                        has_model = bool(multimodal_service.model_name)
                        logger.info(f"  - multimodal 提供商完整性检查:")
                        logger.info(f"    - API密钥: ✅")
                        logger.info(f"    - API地址: {'✅' if has_api_base else '❌'}")
                        logger.info(f"    - 模型名称: {'✅' if has_model else '❌'}")
                        logger.info(f"    - 整体可用性: {'✅' if has_api_base and has_model else '❌'}")
                else:
                    logger.warning("  ❌ 没有配置任何API密钥")
                
                # 检查API配置
                logger.info(f"  - API地址: {multimodal_service.api_base or '未配置'}")
                logger.info(f"  - 模型名称: {multimodal_service.model_name or '未配置'}")
                
                # 检查配置文件
                try:
                    config_file = os.path.join(os.path.expanduser('~'), '.fileneatai', 'multimodal_settings.json')
                    logger.info(f"  - 配置文件: {config_file}")
                    logger.info(f"  - 配置文件存在: {'是' if os.path.exists(config_file) else '否'}")
                    
                    if os.path.exists(config_file):
                        try:
                            import json
                            with open(config_file, 'r', encoding='utf-8') as f:
                                settings = json.load(f)
                            multimodal_config = settings.get('multimodal', {})
                            logger.info(f"  - 多模态配置启用: {'是' if multimodal_config.get('enabled') else '否'}")
                            logger.info(f"  - 配置文件中的API密钥: {'已配置' if multimodal_config.get('api_key') else '未配置'}")
                            logger.info(f"  - 配置文件中的API地址: {multimodal_config.get('api_base', '未配置')}")
                            logger.info(f"  - 配置文件中的模型: {multimodal_config.get('model', '未配置')}")
                        except Exception as config_e:
                            logger.warning(f"  ❌ 读取配置文件失败: {str(config_e)}")
                except Exception as file_e:
                    logger.warning(f"  ❌ 检查配置文件失败: {str(file_e)}")
                
                return "", ""
            
            logger.info("🤖 尝试使用多模态AI进行图片识别")
            
            # 🔧 使用超时保护进行图片识别
            success, content, cost, provider = False, "", 0.0, ""
            try:
                with ThreadPoolExecutor(max_workers=1) as executor:
                    future = executor.submit(
                        multimodal_service.recognize_image_smart,
                        image_path=image_path,
                        prompt="请详细识别这张图片中的所有文字内容，包括中文和英文。"
                    )
                    result = future.result(timeout=LOCAL_MODEL_TIMEOUT)
                    success, content, cost, provider = result
            except FuturesTimeoutError:
                logger.warning(f"⚠️  多模态AI识别超时（{LOCAL_MODEL_TIMEOUT}秒）")
                return "", ""
            except Exception as recognize_e:
                logger.warning(f"⚠️  多模态AI识别异常: {str(recognize_e)}")
                return "", ""
            
            if success and content and len(content.strip()) > 10:
                logger.info(f"✅ 多模态AI识别成功，服务商: {provider}，成本: ${cost:.4f}")
                logger.info(f"【多模态AI识别内容】 {os.path.basename(image_path)}:\n{content}")
                return content, f"多模态AI ({provider})"
            else:
                # 详细显示失败信息，就像用户自定义API服务那样
                logger.error(f"❌ 多模态AI识别失败: {content}")
                logger.error(f"  - 成功状态: {success}")
                logger.error(f"  - 内容长度: {len(content) if content else 0}")
                logger.error(f"  - 服务商: {provider}")
                logger.error(f"  - 成本: ${cost:.4f}")
                if content and not success:
                    # 如果有错误内容，显示完整错误信息
                    logger.error(f"  - 详细错误: {content}")
                return "", ""
                
        except ImportError as e:
            logger.error(f"❌ 多模态AI服务导入失败: {str(e)}")
            return "", ""
        except MemoryError as me:
            logger.error(f"❌ 内存不足: {str(me)}")
            return "", ""
        except Exception as e:
            logger.error(f"❌ 多模态AI识别过程中出现异常: {str(e)}")
            logger.error(f"  - 异常类型: {type(e).__name__}")
            import traceback
            logger.error(f"  - 详细堆栈: {traceback.format_exc()}")
            return "", ""
        finally:
            # 🔧 确保信号量被释放
            _OCR_SEMAPHORE.release()
    
    def _try_local_model_recognition(self, image_path: str, api_key: str, api_base: str, model_name: str, max_chars: int) -> Tuple[str, str]:
        """
        尝试使用本地模式的用户自定义API服务进行图片识别
        
        🔧 修复：添加超时保护和全面的异常处理，防止 access violation
        """
        if not api_key or not api_base or not model_name:
            logger.info("⚠️  本地模式缺少API配置信息，跳过自定义API服务识别")
            return "", ""
        
        # 🔧 使用信号量控制并发
        acquired = _OCR_SEMAPHORE.acquire(timeout=LOCAL_MODEL_TIMEOUT + 10)
        if not acquired:
            logger.warning("⚠️  等待 OCR 信号量超时，跳过本地模型识别")
            return "", ""
        
        try:
            from lib.ai import recognize_image_with_online_model
            logger.info(f"🏠 本地模式：使用用户自定义API服务识别图片")
            logger.info(f"🔧 API服务: {api_base}, 模型: {model_name}")
            
            # 🔧 使用 ThreadPoolExecutor 添加超时保护
            result_text = ""
            with ThreadPoolExecutor(max_workers=1) as executor:
                future = executor.submit(
                    recognize_image_with_online_model,
                    image_path=image_path,
                    api_key=api_key,
                    api_base=api_base,
                    model_name=model_name,
                    max_chars=max_chars
                )
                try:
                    result_text = future.result(timeout=LOCAL_MODEL_TIMEOUT)
                except FuturesTimeoutError:
                    logger.warning(f"⚠️  本地自定义API服务识别超时（{LOCAL_MODEL_TIMEOUT}秒）")
                    return "", ""
                except Exception as future_e:
                    logger.warning(f"⚠️  本地自定义API服务识别异常: {str(future_e)}")
                    return "", ""
            
            if result_text and len(result_text.strip()) > 10:
                logger.info(f"✅ 本地自定义API服务识别成功，内容长度: {len(result_text)} 字符")
                logger.info(f"📄 识别结果：\n{result_text}")
                return result_text, f"本地自定义API服务 ({model_name})"
            else:
                logger.info(f"⚠️  本地自定义API服务识别结果为空或过短")
                return "", ""
                
        except ImportError as ie:
            logger.warning(f"❌ 导入 recognize_image_with_online_model 失败: {str(ie)}")
            return "", ""
        except MemoryError as me:
            logger.error(f"❌ 内存不足: {str(me)}")
            return "", ""
        except Exception as e:
            # 🔧 捕获所有异常，包括可能的 C 扩展异常
            logger.warning(f"❌ 本地自定义API服务识别失败: {str(e)}")
            import traceback
            logger.debug(f"详细错误: {traceback.format_exc()}")
            return "", ""
        finally:
            _OCR_SEMAPHORE.release()
    
    # ❌ 已移除 _try_online_model 方法
    # ChatGPT/DeepSeek等文本模型不适合图片识别，已改为使用多模态AI或Moonshot API
    
    def _try_moonshot_api(self, image_path: str, api_key: str, max_chars: int) -> Tuple[str, str]:
        """尝试使用Moonshot API - VIP用户优先,非VIP用户作为首选方案"""
        
        # 🚀 优先使用传入的API密钥，如果没有则从配置文件中获取Moonshot专用密钥
        moonshot_api_key = api_key  # 优先使用传入的密钥
        
        # 如果传入的密钥为空，尝试从多模态设置中获取专门的Moonshot密钥
        if not moonshot_api_key:
            moonshot_api_key = self._get_moonshot_api_key()
            
        if not moonshot_api_key:
            logger.info("⚠️  未提供API密钥且未配置Moonshot专用API密钥，跳过Moonshot API")
            logger.info("💡 提示：请传入API密钥或在多模态设置中配置Moonshot API密钥")
            return "", ""
        
        try:
            logger.info("🌙 使用 Moonshot API 提取图片文字")
            logger.info(f"🔑 使用的Moonshot API密钥: {moonshot_api_key[:20]}..." if moonshot_api_key and len(moonshot_api_key) > 20 else f"🔑 使用的Moonshot API密钥: {moonshot_api_key}")
            
            # 🔧 添加超时保护，防止网络请求永久阻塞导致程序卡死
            moonshot_text = ""
            with ThreadPoolExecutor(max_workers=1) as executor:
                future = executor.submit(recognize_image_with_moonshot, image_path, moonshot_api_key, max_chars)
                try:
                    moonshot_text = future.result(timeout=MOONSHOT_API_TIMEOUT)
                except FuturesTimeoutError:
                    logger.warning(f"⚠️ Moonshot API 请求超时（{MOONSHOT_API_TIMEOUT}秒），跳过此图片")
                    return "", ""
            
            if moonshot_text and len(moonshot_text.strip()) > 10:
                logger.info(f"✅ Moonshot API 文字提取成功，内容长度: {len(moonshot_text)} 字符")
                logger.info(f"📄 Moonshot API 提取的文字内容：\n{moonshot_text}")
                return moonshot_text, "Moonshot API"
            else:
                logger.info("⚠️  Moonshot API 未提取到有效文字内容")
                return "", ""
                
        except FuturesTimeoutError:
            logger.warning(f"⚠️ Moonshot API 请求超时（{MOONSHOT_API_TIMEOUT}秒），跳过此图片")
            return "", ""
        except Exception as e:
            error_msg = str(e)
            if "401" in error_msg or "Invalid Authentication" in error_msg:
                logger.error(f"❌ Moonshot API 认证失败，请检查API密钥是否正确: {error_msg}")
                logger.info("💡 解决方案：请检查传入的API密钥是否为有效的Moonshot API密钥")
            else:
                logger.warning(f"❌ Moonshot API 文字提取失败: {error_msg}")
            return "", ""
    
    def _try_doubao_api(self, image_path: str, doubao_info: dict, max_chars: int) -> Tuple[str, str]:
        """尝试使用豆包API - 非VIP用户的备用方案"""
        
        if not doubao_info or not doubao_info.get('key'):
            logger.info("⚠️  未配置豆包API密钥，跳过豆包API")
            return "", ""
        
        try:
            from lib.ai import recognize_image_with_online_model
            
            doubao_key = doubao_info.get('key', '')
            doubao_url = doubao_info.get('url', 'https://ark.cn-beijing.volces.com/api/v3')
            doubao_model = doubao_info.get('model', 'doubao-seed-1-6-vision-250815')
            
            logger.info("🔥 使用豆包API提取图片文字")
            logger.info(f"🔑 豆包API密钥: {doubao_key[:20]}..." if doubao_key and len(doubao_key) > 20 else f"🔑 豆包API密钥: {doubao_key}")
            logger.info(f"🌐 豆包API地址: {doubao_url}")
            logger.info(f"🤖 豆包模型: {doubao_model}")
            
            # 🔧 添加超时保护，防止API调用永久阻塞导致崩溃
            doubao_text = ""
            with ThreadPoolExecutor(max_workers=1) as executor:
                future = executor.submit(
                    recognize_image_with_online_model,
                    image_path=image_path,
                    api_key=doubao_key,
                    api_base=doubao_url,
                    model_name=doubao_model,
                    max_chars=max_chars
                )
                try:
                    doubao_text = future.result(timeout=MOONSHOT_API_TIMEOUT)  # 使用相同的超时时间
                except FuturesTimeoutError:
                    logger.warning(f"⚠️ 豆包API 请求超时（{MOONSHOT_API_TIMEOUT}秒），跳过此图片")
                    return "", ""
            
            if doubao_text and len(doubao_text.strip()) > 10:
                logger.info(f"✅ 豆包API 文字提取成功，内容长度: {len(doubao_text)} 字符")
                logger.info(f"📄 豆包API 提取的文字内容：\n{doubao_text}")
                return doubao_text, "豆包API"
            else:
                logger.info("⚠️  豆包API 未提取到有效文字内容")
                return "", ""
        
        except FuturesTimeoutError:
            logger.warning(f"⚠️ 豆包API 请求超时（{MOONSHOT_API_TIMEOUT}秒），跳过此图片")
            return "", ""
        except Exception as e:
            error_msg = str(e)
            if "401" in error_msg or "Invalid Authentication" in error_msg or "authentication" in error_msg.lower():
                logger.error(f"❌ 豆包API 认证失败，请检查API密钥是否正确: {error_msg}")
                logger.info("💡 解决方案：请检查服务器返回的豆包API密钥是否有效")
            else:
                logger.warning(f"❌ 豆包API 文字提取失败: {error_msg}")
            return "", ""
    
    def _get_moonshot_api_key(self) -> str:
        """获取Moonshot专用API密钥"""
        try:
            import json
            
            # 从多模态设置文件中读取Moonshot密钥
            config_file = os.path.join(os.path.expanduser('~'), '.fileneatai', 'multimodal_settings.json')
            if os.path.exists(config_file):
                with open(config_file, 'r', encoding='utf-8') as f:
                    settings = json.load(f)
                
                # 检查是否配置了Moonshot API
                multimodal_config = settings.get('multimodal', {})
                if multimodal_config.get('enabled', False):
                    api_base = multimodal_config.get('api_base', '').strip()
                    api_key = multimodal_config.get('api_key', '').strip()
                    
                    # 判断是否是Moonshot API配置
                    if 'moonshot.cn' in api_base.lower() and api_key:
                        logger.info("✅ 找到Moonshot专用API密钥配置")
                        return api_key
                    elif api_key and not api_base:
                        # 如果只配置了密钥没有配置地址，假设是Moonshot
                        logger.info("✅ 使用多模态设置中的API密钥作为Moonshot密钥")
                        return api_key
            
            logger.info("⚠️  未找到Moonshot专用API密钥配置")
            return ""
            
        except Exception as e:
            logger.warning(f"读取Moonshot API密钥配置失败: {str(e)}")
            return ""


# 全局实例
ocr_optimizer = OCROptimizer()
