#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
检查打包所需的图标文件
"""

import sys
import platform
from pathlib import Path

# Windows 控制台编码修复
if platform.system() == 'Windows':
    import io
    sys.stdout = io.TextIOWrapper(sys.stdout.buffer, encoding='utf-8')
    sys.stderr = io.TextIOWrapper(sys.stderr.buffer, encoding='utf-8')

def check_icons():
    """检查图标文件是否存在"""
    project_dir = Path(__file__).parent
    
    icons = {
        'windows': {
            'file': 'logo.ico',
            'required': True,
            'description': 'Windows 图标文件 (.ico)'
        },
        'macos': {
            'file': 'logo.icns',
            'required': True,
            'description': 'macOS 图标文件 (.icns)'
        },
        'png': {
            'file': 'logo.png',
            'required': False,
            'description': 'PNG 图标文件 (可选，用于转换)'
        }
    }
    
    print("=" * 60)
    print("图标文件检查")
    print("=" * 60)
    print()
    
    all_ok = True
    
    for platform, info in icons.items():
        icon_path = project_dir / info['file']
        exists = icon_path.exists()
        
        status = "✓" if exists else ("❌" if info['required'] else "⚠")
        requirement = "必需" if info['required'] else "可选"
        
        print(f"{status} [{platform.upper():7}] {info['file']:15} - {info['description']} ({requirement})")
        
        if exists:
            size_kb = icon_path.stat().st_size / 1024
            print(f"           路径: {icon_path}")
            print(f"           大小: {size_kb:.1f} KB")
        else:
            if info['required']:
                all_ok = False
                print(f"           ⚠ 缺失！打包可能失败")
        
        print()
    
    print("=" * 60)
    
    if not all_ok:
        print()
        print("❌ 缺少必需的图标文件")
        print()
        print("解决方案:")
        print()
        
        # Windows 图标
        if not (project_dir / 'logo.ico').exists():
            print("1. Windows 图标 (logo.ico):")
            if (project_dir / 'logo.png').exists():
                print("   - 在线转换: https://convertio.co/zh/png-ico/")
                print("   - 或使用 ImageMagick:")
                print("     convert logo.png -define icon:auto-resize=256,128,64,48,32,16 logo.ico")
            else:
                print("   - 请提供 PNG 格式的 logo 图片")
                print("   - 推荐尺寸: 256x256 或更大")
            print()
        
        # macOS 图标
        if not (project_dir / 'logo.icns').exists():
            print("2. macOS 图标 (logo.icns):")
            if (project_dir / 'logo.png').exists():
                print("   - 在线转换: https://cloudconvert.com/png-to-icns")
                print("   - 或在 macOS 上使用命令:")
                print("     mkdir logo.iconset")
                print("     sips -z 16 16 logo.png --out logo.iconset/icon_16x16.png")
                print("     sips -z 32 32 logo.png --out logo.iconset/icon_16x16@2x.png")
                print("     # ... (其他尺寸)")
                print("     iconutil -c icns logo.iconset")
            else:
                print("   - 请提供 PNG 格式的 logo 图片")
                print("   - 推荐尺寸: 1024x1024")
            print()
        
        return False
    else:
        print("✓ 所有必需的图标文件都存在")
        print()
        return True


if __name__ == "__main__":
    success = check_icons()
    sys.exit(0 if success else 1)

