"""
收集 Windows 运行时 DLL 的脚本
用于解决打包后的 "找不到指定的模块" 错误
"""
import os
import sys
import shutil
from pathlib import Path

def find_dll(dll_name, search_paths):
    """在指定路径中查找 DLL"""
    for path in search_paths:
        dll_path = Path(path) / dll_name
        if dll_path.exists():
            return str(dll_path)
    return None

def collect_runtime_dlls():
    """收集所有必要的运行时 DLL"""
    # 搜索路径
    search_paths = [
        r"C:\Windows\System32",
        r"C:\Windows\SysWOW64",
        os.path.join(sys.base_prefix),  # Python 安装目录
    ]
    
    # 需要的 DLL 列表
    required_dlls = [
        # Visual C++ Runtime
        "vcruntime140.dll",
        "vcruntime140_1.dll",
        "msvcp140.dll",
        "msvcp140_1.dll",
        "msvcp140_2.dll",
        # Universal CRT (部分关键的)
        "api-ms-win-crt-runtime-l1-1-0.dll",
        "api-ms-win-crt-stdio-l1-1-0.dll",
        "api-ms-win-crt-heap-l1-1-0.dll",
        "api-ms-win-crt-locale-l1-1-0.dll",
        "api-ms-win-crt-math-l1-1-0.dll",
        "api-ms-win-crt-string-l1-1-0.dll",
        "api-ms-win-crt-time-l1-1-0.dll",
        "api-ms-win-crt-filesystem-l1-1-0.dll",
        "api-ms-win-crt-environment-l1-1-0.dll",
        "api-ms-win-crt-convert-l1-1-0.dll",
        # Python DLL
        "python311.dll",
    ]
    
    found_dlls = []
    missing_dlls = []
    
    print("=" * 60)
    print("Collecting Runtime DLLs")
    print("=" * 60)
    print()
    
    for dll_name in required_dlls:
        dll_path = find_dll(dll_name, search_paths)
        if dll_path:
            found_dlls.append((dll_name, dll_path))
            print(f"[OK] Found: {dll_name}")
            print(f"     Path: {dll_path}")
        else:
            missing_dlls.append(dll_name)
            print(f"[WARN] Missing: {dll_name}")
        print()
    
    # 创建 runtime_dlls 目录
    runtime_dir = Path("runtime_dlls")
    runtime_dir.mkdir(exist_ok=True)
    
    # 复制找到的 DLL
    print("=" * 60)
    print("Copying DLLs to runtime_dlls/")
    print("=" * 60)
    print()
    
    for dll_name, dll_path in found_dlls:
        dest_path = runtime_dir / dll_name
        try:
            shutil.copy2(dll_path, dest_path)
            print(f"[OK] Copied: {dll_name}")
        except Exception as e:
            print(f"[ERROR] Failed to copy {dll_name}: {e}")
    
    print()
    print("=" * 60)
    print("Summary")
    print("=" * 60)
    print(f"Found: {len(found_dlls)}/{len(required_dlls)} DLLs")
    print(f"Missing: {len(missing_dlls)} DLLs")
    if missing_dlls:
        print("\nMissing DLLs:")
        for dll in missing_dlls:
            print(f"  - {dll}")
        print("\nNote: Some api-ms-win-* DLLs may not be available for copying")
        print("      Users will need to install Visual C++ Redistributable")
    
    # 生成 PyInstaller binaries 配置
    print()
    print("=" * 60)
    print("PyInstaller Configuration (for main.spec)")
    print("=" * 60)
    print()
    print("Add these lines to binaries in main.spec:")
    print()
    for dll_name, _ in found_dlls:
        print(f"    ('runtime_dlls/{dll_name}', '.'),")
    
    return found_dlls, missing_dlls

if __name__ == "__main__":
    found, missing = collect_runtime_dlls()
    print()
    input("Press Enter to exit...")

