"""
FileNeatAI 崩溃日志处理模块
用于捕获和记录程序崩溃信息，便于问题排查
增强版：添加用户友好的异常弹窗功能
"""

import sys
import os
import traceback
import platform
import datetime
import threading
import faulthandler
import atexit
from pathlib import Path
from functools import wraps

# 全局变量存储崩溃日志路径
_crash_log_path = None
_is_initialized = False
_main_window = None  # 主窗口引用，用于显示弹窗


def get_crash_log_path():
    """获取崩溃日志文件路径"""
    global _crash_log_path
    
    if _crash_log_path:
        return _crash_log_path
    
    # 确定日志目录
    if getattr(sys, 'frozen', False):
        # 打包后的exe
        base_dir = os.path.dirname(sys.executable)
    else:
        # 开发环境
        base_dir = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
    
    # 创建 logs 目录
    log_dir = os.path.join(base_dir, 'logs')
    try:
        if not os.path.exists(log_dir):
            os.makedirs(log_dir)
    except:
        # 如果无法创建，使用临时目录
        import tempfile
        log_dir = tempfile.gettempdir()
    
    _crash_log_path = os.path.join(log_dir, 'crash.log')
    return _crash_log_path


def get_system_info():
    """收集系统信息"""
    info = []
    info.append(f"操作系统: {platform.system()} {platform.release()} ({platform.version()})")
    info.append(f"架构: {platform.machine()}")
    info.append(f"Python版本: {platform.python_version()}")
    info.append(f"Python实现: {platform.python_implementation()}")
    
    # 尝试获取内存信息
    try:
        import psutil
        mem = psutil.virtual_memory()
        info.append(f"内存: 总计 {mem.total / (1024**3):.1f}GB, 可用 {mem.available / (1024**3):.1f}GB, 使用率 {mem.percent}%")
        
        # 当前进程信息
        process = psutil.Process()
        mem_info = process.memory_info()
        info.append(f"进程内存: RSS {mem_info.rss / (1024**2):.1f}MB, VMS {mem_info.vms / (1024**2):.1f}MB")
    except:
        pass
    
    # Qt/PySide6 版本
    try:
        from PySide6.QtCore import qVersion
        info.append(f"Qt版本: {qVersion()}")
    except:
        pass
    
    try:
        import PySide6
        info.append(f"PySide6版本: {PySide6.__version__}")
    except:
        pass
    
    # 当前工作目录
    info.append(f"当前目录: {os.getcwd()}")
    
    # 可执行文件路径
    if getattr(sys, 'frozen', False):
        info.append(f"可执行文件: {sys.executable}")
    
    return "\n".join(info)


def get_thread_info():
    """获取当前线程信息"""
    info = []
    current = threading.current_thread()
    info.append(f"当前线程: {current.name} (ID: {current.ident})")
    
    # 列出所有活动线程
    info.append(f"活动线程数: {threading.active_count()}")
    for thread in threading.enumerate():
        info.append(f"  - {thread.name} (daemon: {thread.daemon}, alive: {thread.is_alive()})")
    
    return "\n".join(info)


def write_crash_log(error_type, error_value, error_tb, context=""):
    """写入崩溃日志
    
    🔧 安全改进：在多线程环境下，跳过获取所有线程堆栈的操作，
    因为 sys._current_frames() 在多线程并发时可能导致 access violation 崩溃
    """
    crash_log_path = get_crash_log_path()
    
    timestamp = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    separator = "=" * 80
    
    log_content = []
    log_content.append(f"\n{separator}")
    log_content.append(f"崩溃时间: {timestamp}")
    log_content.append(separator)
    
    # 上下文信息
    if context:
        log_content.append(f"\n【上下文】")
        log_content.append(context)
    
    # 错误信息
    log_content.append(f"\n【错误类型】")
    log_content.append(f"{error_type.__name__ if error_type else 'Unknown'}: {error_value}")
    
    # 完整堆栈追踪
    log_content.append(f"\n【堆栈追踪】")
    if error_tb:
        tb_lines = traceback.format_exception(error_type, error_value, error_tb)
        log_content.append("".join(tb_lines))
    
    # 系统信息
    log_content.append(f"\n【系统信息】")
    log_content.append(get_system_info())
    
    # 线程信息
    log_content.append(f"\n【线程信息】")
    log_content.append(get_thread_info())
    
    # 🔧 安全改进：只在主线程或线程数较少时获取所有线程堆栈
    # sys._current_frames() 在多线程并发执行底层操作（如 SSL）时可能导致 access violation
    active_thread_count = threading.active_count()
    is_main_thread = threading.current_thread() is threading.main_thread()
    
    if is_main_thread and active_thread_count <= 5:
        # 只在主线程且线程数较少时才安全地获取所有线程堆栈
        log_content.append(f"\n【所有线程堆栈】")
        try:
            for thread_id, frame in sys._current_frames().items():
                log_content.append(f"\n--- 线程 {thread_id} ---")
                log_content.append("".join(traceback.format_stack(frame)))
        except:
            log_content.append("无法获取线程堆栈")
    else:
        # 多线程环境下跳过，避免 access violation
        log_content.append(f"\n【所有线程堆栈】")
        log_content.append(f"跳过（多线程环境，活动线程数: {active_thread_count}，当前线程: {threading.current_thread().name}）")
        log_content.append("提示：在多线程环境下获取所有线程堆栈可能导致 access violation")
    
    log_content.append(f"\n{separator}\n")
    
    # 写入文件
    try:
        with open(crash_log_path, 'a', encoding='utf-8') as f:
            f.write("\n".join(log_content))
        # 安全打印
        if sys.stdout is not None:
            try:
                print(f"崩溃日志已保存到: {crash_log_path}")
            except:
                pass
    except Exception as e:
        # 安全打印错误信息
        if sys.stderr is not None:
            try:
                print(f"无法写入崩溃日志: {e}", file=sys.stderr)
                print("\n".join(log_content), file=sys.stderr)
            except:
                pass


def global_exception_handler(exctype, value, tb):
    """全局异常处理器 - 增强版：记录日志并显示用户友好的弹窗"""
    # 忽略 KeyboardInterrupt
    if issubclass(exctype, KeyboardInterrupt):
        sys.__excepthook__(exctype, value, tb)
        return
    
    # 写入崩溃日志
    write_crash_log(exctype, value, tb, context="全局未捕获异常")
    
    # 尝试记录到 logger
    try:
        from logger import logger
        error_msg = ''.join(traceback.format_exception(exctype, value, tb))
        logger.critical(f"未捕获的全局异常:\n{error_msg}")
    except:
        pass
    
    # 尝试显示用户友好的弹窗
    try:
        error_type = exctype.__name__
        # 判断是否是网络相关错误
        if any(keyword in error_type for keyword in ['Connection', 'Timeout', 'Proxy', 'SSL', 'HTTP', 'Network', 'Socket']):
            show_network_error_dialog(value)
        else:
            show_crash_dialog(value, "程序运行时发生未捕获的异常")
    except:
        pass
    
    # 调用默认处理器
    sys.__excepthook__(exctype, value, tb)


def thread_exception_handler(args):
    """线程异常处理器 (Python 3.8+)"""
    exctype = args.exc_type
    value = args.exc_value
    tb = args.exc_traceback
    thread = args.thread
    
    context = f"线程 '{thread.name}' (ID: {thread.ident}) 中发生异常"
    write_crash_log(exctype, value, tb, context=context)
    
    # 尝试记录到 logger
    try:
        from logger import logger
        error_msg = ''.join(traceback.format_exception(exctype, value, tb))
        logger.critical(f"线程异常 [{thread.name}]:\n{error_msg}")
    except:
        pass


def qt_message_handler(mode, context, message):
    """Qt 消息处理器"""
    from PySide6.QtCore import QtMsgType
    
    # 根据消息类型处理
    if mode == QtMsgType.QtFatalMsg:
        # 致命错误，写入崩溃日志
        write_crash_log(
            RuntimeError, 
            RuntimeError(f"Qt Fatal: {message}"), 
            None,
            context=f"Qt致命错误\n文件: {context.file}\n行: {context.line}\n函数: {context.function}"
        )
    elif mode == QtMsgType.QtCriticalMsg:
        # 严重错误
        try:
            from logger import logger
            logger.critical(f"Qt Critical: {message} (at {context.file}:{context.line})")
        except:
            pass
    elif mode == QtMsgType.QtWarningMsg:
        # 警告（只记录重要的）
        if "QThread" in message or "segfault" in message.lower() or "crash" in message.lower():
            try:
                from logger import logger
                logger.warning(f"Qt Warning: {message}")
            except:
                pass


def on_exit():
    """程序退出时的处理"""
    crash_log_path = get_crash_log_path()
    
    # 记录正常退出
    try:
        timestamp = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        with open(crash_log_path, 'a', encoding='utf-8') as f:
            f.write(f"\n[{timestamp}] 程序正常退出\n")
    except:
        pass


def init_crash_handler():
    """初始化崩溃处理系统"""
    global _is_initialized
    
    if _is_initialized:
        return
    
    # 启用 faulthandler（捕获 C 级别的崩溃）
    crash_log_path = get_crash_log_path()
    crash_log_file = None
    
    try:
        # 将 faulthandler 输出重定向到崩溃日志文件
        crash_log_file = open(crash_log_path, 'a', encoding='utf-8')
        faulthandler.enable(file=crash_log_file)
    except Exception:
        # 如果无法写入文件，尝试使用 stderr（如果可用）
        try:
            if sys.stderr is not None:
                faulthandler.enable(file=sys.stderr)
        except Exception:
            pass  # 在无控制台模式下，stderr 为 None，跳过 faulthandler
    
    # 设置全局异常处理器
    sys.excepthook = global_exception_handler
    
    # 设置线程异常处理器 (Python 3.8+)
    if hasattr(threading, 'excepthook'):
        threading.excepthook = thread_exception_handler
    
    # 设置 Qt 消息处理器
    try:
        from PySide6.QtCore import qInstallMessageHandler
        qInstallMessageHandler(qt_message_handler)
    except Exception:
        pass
    
    # 注册退出处理
    atexit.register(on_exit)
    
    # 记录启动信息
    try:
        timestamp = datetime.datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        with open(crash_log_path, 'a', encoding='utf-8') as f:
            f.write(f"\n{'='*80}\n")
            f.write(f"[{timestamp}] 程序启动\n")
            f.write(f"{'='*80}\n")
            f.write(f"{get_system_info()}\n")
    except Exception:
        pass
    
    _is_initialized = True
    
    # 安全打印（在无控制台模式下不会报错）
    try:
        if sys.stdout is not None:
            print(f"崩溃日志系统已初始化，日志文件: {crash_log_path}")
    except Exception:
        pass


def log_operation(operation_name, details=""):
    """记录当前操作（用于崩溃时知道在做什么）"""
    try:
        from logger import logger
        logger.info(f"[操作] {operation_name}: {details}")
    except:
        pass


# 方便的装饰器，用于包装可能崩溃的函数
def crash_protected(func):
    """装饰器：保护函数不会因异常而崩溃程序"""
    def wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except Exception as e:
            error_tb = sys.exc_info()[2]
            context = f"在函数 '{func.__name__}' 中发生异常"
            write_crash_log(type(e), e, error_tb, context=context)
            raise  # 重新抛出异常
    return wrapper


# 用于在关键位置记录状态
class CrashContext:
    """上下文管理器：记录关键操作"""
    _current_context = []
    
    def __init__(self, description):
        self.description = description
    
    def __enter__(self):
        CrashContext._current_context.append(self.description)
        return self
    
    def __exit__(self, exc_type, exc_val, exc_tb):
        CrashContext._current_context.pop()
        if exc_type is not None:
            # 发生异常时记录上下文
            context = " -> ".join(CrashContext._current_context + [self.description])
            write_crash_log(exc_type, exc_val, exc_tb, context=f"操作链: {context}")
        return False  # 不抑制异常
    
    @classmethod
    def get_current_context(cls):
        return " -> ".join(cls._current_context) if cls._current_context else "无"


def set_main_window(window):
    """设置主窗口引用，用于显示弹窗"""
    global _main_window
    _main_window = window


def show_error_dialog(title, message, detail=None, suggestion=None):
    """
    显示用户友好的错误弹窗
    
    :param title: 弹窗标题
    :param message: 主要错误信息
    :param detail: 详细错误信息（可选）
    :param suggestion: 建议操作（可选）
    """
    try:
        from PySide6.QtWidgets import QMessageBox, QApplication
        from PySide6.QtCore import QMetaObject, Qt, Q_ARG
        
        def _show_dialog():
            try:
                msg_box = QMessageBox()
                msg_box.setIcon(QMessageBox.Critical)
                msg_box.setWindowTitle(title)
                
                # 构建显示文本
                full_message = message
                if suggestion:
                    full_message += f"\n\n💡 建议: {suggestion}"
                
                msg_box.setText(full_message)
                
                if detail:
                    msg_box.setDetailedText(detail)
                
                msg_box.setStandardButtons(QMessageBox.Ok)
                msg_box.exec()
            except Exception as e:
                # 如果弹窗失败，至少打印到控制台
                if sys.stderr:
                    print(f"无法显示错误弹窗: {e}", file=sys.stderr)
        
        # 检查是否在主线程
        app = QApplication.instance()
        if app:
            if threading.current_thread() is threading.main_thread():
                _show_dialog()
            else:
                # 在子线程中，需要通过信号槽机制在主线程中显示
                # 使用 QTimer.singleShot 在主线程中执行
                from PySide6.QtCore import QTimer
                QTimer.singleShot(0, _show_dialog)
        else:
            # 没有 QApplication，打印到控制台
            if sys.stderr:
                print(f"[{title}] {message}", file=sys.stderr)
                if detail:
                    print(f"详情: {detail}", file=sys.stderr)
    except ImportError:
        # PySide6 不可用
        if sys.stderr:
            print(f"[{title}] {message}", file=sys.stderr)
    except Exception as e:
        if sys.stderr:
            print(f"显示错误弹窗失败: {e}", file=sys.stderr)


def show_network_error_dialog(error, endpoint=None):
    """
    显示网络错误弹窗
    
    :param error: 异常对象
    :param endpoint: API 端点（可选）
    """
    error_type = type(error).__name__
    error_msg = str(error)
    
    # 根据错误类型给出友好提示
    if "ConnectionError" in error_type or "connection" in error_msg.lower():
        title = "网络连接失败"
        message = "无法连接到服务器，请检查您的网络连接。"
        suggestion = "请检查网络连接后重启软件重试"
    elif "Timeout" in error_type or "timeout" in error_msg.lower():
        title = "网络请求超时"
        message = "服务器响应超时，可能是网络不稳定或服务器繁忙。"
        suggestion = "请稍后重启软件重试"
    elif "ProxyError" in error_type or "proxy" in error_msg.lower():
        title = "代理连接失败"
        message = "通过代理连接服务器失败，请检查代理设置。"
        suggestion = "请检查代理设置或关闭代理后重启软件重试"
    elif "SSLError" in error_type or "ssl" in error_msg.lower():
        title = "SSL/TLS 错误"
        message = "安全连接失败，可能是证书问题或网络被拦截。"
        suggestion = "请检查系统时间是否正确，或重启软件重试"
    else:
        title = "网络请求失败"
        message = f"请求服务器时发生错误: {error_type}"
        suggestion = "请重启软件重试，如问题持续请联系客服"
    
    detail = f"错误类型: {error_type}\n错误信息: {error_msg}"
    if endpoint:
        detail += f"\n请求地址: {endpoint}"
    
    # 记录到日志
    try:
        from logger import logger
        logger.error(f"🌐 {title}: {error_msg}")
        if endpoint:
            logger.error(f"   请求地址: {endpoint}")
    except:
        pass
    
    show_error_dialog(title, message, detail, suggestion)


def show_crash_dialog(error, context=None):
    """
    显示崩溃错误弹窗
    
    :param error: 异常对象
    :param context: 上下文信息（可选）
    """
    error_type = type(error).__name__
    error_msg = str(error)
    
    title = "程序发生错误"
    message = f"程序运行时发生了一个错误:\n{error_type}: {error_msg}"
    
    detail = f"错误类型: {error_type}\n错误信息: {error_msg}"
    if context:
        detail += f"\n上下文: {context}"
    
    # 获取堆栈信息
    tb_str = traceback.format_exc()
    if tb_str and tb_str != "NoneType: None\n":
        detail += f"\n\n堆栈追踪:\n{tb_str}"
    
    suggestion = "请重启软件重试。如果问题持续出现，请将日志文件发送给客服。"
    
    show_error_dialog(title, message, detail, suggestion)


def safe_execute(func=None, *, show_dialog=True, default_return=None, context=None):
    """
    安全执行装饰器：捕获异常并显示友好弹窗
    
    用法:
    @safe_execute
    def my_function():
        ...
    
    @safe_execute(show_dialog=True, default_return=None, context="处理文件")
    def my_function():
        ...
    
    :param func: 被装饰的函数
    :param show_dialog: 是否显示错误弹窗
    :param default_return: 发生异常时的默认返回值
    :param context: 上下文描述
    """
    def decorator(fn):
        @wraps(fn)
        def wrapper(*args, **kwargs):
            try:
                return fn(*args, **kwargs)
            except Exception as e:
                # 记录到日志
                try:
                    from logger import logger
                    ctx = context or fn.__name__
                    logger.error(f"❌ [{ctx}] 发生异常: {type(e).__name__}: {str(e)}")
                    logger.error(traceback.format_exc())
                except:
                    pass
                
                # 写入崩溃日志
                error_tb = sys.exc_info()[2]
                ctx = context or f"函数 '{fn.__name__}'"
                write_crash_log(type(e), e, error_tb, context=ctx)
                
                # 显示弹窗
                if show_dialog:
                    # 判断是否是网络相关错误
                    error_type = type(e).__name__
                    if any(keyword in error_type for keyword in ['Connection', 'Timeout', 'Proxy', 'SSL', 'HTTP']):
                        show_network_error_dialog(e)
                    else:
                        show_crash_dialog(e, ctx)
                
                return default_return
        return wrapper
    
    # 支持 @safe_execute 和 @safe_execute(...) 两种用法
    if func is not None:
        return decorator(func)
    return decorator


def safe_execute_async(show_dialog=True, default_return=None, context=None):
    """
    异步安全执行装饰器：用于在线程中执行的函数
    与 safe_execute 类似，但会通过主线程显示弹窗
    """
    def decorator(fn):
        @wraps(fn)
        def wrapper(*args, **kwargs):
            try:
                return fn(*args, **kwargs)
            except Exception as e:
                # 记录到日志
                try:
                    from logger import logger
                    ctx = context or fn.__name__
                    logger.error(f"❌ [异步-{ctx}] 发生异常: {type(e).__name__}: {str(e)}")
                    logger.error(traceback.format_exc())
                except:
                    pass
                
                # 写入崩溃日志
                error_tb = sys.exc_info()[2]
                ctx = context or f"异步函数 '{fn.__name__}'"
                write_crash_log(type(e), e, error_tb, context=ctx)
                
                # 显示弹窗（会自动处理线程问题）
                if show_dialog:
                    error_type = type(e).__name__
                    if any(keyword in error_type for keyword in ['Connection', 'Timeout', 'Proxy', 'SSL', 'HTTP']):
                        show_network_error_dialog(e)
                    else:
                        show_crash_dialog(e, ctx)
                
                return default_return
        return wrapper
    return decorator

