# -*- coding: utf-8 -*-
"""
🤖 本地模型管理对话框 - 下载、加载和管理本地大语言模型
"""

import os
import sys
import threading
from PySide6.QtCore import Qt, Signal, QTimer, QThread
from PySide6.QtWidgets import (QDialog, QVBoxLayout, QHBoxLayout, QLabel, QPushButton,
                              QGroupBox, QRadioButton, QProgressBar, QFrame,
                              QScrollArea, QWidget, QMessageBox, QCheckBox,
                              QButtonGroup, QSizePolicy)
from PySide6.QtGui import QIcon, QFont
from logger import logger


class DownloadThread(QThread):
    """下载线程"""
    progress = Signal(int, int, float)  # downloaded, total, speed
    finished = Signal(bool, str)  # success, message
    
    def __init__(self, model_id: str):
        super().__init__()
        self.model_id = model_id
        self._downloader = None
    
    def run(self):
        from lib.local_model import ModelDownloader
        
        def on_progress(downloaded, total, speed):
            self.progress.emit(downloaded, total, speed)
        
        def on_complete(success, message):
            self.finished.emit(success, message)
        
        self._downloader = ModelDownloader(
            self.model_id,
            progress_callback=on_progress,
            complete_callback=on_complete
        )
        self._downloader._download_thread()  # 直接在这个线程中运行
    
    def stop(self):
        if self._downloader:
            self._downloader.stop()


class LoadModelThread(QThread):
    """加载模型线程"""
    progress = Signal(str)  # status message
    finished = Signal(bool, str)  # success, message
    
    def __init__(self, model_path: str, use_gpu: bool = False):
        super().__init__()
        self.model_path = model_path
        self.use_gpu = use_gpu
    
    def run(self):
        from lib.local_model import get_local_model
        
        def on_progress(msg):
            self.progress.emit(msg)
        
        try:
            model = get_local_model()
            n_gpu_layers = -1 if self.use_gpu else 0
            
            success = model.load(
                self.model_path,
                n_ctx=2048,
                n_gpu_layers=n_gpu_layers,
                progress_callback=on_progress
            )
            
            if success:
                self.finished.emit(True, "模型加载成功！")
            else:
                self.finished.emit(False, "模型加载失败")
        except Exception as e:
            self.finished.emit(False, f"加载失败: {str(e)}")


class ModelCard(QFrame):
    """模型卡片组件"""
    
    download_clicked = Signal(str)  # model_id
    load_clicked = Signal(str, str)  # model_id, model_path
    delete_clicked = Signal(str, str)  # model_id, model_path
    
    def __init__(self, model_info: dict, parent=None):
        super().__init__(parent)
        self.model_info = model_info
        self.model_id = model_info["id"]
        self.setup_ui()
    
    def setup_ui(self):
        self.setFrameStyle(QFrame.StyledPanel)
        self.setStyleSheet("""
            ModelCard {
                background-color: #2a2a2a;
                border: 1px solid #3a3a3a;
                border-radius: 8px;
                padding: 10px;
            }
            ModelCard:hover {
                border: 1px solid #19AF39;
            }
        """)
        
        layout = QVBoxLayout(self)
        layout.setContentsMargins(15, 15, 15, 15)
        layout.setSpacing(10)
        
        # 标题行
        title_layout = QHBoxLayout()
        
        # 模型名称
        name_label = QLabel(self.model_info["name"])
        name_label.setStyleSheet("""
            QLabel {
                font-size: 16px;
                font-weight: bold;
                color: #ffffff;
            }
        """)
        title_layout.addWidget(name_label)
        
        # 推荐标签
        if self.model_info.get("recommended"):
            rec_label = QLabel("⭐ 推荐")
            rec_label.setStyleSheet("""
                QLabel {
                    background-color: #19AF39;
                    color: white;
                    padding: 2px 8px;
                    border-radius: 4px;
                    font-size: 12px;
                }
            """)
            title_layout.addWidget(rec_label)
        
        title_layout.addStretch()
        
        # 状态标签
        if self.model_info.get("downloaded"):
            status_label = QLabel("✅ 已下载")
            status_label.setStyleSheet("color: #19AF39; font-size: 13px;")
        else:
            status_label = QLabel(f"📦 {self.model_info['size_mb']}MB")
            status_label.setStyleSheet("color: #888888; font-size: 13px;")
        title_layout.addWidget(status_label)
        
        layout.addLayout(title_layout)
        
        # 描述
        desc_label = QLabel(self.model_info["description"])
        desc_label.setWordWrap(True)
        desc_label.setStyleSheet("color: #aaaaaa; font-size: 12px;")
        layout.addWidget(desc_label)
        
        # 内存需求
        mem_label = QLabel(f"💾 内存需求: 约 {self.model_info['memory_required_mb']}MB")
        mem_label.setStyleSheet("color: #888888; font-size: 11px;")
        layout.addWidget(mem_label)
        
        # 按钮行
        btn_layout = QHBoxLayout()
        btn_layout.addStretch()
        
        if self.model_info.get("downloaded"):
            # 加载按钮
            self.load_btn = QPushButton("🚀 加载模型")
            self.load_btn.setStyleSheet("""
                QPushButton {
                    background-color: #19AF39;
                    color: white;
                    border: none;
                    padding: 8px 20px;
                    border-radius: 4px;
                    font-size: 13px;
                }
                QPushButton:hover {
                    background-color: #16973a;
                }
            """)
            self.load_btn.clicked.connect(self._on_load_clicked)
            btn_layout.addWidget(self.load_btn)
            
            # 删除按钮
            self.delete_btn = QPushButton("🗑️")
            self.delete_btn.setStyleSheet("""
                QPushButton {
                    background-color: #dc3545;
                    color: white;
                    border: none;
                    padding: 8px 12px;
                    border-radius: 4px;
                    font-size: 13px;
                }
                QPushButton:hover {
                    background-color: #c82333;
                }
            """)
            self.delete_btn.setToolTip("删除模型")
            self.delete_btn.clicked.connect(self._on_delete_clicked)
            btn_layout.addWidget(self.delete_btn)
        else:
            # 下载按钮
            self.download_btn = QPushButton("⬇️ 下载")
            self.download_btn.setStyleSheet("""
                QPushButton {
                    background-color: #007bff;
                    color: white;
                    border: none;
                    padding: 8px 20px;
                    border-radius: 4px;
                    font-size: 13px;
                }
                QPushButton:hover {
                    background-color: #0069d9;
                }
            """)
            self.download_btn.clicked.connect(self._on_download_clicked)
            btn_layout.addWidget(self.download_btn)
        
        layout.addLayout(btn_layout)
    
    def _on_download_clicked(self):
        self.download_clicked.emit(self.model_id)
    
    def _on_load_clicked(self):
        self.load_clicked.emit(self.model_id, self.model_info.get("path", ""))
    
    def _on_delete_clicked(self):
        self.delete_clicked.emit(self.model_id, self.model_info.get("path", ""))


class LocalModelDialog(QDialog):
    """本地模型管理对话框"""
    
    model_loaded = Signal(str)  # model_path
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("🤖 内置本地模型管理")
        self.setModal(True)
        self.setFixedSize(600, 650)
        
        self._download_thread: DownloadThread = None
        self._load_thread: LoadModelThread = None
        self._current_download_model_id = None
        
        self.setup_ui()
        self.refresh_models()
        self.check_llama_cpp()
        self.center_on_screen()
    
    def setup_ui(self):
        self.setStyleSheet("""
            QDialog {
                background-color: #1e1e1e;
                color: #ffffff;
            }
            QGroupBox {
                font-size: 14px;
                font-weight: bold;
                color: #ffffff;
                background-color: #1e1e1e;
                border: 2px solid #3a3a3a;
                border-radius: 8px;
                margin-top: 15px;
                padding-top: 10px;
            }
            QGroupBox::title {
                subcontrol-origin: margin;
                left: 10px;
                padding: 0 5px;
            }
            QLabel {
                color: #ffffff;
            }
            QCheckBox {
                color: #ffffff;
                font-size: 13px;
            }
            QCheckBox::indicator {
                width: 18px;
                height: 18px;
            }
            QCheckBox::indicator:checked {
                background-color: #19AF39;
                border: 2px solid #19AF39;
                border-radius: 4px;
            }
        """)
        
        layout = QVBoxLayout(self)
        layout.setContentsMargins(20, 20, 20, 20)
        layout.setSpacing(15)
        
        # 标题
        title_label = QLabel("🤖 内置本地模型")
        title_label.setStyleSheet("""
            QLabel {
                font-size: 20px;
                font-weight: bold;
                color: #19AF39;
            }
        """)
        layout.addWidget(title_label)
        
        # 说明
        info_label = QLabel("""
无需配置 Ollama，直接下载并加载本地大语言模型。
模型将下载到本地，支持离线使用。
        """)
        info_label.setWordWrap(True)
        info_label.setStyleSheet("color: #aaaaaa; font-size: 12px;")
        layout.addWidget(info_label)
        
        # 依赖检查提示
        self.dep_warning = QLabel("⚠️ 正在检查 llama-cpp-python 依赖...")
        self.dep_warning.setWordWrap(True)
        self.dep_warning.setStyleSheet("""
            QLabel {
                background-color: #3a3a00;
                color: #ffcc00;
                padding: 10px;
                border-radius: 4px;
                font-size: 12px;
            }
        """)
        layout.addWidget(self.dep_warning)
        
        # 当前状态
        self.status_group = QGroupBox("📊 当前状态")
        status_layout = QVBoxLayout(self.status_group)
        
        self.status_label = QLabel("未加载任何模型")
        self.status_label.setStyleSheet("color: #888888; font-size: 13px;")
        status_layout.addWidget(self.status_label)
        
        # GPU 选项
        self.gpu_checkbox = QCheckBox("🎮 使用 GPU 加速（需要 CUDA）")
        self.gpu_checkbox.setChecked(False)
        status_layout.addWidget(self.gpu_checkbox)
        
        # 卸载按钮
        self.unload_btn = QPushButton("📤 卸载当前模型")
        self.unload_btn.setEnabled(False)
        self.unload_btn.setStyleSheet("""
            QPushButton {
                background-color: #6c757d;
                color: white;
                border: none;
                padding: 8px 16px;
                border-radius: 4px;
            }
            QPushButton:hover {
                background-color: #5a6268;
            }
            QPushButton:disabled {
                background-color: #444444;
                color: #666666;
            }
        """)
        self.unload_btn.clicked.connect(self.unload_model)
        status_layout.addWidget(self.unload_btn)
        
        layout.addWidget(self.status_group)
        
        # 可用模型列表
        models_group = QGroupBox("📦 可用模型")
        models_layout = QVBoxLayout(models_group)
        
        # 滚动区域
        scroll = QScrollArea()
        scroll.setWidgetResizable(True)
        scroll.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        scroll.setStyleSheet("""
            QScrollArea {
                border: none;
                background-color: transparent;
            }
            QScrollBar:vertical {
                background-color: #2a2a2a;
                width: 10px;
                border-radius: 5px;
            }
            QScrollBar::handle:vertical {
                background-color: #19AF39;
                border-radius: 5px;
                min-height: 20px;
            }
        """)
        
        self.models_container = QWidget()
        self.models_layout = QVBoxLayout(self.models_container)
        self.models_layout.setContentsMargins(0, 0, 0, 0)
        self.models_layout.setSpacing(10)
        
        scroll.setWidget(self.models_container)
        models_layout.addWidget(scroll)
        
        layout.addWidget(models_group)
        
        # 下载进度
        self.progress_group = QGroupBox("⬇️ 下载进度")
        progress_layout = QVBoxLayout(self.progress_group)
        
        self.progress_label = QLabel("等待下载...")
        self.progress_label.setStyleSheet("color: #aaaaaa;")
        progress_layout.addWidget(self.progress_label)
        
        self.progress_bar = QProgressBar()
        self.progress_bar.setStyleSheet("""
            QProgressBar {
                border: 1px solid #3a3a3a;
                border-radius: 4px;
                background-color: #2a2a2a;
                text-align: center;
                color: white;
            }
            QProgressBar::chunk {
                background-color: #19AF39;
                border-radius: 3px;
            }
        """)
        self.progress_bar.setValue(0)
        progress_layout.addWidget(self.progress_bar)
        
        self.cancel_btn = QPushButton("❌ 取消下载")
        self.cancel_btn.setEnabled(False)
        self.cancel_btn.setStyleSheet("""
            QPushButton {
                background-color: #dc3545;
                color: white;
                border: none;
                padding: 8px 16px;
                border-radius: 4px;
            }
            QPushButton:hover {
                background-color: #c82333;
            }
            QPushButton:disabled {
                background-color: #444444;
                color: #666666;
            }
        """)
        self.cancel_btn.clicked.connect(self.cancel_download)
        progress_layout.addWidget(self.cancel_btn)
        
        self.progress_group.hide()  # 初始隐藏
        layout.addWidget(self.progress_group)
        
        # 底部按钮
        btn_layout = QHBoxLayout()
        
        self.refresh_btn = QPushButton("🔄 刷新")
        self.refresh_btn.clicked.connect(self.refresh_models)
        self.refresh_btn.setStyleSheet("""
            QPushButton {
                background-color: #17a2b8;
                color: white;
                border: none;
                padding: 8px 16px;
                border-radius: 4px;
            }
            QPushButton:hover {
                background-color: #138496;
            }
        """)
        btn_layout.addWidget(self.refresh_btn)
        
        btn_layout.addStretch()
        
        self.close_btn = QPushButton("关闭")
        self.close_btn.clicked.connect(self.close)
        self.close_btn.setStyleSheet("""
            QPushButton {
                background-color: #6c757d;
                color: white;
                border: none;
                padding: 8px 20px;
                border-radius: 4px;
            }
            QPushButton:hover {
                background-color: #5a6268;
            }
        """)
        btn_layout.addWidget(self.close_btn)
        
        layout.addLayout(btn_layout)
    
    def check_llama_cpp(self):
        """检查 llama-cpp-python 是否可用"""
        from lib.local_model import is_llama_cpp_available
        
        if is_llama_cpp_available():
            self.dep_warning.setText("✅ llama-cpp-python 已安装，可以使用本地模型")
            self.dep_warning.setStyleSheet("""
                QLabel {
                    background-color: #1a3a1a;
                    color: #19AF39;
                    padding: 10px;
                    border-radius: 4px;
                    font-size: 12px;
                }
            """)
        else:
            self.dep_warning.setText(
                "⚠️ llama-cpp-python 未安装\n\n"
                "请运行以下命令安装：\n"
                "pip install llama-cpp-python\n\n"
                "如果要使用 GPU 加速，请安装 CUDA 版本：\n"
                "pip install llama-cpp-python --extra-index-url https://abetlen.github.io/llama-cpp-python/whl/cu121"
            )
            self.dep_warning.setStyleSheet("""
                QLabel {
                    background-color: #3a2a2a;
                    color: #ff6b6b;
                    padding: 10px;
                    border-radius: 4px;
                    font-size: 12px;
                }
            """)
    
    def refresh_models(self):
        """刷新模型列表"""
        from lib.local_model import get_available_models, get_local_model
        
        # 清空现有内容
        while self.models_layout.count():
            item = self.models_layout.takeAt(0)
            if item.widget():
                item.widget().deleteLater()
        
        # 获取模型列表
        models = get_available_models()
        
        # 添加模型卡片
        for model_info in models:
            card = ModelCard(model_info)
            card.download_clicked.connect(self.start_download)
            card.load_clicked.connect(self.load_model)
            card.delete_clicked.connect(self.delete_model)
            self.models_layout.addWidget(card)
        
        self.models_layout.addStretch()
        
        # 更新当前状态
        model = get_local_model()
        if model.is_loaded():
            model_path = model.get_model_path()
            model_name = os.path.basename(model_path) if model_path else "未知"
            self.status_label.setText(f"✅ 已加载: {model_name}")
            self.status_label.setStyleSheet("color: #19AF39; font-size: 13px;")
            self.unload_btn.setEnabled(True)
        else:
            self.status_label.setText("未加载任何模型")
            self.status_label.setStyleSheet("color: #888888; font-size: 13px;")
            self.unload_btn.setEnabled(False)
    
    def start_download(self, model_id: str):
        """开始下载模型"""
        from lib.local_model import is_llama_cpp_available
        
        if not is_llama_cpp_available():
            QMessageBox.warning(self, "依赖缺失", 
                              "请先安装 llama-cpp-python 依赖\n\n"
                              "pip install llama-cpp-python")
            return
        
        if self._download_thread and self._download_thread.isRunning():
            QMessageBox.warning(self, "下载中", "已有下载任务正在进行")
            return
        
        # 确认下载
        from lib.local_model import AVAILABLE_MODELS
        model_info = AVAILABLE_MODELS.get(model_id, {})
        size_mb = model_info.get("size_mb", 0)
        
        reply = QMessageBox.question(
            self, "确认下载",
            f"确定要下载模型 {model_info.get('name', model_id)} 吗？\n\n"
            f"文件大小：约 {size_mb}MB\n"
            f"下载可能需要几分钟，请耐心等待。",
            QMessageBox.Yes | QMessageBox.No
        )
        
        if reply != QMessageBox.Yes:
            return
        
        # 显示进度
        self.progress_group.show()
        self.progress_bar.setValue(0)
        self.progress_label.setText(f"正在下载 {model_info.get('name', model_id)}...")
        self.cancel_btn.setEnabled(True)
        self._current_download_model_id = model_id
        
        # 开始下载
        self._download_thread = DownloadThread(model_id)
        self._download_thread.progress.connect(self._on_download_progress)
        self._download_thread.finished.connect(self._on_download_finished)
        self._download_thread.start()
    
    def _on_download_progress(self, downloaded: int, total: int, speed: float):
        """下载进度更新"""
        if total > 0:
            percent = int(downloaded * 100 / total)
            self.progress_bar.setValue(percent)
            
            downloaded_mb = downloaded / 1024 / 1024
            total_mb = total / 1024 / 1024
            self.progress_label.setText(
                f"已下载: {downloaded_mb:.1f}MB / {total_mb:.1f}MB ({percent}%) - 速度: {speed:.1f}MB/s"
            )
    
    def _on_download_finished(self, success: bool, message: str):
        """下载完成"""
        self.cancel_btn.setEnabled(False)
        self._current_download_model_id = None
        
        if success:
            self.progress_label.setText("✅ 下载完成！")
            self.progress_bar.setValue(100)
            QMessageBox.information(self, "下载完成", "模型下载成功！\n\n点击「加载模型」开始使用。")
            # 刷新列表
            QTimer.singleShot(500, self.refresh_models)
        else:
            self.progress_label.setText(f"❌ {message}")
            if "取消" not in message:
                QMessageBox.warning(self, "下载失败", message)
        
        # 延迟隐藏进度
        QTimer.singleShot(3000, lambda: self.progress_group.hide())
    
    def cancel_download(self):
        """取消下载"""
        if self._download_thread and self._download_thread.isRunning():
            self._download_thread.stop()
            self.progress_label.setText("正在取消下载...")
    
    def load_model(self, model_id: str, model_path: str):
        """加载模型"""
        from lib.local_model import is_llama_cpp_available
        
        if not is_llama_cpp_available():
            QMessageBox.warning(self, "依赖缺失", 
                              "请先安装 llama-cpp-python 依赖")
            return
        
        if not model_path or not os.path.exists(model_path):
            QMessageBox.warning(self, "错误", f"模型文件不存在: {model_path}")
            return
        
        if self._load_thread and self._load_thread.isRunning():
            QMessageBox.warning(self, "加载中", "模型正在加载中...")
            return
        
        # 显示加载中
        self.status_label.setText("⏳ 正在加载模型...")
        self.status_label.setStyleSheet("color: #ffcc00; font-size: 13px;")
        
        # 开始加载
        use_gpu = self.gpu_checkbox.isChecked()
        self._load_thread = LoadModelThread(model_path, use_gpu)
        self._load_thread.progress.connect(lambda msg: self.status_label.setText(f"⏳ {msg}"))
        self._load_thread.finished.connect(self._on_load_finished)
        self._load_thread.start()
    
    def _on_load_finished(self, success: bool, message: str):
        """加载完成"""
        if success:
            from lib.local_model import get_local_model
            model = get_local_model()
            model_path = model.get_model_path()
            model_name = os.path.basename(model_path) if model_path else "未知"
            
            self.status_label.setText(f"✅ 已加载: {model_name}")
            self.status_label.setStyleSheet("color: #19AF39; font-size: 13px;")
            self.unload_btn.setEnabled(True)
            
            QMessageBox.information(self, "加载成功", 
                                   f"模型加载成功！\n\n"
                                   f"现在可以在「模型选择」中选择「本地模型」进行文件整理。")
            
            # 发送信号
            self.model_loaded.emit(model_path)
        else:
            self.status_label.setText(f"❌ {message}")
            self.status_label.setStyleSheet("color: #ff6b6b; font-size: 13px;")
            self.unload_btn.setEnabled(False)
            
            QMessageBox.warning(self, "加载失败", message)
    
    def unload_model(self):
        """卸载模型"""
        from lib.local_model import get_local_model
        
        reply = QMessageBox.question(
            self, "确认卸载",
            "确定要卸载当前加载的模型吗？",
            QMessageBox.Yes | QMessageBox.No
        )
        
        if reply == QMessageBox.Yes:
            model = get_local_model()
            model.unload()
            
            self.status_label.setText("未加载任何模型")
            self.status_label.setStyleSheet("color: #888888; font-size: 13px;")
            self.unload_btn.setEnabled(False)
            
            QMessageBox.information(self, "卸载成功", "模型已卸载，内存已释放。")
    
    def delete_model(self, model_id: str, model_path: str):
        """删除模型"""
        if not model_path or not os.path.exists(model_path):
            return
        
        reply = QMessageBox.question(
            self, "确认删除",
            f"确定要删除模型文件吗？\n\n{os.path.basename(model_path)}\n\n此操作不可恢复！",
            QMessageBox.Yes | QMessageBox.No
        )
        
        if reply == QMessageBox.Yes:
            try:
                # 先卸载（如果是当前加载的模型）
                from lib.local_model import get_local_model
                model = get_local_model()
                if model.is_loaded() and model.get_model_path() == model_path:
                    model.unload()
                
                # 删除文件
                os.remove(model_path)
                QMessageBox.information(self, "删除成功", "模型文件已删除。")
                
                # 刷新列表
                self.refresh_models()
            except Exception as e:
                QMessageBox.warning(self, "删除失败", f"删除失败: {str(e)}")
    
    def center_on_screen(self):
        """居中显示"""
        from PySide6.QtWidgets import QApplication
        screen = QApplication.primaryScreen()
        screen_geometry = screen.geometry()
        x = (screen_geometry.width() - self.width()) // 2
        y = (screen_geometry.height() - self.height()) // 2
        self.move(x, y)
    
    def closeEvent(self, event):
        """关闭事件"""
        # 如果正在下载，询问是否取消
        if self._download_thread and self._download_thread.isRunning():
            reply = QMessageBox.question(
                self, "正在下载",
                "模型正在下载中，关闭窗口将取消下载。\n\n确定要关闭吗？",
                QMessageBox.Yes | QMessageBox.No
            )
            
            if reply == QMessageBox.Yes:
                self._download_thread.stop()
                self._download_thread.wait(5000)
                event.accept()
            else:
                event.ignore()
                return
        
        event.accept()



