"""
FileNeatAI 依赖检查工具
用于检查系统是否满足运行要求
"""

import sys
import os
import ctypes
import winreg
from pathlib import Path

def check_admin():
    """检查是否有管理员权限"""
    try:
        return ctypes.windll.shell32.IsUserAnAdmin()
    except:
        return False

def check_vcredist():
    """检查 Visual C++ 运行时库是否安装"""
    required_versions = [
        "Microsoft Visual C++ 2015-2022 Redistributable",
        "Microsoft Visual C++ 2015 Redistributable",
        "Microsoft Visual C++ 2017 Redistributable",
        "Microsoft Visual C++ 2019 Redistributable",
        "Microsoft Visual C++ 2022 Redistributable",
    ]
    
    installed = []
    try:
        # 检查64位程序
        key_path = r"SOFTWARE\Microsoft\Windows\CurrentVersion\Uninstall"
        with winreg.OpenKey(winreg.HKEY_LOCAL_MACHINE, key_path) as key:
            for i in range(winreg.QueryInfoKey(key)[0]):
                try:
                    subkey_name = winreg.EnumKey(key, i)
                    with winreg.OpenKey(key, subkey_name) as subkey:
                        display_name = winreg.QueryValueEx(subkey, "DisplayName")[0]
                        for required in required_versions:
                            if required in display_name:
                                installed.append(display_name)
                except:
                    continue
    except Exception as e:
        print(f"检查注册表时出错: {e}")
    
    return installed

def check_python_dll():
    """检查 python311.dll 是否存在"""
    dll_locations = [
        "python311.dll",  # 当前目录
        "_internal/python311.dll",  # PyInstaller 内部目录
        os.path.join(os.path.dirname(sys.executable), "python311.dll"),
        "C:\\Windows\\System32\\python311.dll",
        "C:\\Windows\\SysWOW64\\python311.dll",
    ]
    
    found = []
    for location in dll_locations:
        if os.path.exists(location):
            found.append(os.path.abspath(location))
    
    return found

def check_universal_crt():
    """检查 Universal CRT (UCRT) 是否安装"""
    # 检查关键的 api-ms-win-core DLL
    critical_dlls = [
        "api-ms-win-core-path-l1-1-0.dll",
        "api-ms-win-crt-runtime-l1-1-0.dll",
        "ucrtbase.dll",
    ]
    
    found = {}
    system_dirs = [
        "C:\\Windows\\System32",
        "C:\\Windows\\SysWOW64",
    ]
    
    for dll in critical_dlls:
        for sys_dir in system_dirs:
            dll_path = os.path.join(sys_dir, dll)
            if os.path.exists(dll_path):
                found[dll] = dll_path
                break
    
    return found, len(found) >= 2  # 至少找到 2 个关键 DLL 才算通过

def check_system_info():
    """获取系统信息"""
    import platform
    info = {
        "系统": platform.system(),
        "版本": platform.version(),
        "架构": platform.machine(),
        "Python版本": platform.python_version(),
    }
    
    # 检测 Windows 版本
    try:
        import sys
        if sys.platform == 'win32':
            win_ver = sys.getwindowsversion()
            if win_ver.major == 6 and win_ver.minor == 1:
                info["Windows版本"] = "Windows 7"
            elif win_ver.major == 6 and win_ver.minor == 2:
                info["Windows版本"] = "Windows 8"
            elif win_ver.major == 6 and win_ver.minor == 3:
                info["Windows版本"] = "Windows 8.1"
            elif win_ver.major == 10:
                info["Windows版本"] = "Windows 10/11"
            else:
                info["Windows版本"] = f"未知 ({win_ver.major}.{win_ver.minor})"
    except:
        pass
    
    return info

def main():
    print("=" * 60)
    print("FileNeatAI 系统依赖检查工具")
    print("=" * 60)
    print()
    
    # 检查系统信息
    print("【系统信息】")
    sys_info = check_system_info()
    for key, value in sys_info.items():
        print(f"  {key}: {value}")
    print()
    
    # 检查管理员权限
    print("【管理员权限】")
    if check_admin():
        print("  ✅ 当前以管理员身份运行")
    else:
        print("  ⚠️  未以管理员身份运行")
        print("  建议：右键程序，选择'以管理员身份运行'")
    print()
    
    # 检查 VC++ 运行时库
    print("【Visual C++ 运行时库】")
    vcredist = check_vcredist()
    if vcredist:
        print(f"  ✅ 已安装 {len(vcredist)} 个版本:")
        for vc in vcredist:
            print(f"     - {vc}")
    else:
        print("  ❌ 未检测到 Visual C++ 2015-2022 运行时库")
        print("  解决方案：")
        print("     1. 访问：https://aka.ms/vs/17/release/vc_redist.x64.exe")
        print("     2. 下载并安装")
        print("     3. 重启电脑")
    print()
    
    # 检查 Python DLL
    print("【Python DLL】")
    python_dlls = check_python_dll()
    if python_dlls:
        print(f"  ✅ 找到 python311.dll:")
        for dll in python_dlls:
            print(f"     - {dll}")
    else:
        print("  ❌ 未找到 python311.dll")
        print("  这可能是打包问题，请重新下载软件包")
    print()
    
    # 检查 Universal CRT
    print("【Universal CRT (通用 C 运行时)】")
    ucrt_dlls, ucrt_ok = check_universal_crt()
    if ucrt_ok:
        print(f"  ✅ Universal CRT 已安装 (找到 {len(ucrt_dlls)} 个关键 DLL)")
        for dll, path in ucrt_dlls.items():
            print(f"     - {dll}")
    else:
        print("  ❌ Universal CRT 未完整安装")
        if ucrt_dlls:
            print(f"  部分 DLL 已找到:")
            for dll, path in ucrt_dlls.items():
                print(f"     - {dll}")
        print()
        print("  解决方案：")
        print("     1. 安装 Visual C++ 2015-2022 运行时库")
        print("        https://aka.ms/vs/17/release/vc_redist.x64.exe")
        print("     2. 如果您使用 Windows 7/8，还需要安装 KB2999226 更新")
        print("        https://support.microsoft.com/zh-cn/topic/c0514201")
        print("     3. 重启电脑")
    print()
    
    # 总结
    print("=" * 60)
    print("【检查结果】")
    all_ok = vcredist and python_dlls and ucrt_ok
    
    if all_ok:
        print("  ✅ 所有依赖项检查通过！")
        print("  如果仍然无法运行，请：")
        print("     1. 重启电脑")
        print("     2. 临时关闭杀毒软件")
        print("     3. 以管理员身份运行")
    else:
        print("  ⚠️  发现缺少依赖项")
        print()
        if not vcredist:
            print("  ❌ 缺少: Visual C++ 运行时库")
        if not ucrt_ok:
            print("  ❌ 缺少: Universal CRT (通用 C 运行时)")
        if not python_dlls:
            print("  ❌ 缺少: python311.dll")
        print()
        print("  请按照上述提示安装缺少的组件")
        print()
        print("  ⭐ 重要提示：")
        if sys_info.get("Windows版本") in ["Windows 7", "Windows 8"]:
            print("  检测到您使用的是 Windows 7/8 系统")
            print("  请先安装 KB2999226 更新，然后再安装 VC++ 运行时库")
    print("=" * 60)
    print()
    
    input("按任意键退出...")

if __name__ == "__main__":
    try:
        main()
    except Exception as e:
        print(f"检查过程中出现错误: {e}")
        import traceback
        traceback.print_exc()
        input("按任意键退出...")

