import sys
import re
import time
# 平台特定模块的条件导入
if sys.platform.startswith('win'):
    import winreg
else:
    winreg = None
import random
import pytesseract
from PIL import Image
import os
import shutil
import sys
import subprocess
import configparser
from lib.api import json,cbc_decrypt,getKey,getInfo
from lib.constant import supported_extensions

from logger import logger

# wmi 模块导入
try:
    import wmi
except ImportError:
    wmi = None


class Common(object):
    machine_data = None

    @staticmethod
    def copy_files(src_dir, dst_dir):
        """
        将 src_dir 目录下的所有文件拷贝到 dst_dir 目录
        :param src_dir: 源目录路径
        :param dst_dir: 目标目录路径
        """
        # 确保目标目录存在，如果不存在则创建
        if not os.path.exists(dst_dir):
            os.makedirs(dst_dir)

        # 遍历源目录下的所有文件
        for filename in os.listdir(src_dir):
            # 获取文件的完整路径
            src_file = os.path.join(src_dir, filename)
            dst_file = os.path.join(dst_dir, filename)

            # 如果是文件，则拷贝
            if os.path.isfile(src_file):
                shutil.copy2(src_file, dst_file)



    @staticmethod
    def getctime(filepath):
        return time.strftime('%Y-%m-%d %H:%M:%S', time.localtime(os.path.getctime(filepath)))


    @staticmethod
    def getfilesize(filepath):
        return "{:.2f}MB".format(os.path.getsize(filepath) / (1024 * 1024))


    @staticmethod
    def getfilesuffix(filepath):
        suffix = os.path.splitext(filepath)[-1][1:]
        return suffix if suffix else ''
    

    @staticmethod
    def getRandomStr(length):
        return ''.join(random.sample(
            ['z', 'y', 'x', 'w', 'v', 'u', 't', 's', 'r', 'q', 'p', 'o', 'n', 'm', 'l', 'k', 'j', 'i', 'h', 'g', 'f','e','d', 'c', 'b', 'a'], length))

    @staticmethod
    def configset(config: dict):
        '''
        配置文件保存
        :param config:
        :return:
        '''
        userpath = Common.getConfigPath()

        cf = configparser.ConfigParser()
        cf.add_section("global")

        # 添加secion下详细参数以及value, value要用string存储
        cf.set("global", "localapiserver", config.get('localapiserver', ''))
        cf.set("global", "localapikey", config.get('localapikey', ''))
        cf.set("global", "localmodel", config.get('localmodel', ''))
        cf.set("global", "onlinemodel", config.get('onlinemodel', 'deepseek'))
        cf.set("global", "language", config.get('language', ''))
        cf.set("global", "tesseract", config.get('tesseract', ''))
        # 添加存储路径配置保存
        cf.set("global", "storage_path", config.get('storage_path', ''))
        # 添加提示词配置保存
        cf.set("global", "page1_custom_prompt", config.get('page1_custom_prompt', ''))
        cf.set("global", "page2_custom_prompt", config.get('page2_custom_prompt', ''))
        cf.set("global", "page3_custom_prompt", config.get('page3_custom_prompt', ''))

        # 写存储文件
        with open("{0}{1}config.ini".format(userpath, os.sep), "w+", encoding='utf-8') as f:
            cf.write(f)

    @staticmethod
    def getConfig():
        '''
        获取配置文件的内容
        :return:
        '''
        userpath = Common.getConfigPath()

        cf = configparser.ConfigParser()
        configpath = "{0}{1}config.ini".format(userpath, os.sep)
        if not os.path.exists(configpath):
            Common.configset({})
            # return None

        cf.read(configpath, encoding='utf-8')

        return cf

    @staticmethod
    def save_custom_prompts(page1_prompt='', page2_prompt='', page3_prompt=''):
        '''
        保存自定义提示词到配置文件
        :param page1_prompt: 页面1的提示词
        :param page2_prompt: 页面2的提示词  
        :param page3_prompt: 页面3的提示词
        :return:
        '''
        # 获取现有配置
        cf = Common.getConfig()
        
        # 获取现有配置值，如果不存在则使用空字符串
        config = {
            'localapiserver': cf.get('global', 'localapiserver') if cf and cf.has_option('global', 'localapiserver') else '',
            'localapikey': cf.get('global', 'localapikey') if cf and cf.has_option('global', 'localapikey') else '',
            'localmodel': cf.get('global', 'localmodel') if cf and cf.has_option('global', 'localmodel') else '',
            'onlinemodel': cf.get('global', 'onlinemodel') if cf and cf.has_option('global', 'onlinemodel') else 'deepseek',
            'language': cf.get('global', 'language') if cf and cf.has_option('global', 'language') else '',
            'tesseract': cf.get('global', 'tesseract') if cf and cf.has_option('global', 'tesseract') else '',
            'page1_custom_prompt': page1_prompt,
            'page2_custom_prompt': page2_prompt,
            'page3_custom_prompt': page3_prompt
        }
        
        # 保存配置
        Common.configset(config)

    @staticmethod
    def load_custom_prompts():
        '''
        从配置文件加载自定义提示词
        :return: dict 包含三个页面的提示词
        '''
        cf = Common.getConfig()
        prompts = {
            'page1_custom_prompt': cf.get('global', 'page1_custom_prompt') if cf and cf.has_option('global', 'page1_custom_prompt') else '',
            'page2_custom_prompt': cf.get('global', 'page2_custom_prompt') if cf and cf.has_option('global', 'page2_custom_prompt') else '',
            'page3_custom_prompt': cf.get('global', 'page3_custom_prompt') if cf and cf.has_option('global', 'page3_custom_prompt') else ''
        }
        return prompts

    @staticmethod
    def load_storage_path():
        '''
        从配置文件加载存储路径
        :return: str 存储路径
        '''
        cf = Common.getConfig()
        return cf.get('global', 'storage_path') if cf and cf.has_option('global', 'storage_path') else ''

    @staticmethod
    def save_storage_path(storage_path=''):
        '''
        保存存储路径到配置文件
        :param storage_path: 存储路径
        :return:
        '''
        # 获取现有配置
        cf = Common.getConfig()
        
        # 获取现有配置值，如果不存在则使用空字符串
        config = {
            'localapiserver': cf.get('global', 'localapiserver') if cf and cf.has_option('global', 'localapiserver') else '',
            'localapikey': cf.get('global', 'localapikey') if cf and cf.has_option('global', 'localapikey') else '',
            'localmodel': cf.get('global', 'localmodel') if cf and cf.has_option('global', 'localmodel') else '',
            'onlinemodel': cf.get('global', 'onlinemodel') if cf and cf.has_option('global', 'onlinemodel') else 'deepseek',
            'language': cf.get('global', 'language') if cf and cf.has_option('global', 'language') else '',
            'tesseract': cf.get('global', 'tesseract') if cf and cf.has_option('global', 'tesseract') else '',
            'storage_path': storage_path,
            'page1_custom_prompt': cf.get('global', 'page1_custom_prompt') if cf and cf.has_option('global', 'page1_custom_prompt') else '',
            'page2_custom_prompt': cf.get('global', 'page2_custom_prompt') if cf and cf.has_option('global', 'page2_custom_prompt') else '',
            'page3_custom_prompt': cf.get('global', 'page3_custom_prompt') if cf and cf.has_option('global', 'page3_custom_prompt') else ''
        }
        
        # 保存配置
        Common.configset(config)

    @staticmethod
    def getOptionData(filelist, machine_data, basenow=False, folder_classify_mode=False):
        '''
        获取配置选项
        :param filelist:
        :param machine_data:
        :return:
        '''

        cf = Common.getConfig()
        isLocal = False
        if cf.get('global', 'onlinemodel') in ['本地模型', 'Local model', 'local']:
            isLocal = True

        # 检查是否有本地解锁权限（完全离线模式）
        local_unlock = machine_data.get('host', {}).get('local_unlock', 0)
        
        # 如果没有本地解锁权限且选择了本地模型，返回错误
        if isLocal and local_unlock != 1:
            if cf.get('global', 'language') == 'chinese':
                return False, "您没有本地模型使用权限，请联系供应商获取离线License"
            else:
                return False, "You don't have permission to use local model, please contact vendor for offline license"

        key = 'ollama'
        # 只有在没有local_unlock权限且使用在线模型时才请求接口
        if not isLocal and local_unlock != 1:
            flag, ret = getKey(Common.get_machine_code())
            if not flag:
                return False, ret

            if len(filelist) < 1:
                if cf.get('global', 'language') == 'chinese':
                    return False, "文件数量不能低于1"
                else:
                    return False, "The number of files cannot be less than 1"

            data = json.loads(cbc_decrypt(ret['data']))
            
            # 🚀 根据在线模型类型选择相应的 API key
            online_model = cf.get('global', 'onlinemodel')
            
            # 根据不同的在线模型选择对应的 key
            # DeepSeek: 使用 deepseekkey（用于文本分析）
            # ChatGPT: 使用 key（旧格式）或 chatgptkey（新格式）
            if online_model == 'deepseek':
                # DeepSeek 模式：优先使用新格式的 deepseekkey
                key = data.get('deepseekkey', '')
                if not key:
                    # 兼容旧格式：使用 dp_key
                    key = data.get('dp_key', '')
                if not key:
                    # 再次兼容：使用 key
                    key = data.get('key', '')
            elif online_model == 'chatgpt':
                # ChatGPT 模式：优先使用新格式的 chatgptkey，否则使用旧的 key
                key = data.get('chatgptkey', data.get('key', ''))
            else:
                # 其他情况：兼容旧的 API 格式
                key = data.get('key', '')
            
            if not key:
                logger.warning(f"⚠️  警告：未能从 API 返回数据中获取有效的 key，在线模型: {online_model}")
                logger.warning(f"   API 返回的数据字段: {list(data.keys())}")


        options = {
            'userpath': os.path.dirname(filelist[0]),
            'filelist': filelist,
            'key': key,
            'mode': 'copy',
            'rename': False,
            'file_count': machine_data['setting']['trial_file_count'],
            'vip_file_count': machine_data['setting']['vip_file_count'],
            'is_vip': machine_data['host']['is_vip'],
            'local_unlock': machine_data['host'].get('local_unlock', 0),  # 🔓 添加本地解锁权限
            # 'basenow': True if self.checkBox_now.isChecked() else False,
            'basenow': basenow,
            'islocal': True if cf.get('global', 'onlinemodel') in ['本地模型', 'Local model', 'local'] else False,
            'localapiserver': cf.get('global', 'localapiserver'),
            'localapikey': cf.get('global', 'localapikey'),
            'localmodel': cf.get('global', 'localmodel'),
            'onlinemodel': cf.get('global', 'onlinemodel'),
            'language': cf.get('global', 'language'),
            'tesseract_path': r'',
            'pic_mode': 'ai', # 图片识别模式 ai
            'enable_aggregation': True,  # 启用智能聚合功能
            'max_categories': 8,  # 最大类别数量
            'folder_classify_mode': folder_classify_mode,  # 文件夹分类模式
            'thread_count': cf.get('global', 'thread_count') if cf.has_option('global', 'thread_count') else '4'  # 线程数配置
        }
        return True, options

    @staticmethod
    def getConfigPath():
        if sys.platform.startswith('win'):
            userpath = "{0}{1}fileneatai".format(os.getenv("APPDATA"), os.sep)

        elif sys.platform.startswith('darwin'):
            userpath = "{0}/Library/Application Support{1}fileneatai".format(os.environ['HOME'], os.sep)
        else:
            # Linux系统：使用当前目录下的配置文件夹
            userpath = "{0}{1}fileneatai".format(os.getcwd(), os.sep)

        if not os.path.exists(userpath):
            os.mkdir(userpath)

        return userpath


    @staticmethod
    def get_max_chars_for_user(is_vip: bool, is_local: bool) -> int:
        """根据用户类型和模型类型获取文件读取的最大字符数
        
        Args:
            is_vip: 是否为VIP用户
            is_local: 是否使用本地模型
            
        Returns:
            int: 最大字符数
            
        规则:
        - 非VIP用户: 5000字符
        - VIP用户 + 本地模型: 5000字符  
        - VIP用户 + 在线模型: 3000字符
        """
        if not is_vip:
            # 非VIP用户统一使用5000字符
            return 5000
        else:
            # VIP用户根据模型类型决定
            if is_local:
                # VIP + 本地模型: 5000字符
                return 5000
            else:
                # VIP + 在线模型: 3000字符
                return 3000

    @staticmethod
    def getLeftKeywords(data, keyword:list):
        for line in data:
            for val in data[line]:
                if val in keyword:
                    keyword.remove(val)

        return keyword


    @staticmethod
    def get_machine_code():
        if sys.platform.startswith('win'):
            try:
                if wmi is None:
                    logger.error("WMI模块未安装，请先安装：pip install WMI")
                    return None
                
                c = wmi.WMI()
                machine_id = None
                
                # 优先尝试获取 SMBIOS 提供的机器 UUID
                try:
                    uuid = c.Win32_ComputerSystemProduct()[0].UUID.strip()
                    logger.info("尝试获取UUID: {0}".format(uuid))
                except Exception as e:
                    logger.error("获取UUID失败: {0}".format(str(e)))
                    uuid = ""

                # 检查UUID是否有效
                if uuid and uuid != "00000000-0000-0000-0000-000000000000":
                    machine_id = uuid
                    logger.info("使用UUID作为机器码: {0}".format(machine_id))
                else:
                    # 如果UUID无效，则退而求其次尝试获取BIOS序列号
                    try:
                        bios_sn = c.Win32_BIOS()[0].SerialNumber.strip()
                        logger.info("尝试获取BIOS序列号: {0}".format(bios_sn))
                    except Exception as e:
                        logger.error("获取BIOS序列号失败: {0}".format(str(e)))
                        bios_sn = ""
                    
                    if bios_sn and "To be filled" not in bios_sn:
                        machine_id = bios_sn
                        logger.info("使用BIOS序列号作为机器码: {0}".format(machine_id))
                    else:
                        # 再不行可以考虑主板序列号作为最后手段
                        try:
                            baseboard_sn = c.Win32_BaseBoard()[0].SerialNumber.strip()
                            logger.info("尝试获取主板序列号: {0}".format(baseboard_sn))
                        except Exception as e:
                            logger.error("获取主板序列号失败: {0}".format(str(e)))
                            baseboard_sn = ""
                        
                        if baseboard_sn and "To be filled" not in baseboard_sn:
                            machine_id = baseboard_sn
                            logger.info("使用主板序列号作为机器码: {0}".format(machine_id))
                        else:
                            # 硬件标识都缺失的话，最后考虑操作系统的MachineGuid作为备选
                            try:
                                if winreg is not None:
                                    key = winreg.OpenKey(winreg.HKEY_LOCAL_MACHINE,
                                                        r"SOFTWARE\Microsoft\Cryptography")
                                    machine_id = winreg.QueryValueEx(key, "MachineGuid")[0]
                                    winreg.CloseKey(key)
                                    logger.info("使用MachineGuid作为机器码: {0}".format(machine_id))
                                else:
                                    logger.error("winreg模块不可用")
                                    return None
                            except Exception as e:
                                logger.error("获取MachineGuid失败: {0}".format(str(e)))
                                return None
                
                return machine_id
                
            except Exception as e:
                logger.error("获取机器码失败: {0}".format(str(e)))
                return None


        elif sys.platform.startswith('darwin'):
            command = "ioreg -rd1 -c IOPlatformExpertDevice"
            output = subprocess.check_output(command, shell=True).decode("utf-8")

            pattern = re.compile(r'"IOPlatformUUID" = "(.*?)"')
            match = pattern.search(output)

            if match:
                machine_code = match.group(1)
                return machine_code
            return None
        else:
            # Linux系统
            try:
                machine_id = None

                # 优先尝试读取 /etc/machine-id (systemd 提供的机器唯一标识符)
                try:
                    with open('/etc/machine-id', 'r') as f:
                        machine_id = f.read().strip()
                        if machine_id:
                            logger.info("使用/etc/machine-id作为机器码: {0}".format(machine_id))
                            return machine_id
                except Exception as e:
                    logger.error("读取/etc/machine-id失败: {0}".format(str(e)))

                # 备选：尝试读取 /var/lib/dbus/machine-id
                try:
                    with open('/var/lib/dbus/machine-id', 'r') as f:
                        machine_id = f.read().strip()
                        if machine_id:
                            logger.info("使用/var/lib/dbus/machine-id作为机器码: {0}".format(machine_id))
                            return machine_id
                except Exception as e:
                    logger.error("读取/var/lib/dbus/machine-id失败: {0}".format(str(e)))

                # 尝试DMI信息：产品UUID
                try:
                    with open('/sys/class/dmi/id/product_uuid', 'r') as f:
                        uuid = f.read().strip()
                        if uuid and uuid != "00000000-0000-0000-0000-000000000000":
                            machine_id = uuid
                            logger.info("使用DMI产品UUID作为机器码: {0}".format(machine_id))
                            return machine_id
                except Exception as e:
                    logger.error("读取DMI产品UUID失败: {0}".format(str(e)))

                # 尝试主板序列号
                try:
                    with open('/sys/class/dmi/id/board_serial', 'r') as f:
                        board_serial = f.read().strip()
                        if board_serial and board_serial not in ["None", "To be filled by O.E.M.", "0"]:
                            machine_id = board_serial
                            logger.info("使用主板序列号作为机器码: {0}".format(machine_id))
                            return machine_id
                except Exception as e:
                    logger.error("读取主板序列号失败: {0}".format(str(e)))

                # 尝试产品序列号
                try:
                    with open('/sys/class/dmi/id/product_serial', 'r') as f:
                        product_serial = f.read().strip()
                        if product_serial and product_serial not in ["None", "To be filled by O.E.M.", "0"]:
                            machine_id = product_serial
                            logger.info("使用产品序列号作为机器码: {0}".format(machine_id))
                            return machine_id
                except Exception as e:
                    logger.error("读取产品序列号失败: {0}".format(str(e)))

                # 最后备选：尝试CPU信息生成唯一标识
                try:
                    with open('/proc/cpuinfo', 'r') as f:
                        cpuinfo = f.read()
                        # 查找处理器序列号或其他唯一信息
                        for line in cpuinfo.split('\n'):
                            if 'Serial' in line:
                                serial = line.split(':')[1].strip()
                                if serial and serial != "0000000000000000":
                                    machine_id = serial
                                    logger.info("使用CPU序列号作为机器码: {0}".format(machine_id))
                                    return machine_id
                except Exception as e:
                    logger.error("读取CPU信息失败: {0}".format(str(e)))

                # 如果所有方法都失败，生成基于MAC地址的唯一标识
                try:
                    import uuid
                    mac = uuid.getnode()
                    machine_id = str(mac)
                    logger.info("使用MAC地址作为机器码: {0}".format(machine_id))
                    return machine_id
                except Exception as e:
                    logger.error("获取MAC地址失败: {0}".format(str(e)))

                logger.error("所有获取机器码的方法都失败了")
                return None

            except Exception as e:
                logger.error("Linux平台获取机器码失败: {0}".format(str(e)))
                return None


    @staticmethod
    def getPicText(tess_path, path):
        pytesseract.pytesseract.tesseract_cmd = tess_path
        image = Image.open(path)
        text = pytesseract.image_to_string(image, lang='chi_sim+eng')
        return text


    @staticmethod
    def checkcommand(cmd, flag):
        if sys.platform.startswith('win'):
            x = subprocess.run(cmd, stdout=subprocess.PIPE, shell=True)
            res = x.stdout.decode('utf-8')
            data = re.search(flag, res)
            if not data:
                return False
        elif sys.platform.startswith('darwin'):
            res = subprocess.check_output(cmd, shell=True).decode("utf-8")
            data = re.search(flag, res)
            if not data:
                return False

        return True

    @staticmethod
    def getValidFileList(filelist):
        lists = []
        for file in filelist:
            flag, suffix = Common.assetIdentify(file)
            if not flag:
                continue
            lists.append(file)
        return lists


    @staticmethod
    def assetIdentify(filename:str):
        info = os.path.splitext(filename.lower())
        if not info[1]:
            # print("没有后缀名 {0}".format(filename))
            return False, "没有后缀名 {0}".format(filename)
        else:
            suffix = info[1][1:]

        if suffix in supported_extensions:
            return 'pic', suffix

        return 'text', suffix


    @staticmethod
    def checkollama():
        return Common.checkcommand('ollama -v', 'ollama version')

    @staticmethod
    def checkllava():
        return Common.checkcommand('ollama list', 'llava')

    @staticmethod
    def checktessact(path):
        cmd = r'"{0}" -v'.format(path)
        print(cmd)
        return Common.checkcommand(cmd, 'libjpeg')

    @staticmethod
    def installllava():
        if sys.platform.startswith('win'):
            os.system('start cmd /k "ollama run llava:latest"')
        elif sys.platform.startswith('darwin'):
            os.system('ollama run llava:latest')
        else:
            pass

    # 获取一个文件夹下所有的文件名
    @staticmethod
    def sanitize_filename(filename):
        """
        清理文件名或文件夹名，移除Windows不允许的字符
        :param filename: 原始文件名或文件夹名
        :return: 清理后的文件名
        """
        if not filename:
            return "未命名"
        
        # Windows文件系统不允许的字符
        invalid_chars = ['/', '\\', ':', '*', '?', '"', '<', '>', '|', 
                        '，', '。', '！', '？', '；', '：', ''', ''', '"', '"', 
                        '【', '】', '、', '[', ']', '{', '}', '(', ')', 
                        '&', '%', '$', '#', '@', '!', '^', '~', '`']
        
        # 替换非法字符为下划线
        cleaned_name = filename
        for char in invalid_chars:
            cleaned_name = cleaned_name.replace(char, '_')
        
        # 将连续的下划线替换为单个下划线
        cleaned_name = re.sub(r'_+', '_', cleaned_name)
        
        # 移除开头和结尾的下划线
        cleaned_name = cleaned_name.strip('_')
        
        # 移除开头和结尾的空格和点号
        cleaned_name = cleaned_name.strip(' .')
        
        # 限制长度（Windows文件名最大255字符，但为了兼容性限制为100）
        if len(cleaned_name) > 100:
            cleaned_name = cleaned_name[:100]
        
        # 最终检查：确保文件名不为空
        if not cleaned_name:
            cleaned_name = "未命名"
        
        return cleaned_name

    @staticmethod
    def is_temp_file(file_path):
        """
        检测是否为临时文件
        返回True表示是临时文件，需要过滤掉
        """
        filename = os.path.basename(file_path)
        filename_lower = filename.lower()
        
        # Office/WPS 临时文件模式（以波浪号开头）
        if filename.startswith('~'):
            return True
        
        # Office 临时文件（~$ 开头）
        if filename.startswith('~$'):
            return True
        
        # 常见临时文件扩展名
        temp_extensions = ['.tmp', '.temp', '.bak', '.backup', '.swp', '.swo']
        for ext in temp_extensions:
            if filename_lower.endswith(ext):
                return True
        
        # 以点波浪号开头的临时文件
        if filename.startswith('.~'):
            return True
        
        # 系统隐藏文件（以点开头，但排除当前目录和上级目录）
        if filename.startswith('.') and filename not in ['.', '..']:
            # 但保留一些重要的配置文件
            important_dotfiles = ['.gitignore', '.env', '.config', '.htaccess']
            if not any(filename.lower().startswith(important) for important in important_dotfiles):
                return True
        
        # Windows 系统文件
        if filename_lower in ['thumbs.db', 'desktop.ini', '.ds_store']:
            return True
        
        # Office 锁定文件
        if filename.startswith('~$') or filename.startswith('.~lock'):
            return True
        
        # LibreOffice/OpenOffice 临时文件
        if filename.startswith('.~lock.'):
            return True
        
        # AutoCAD 临时文件
        if filename_lower.endswith('.dwl') or filename_lower.endswith('.dwl2'):
            return True
        
        # Adobe 临时文件
        if filename.startswith('~tmp') or filename_lower.endswith('.ffs_tmp'):
            return True
        
        return False


if __name__ == '__main__':
    Common.checkllava()