# -*- coding: utf-8 -*-
"""
OCR优化器 - 按照用户要求的优先级顺序进行图片识别

🔧 2026-01-22 修复：
- 添加全面的异常捕获，防止 access violation 导致程序崩溃
- 添加超时保护机制
- 优化多线程调用的稳定性
"""

import os
import sys
import threading
from pathlib import Path
from typing import Optional, Tuple
from concurrent.futures import ThreadPoolExecutor, TimeoutError as FuturesTimeoutError
from logger import logger

# 🔧 2026-01-23 优化：缩短超时时间，防止卡住
# Moonshot API 调用超时时间（秒）
MOONSHOT_API_TIMEOUT = 30

# 🔧 本地模型识别超时时间（秒）
LOCAL_MODEL_TIMEOUT = 25

# 🔧 信号量等待超时时间（秒）
SEMAPHORE_TIMEOUT = 15

# 🔧 2026-01-23 修改：使用全局互斥锁替代信号量，确保同一时间只有一个 OCR 任务
_OCR_GLOBAL_LOCK = threading.Lock()

def _debug_log_ocr(msg: str):
    """OCR 调试日志"""
    thread_id = threading.current_thread().ident
    logger.info(f"[OCR-DEBUG][Thread-{thread_id}] {msg}")

try:
    import pytesseract
    from PIL import Image
    HAS_TESSERACT = True
except ImportError:
    HAS_TESSERACT = False

try:
    from lib.ai import recognize_image_with_moonshot
except ImportError:
    recognize_image_with_moonshot = lambda *args, **kwargs: ""


class OCROptimizer:
    """OCR优化器 - 按照用户要求的优先级顺序进行图片识别
    
    在线模式（ChatGPT/DeepSeek等文本模型）的优先级：
    VIP用户:
        1. Moonshot API (文件提取)
        2. Tesseract OCR
    
    非VIP用户:
        1. Moonshot API (文件提取) - 首选方案
        2. 豆包API - 如果Moonshot API未识别出内容
        3. Tesseract OCR
    
    本地模式的优先级：
    1. 多模态AI - 如果配置了
    2. 用户自定义API服务
    3. Tesseract OCR
    
    注意：ChatGPT/DeepSeek等文本模型不用于图片识别
    """
    
    def __init__(self):
        self.methods_order = [
            "multimodal_ai",  # 1. 多模态AI (优先)
            "moonshot_api",   # 2. Moonshot API (备用)
            "tesseract",      # 3. 用户配置的Tesseract
        ]
    
    def recognize_image_with_priority(self, image_path: str, max_chars: int = 5000, 
                                    api_key: str = '', is_local: bool = False,
                                    image_recognition_method: str = "tesseract",
                                    online_model: str = None, api_base: str = None, model_name: str = None,
                                    is_vip: bool = False, doubao_info: dict = None) -> Tuple[str, str]:
        """
        按照优先级顺序识别图片
        
        Args:
            image_path: 图片路径
            max_chars: 最大字符数
            api_key: API密钥 (VIP用户使用Moonshot API)
            is_local: 是否本地模型
            image_recognition_method: 用户配置的识别方法
            online_model: 在线模型类型 (chatgpt/deepseek/None)
            is_vip: 是否VIP用户
            doubao_info: 豆包配置信息（非VIP用户使用）
            
        Returns:
            Tuple[str, str]: (识别结果文本, 识别方法)
        """
        
        logger.info(f"🔍 开始按优先级顺序识别图片: {os.path.basename(image_path)}")
        logger.info(f"🎯 模型类型: {online_model}, 本地模型: {is_local}, VIP状态: {is_vip}")
        
        # 🔥 最高优先级：本地视觉模型（如果已加载，无需网络）
        text, method = self._try_local_vision_model(image_path, max_chars)
        if text:
            return text, method
        
        # 🚀 根据模型类型和VIP状态使用不同的识别优先级
        if is_local:
            logger.info("🏠 本地模式识别优先级: 多模态AI -> 用户自定义API -> Tesseract")
        else:
            # 在线模式：VIP和非VIP用户使用不同的策略
            if is_vip:
                logger.info("🌙 在线模式(VIP用户)识别优先级: Moonshot API -> Tesseract OCR")
            else:
                logger.info("🌙🔥 在线模式(非VIP用户)识别优先级: Moonshot API -> 豆包API -> Tesseract OCR")
                logger.info("💡 非VIP用户首先尝试Moonshot API，如果没识别出内容再使用豆包API")
            
            if online_model in ['chatgpt', 'deepseek']:
                logger.info(f"💡 注意：{online_model}为文本模型，不用于图片识别")
        
        # 在线模式的识别逻辑
        if not is_local:
            # 🚀 VIP用户：直接使用Moonshot API，不尝试多模态AI
            if is_vip:
                if api_key:
                    logger.info("🌙 VIP用户：使用专属Moonshot API进行图片识别")
                    text, method = self._try_moonshot_api(image_path, api_key, max_chars)
                    if text:
                        return text, method
                    else:
                        logger.info("⚠️  Moonshot API识别失败，将尝试Tesseract OCR")
                else:
                    logger.warning("⚠️  VIP用户：未提供Moonshot API密钥")
            
            # 🚀 非VIP用户：首先尝试Moonshot API，如果没识别出内容再用豆包API
            else:
                # 1. 先尝试Moonshot API
                if api_key:
                    logger.info("🌙 非VIP用户：首先尝试使用Moonshot API进行图片识别")
                    text, method = self._try_moonshot_api(image_path, api_key, max_chars)
                    if text:
                        logger.info("✅ Moonshot API识别成功，使用该结果")
                        return text, method
                    else:
                        logger.info("⚠️  Moonshot API未识别出内容，将尝试豆包API")
                else:
                    logger.info("⚠️  非VIP用户：未提供Moonshot API密钥，直接尝试豆包API")
                
                # 2. 如果Moonshot API没识别出东西，再尝试豆包API
                if doubao_info and doubao_info.get('key'):
                    logger.info("🔥 非VIP用户：使用豆包API进行图片识别（备用方案）")
                    text, method = self._try_doubao_api(image_path, doubao_info, max_chars)
                    if text:
                        return text, method
                    else:
                        logger.info("⚠️  豆包API识别失败，将尝试Tesseract OCR")
                else:
                    logger.warning("⚠️  非VIP用户：未配置豆包API")
        
        # 本地模式：可以尝试多模态AI（如果用户手动配置了）
        if is_local:
            logger.info("🏠 本地模式：尝试使用多模态AI服务进行图片识别")
            text, method = self._try_multimodal_ai(image_path)
            if text:
                return text, method
            
            # 如果多模态AI失败，尝试用户自定义API服务（作为备用）
            logger.info("🏠 本地模式：多模态AI失败，尝试使用用户自定义API服务")
            text, method = self._try_local_model_recognition(image_path, api_key, api_base, model_name, max_chars)
            if text:
                return text, method
        
        # 最后尝试用户配置的Tesseract OCR
        text, method = self._try_tesseract(image_path, image_recognition_method)
        if text:
            return text, method
        
        # 所有方法都失败，返回空结果（load_document会提取基本图片信息）
        logger.warning(f"❌ 图片 {os.path.basename(image_path)} 所有识别方法均失败")
        logger.info("🔄 将跳过此图片或使用基本图片信息")
        return "", "识别失败"
    
    def _try_local_vision_model(self, image_path: str, max_chars: int = 5000) -> Tuple[str, str]:
        """尝试使用本地视觉模型识别图片（最高优先级）"""
        try:
            from lib.local_model import get_vision_model
            vision_model = get_vision_model()
            
            if not vision_model.is_loaded():
                return "", ""
            
            logger.info("📷 本地视觉模型已加载，使用本地视觉模型进行识别...")
            _debug_log_ocr("开始本地视觉模型识别")
            
            text = vision_model.recognize_image(
                image_path,
                prompt="请详细识别这张图片中的所有文字内容，包括中文和英文。只返回文字内容，不要解释。",
                max_tokens=1500
            )
            
            if text and text.strip():
                result = text.strip()[:max_chars]
                logger.info(f"✅ 本地视觉模型识别成功，结果长度: {len(result)}")
                return result, "本地视觉模型"
            else:
                logger.info("⚠️ 本地视觉模型未能识别内容")
                return "", ""
                
        except ImportError:
            return "", ""
        except Exception as e:
            logger.warning(f"⚠️ 本地视觉模型识别异常: {str(e)}")
            _debug_log_ocr(f"本地视觉模型异常: {str(e)}")
            return "", ""
    
    def _try_tesseract(self, image_path: str, image_recognition_method: str) -> Tuple[str, str]:
        """尝试使用Tesseract OCR"""
        if not HAS_TESSERACT:
            logger.info("⚠️  pytesseract未安装，跳过Tesseract方法")
            return "", ""
        
        if image_recognition_method not in ["tesseract", "both"]:
            logger.info("🔧 用户未配置Tesseract方法，跳过")
            return "", ""
        
        processed_image_path = None
        try:
            logger.info("🎯 优先使用Tesseract OCR（用户配置）")
            
            # 🚀 优化：使用增强版图片预处理提高识别率
            try:
                from lib.ai import preprocess_image_for_ocr
                processed_image_path = preprocess_image_for_ocr(image_path)
                logger.info(f"图片预处理完成: {image_path} -> {processed_image_path}")
            except Exception as preprocess_e:
                logger.warning(f"图片预处理失败，使用原始图片: {str(preprocess_e)}")
                processed_image_path = image_path
            
            # 使用预处理后的图片进行OCR
            img = Image.open(processed_image_path)
            tesseract_text = pytesseract.image_to_string(img, lang='chi_sim+eng')
            img.close()
            
            if tesseract_text and len(tesseract_text.strip()) > 5:
                logger.info(f"✅ Tesseract识别成功，内容长度: {len(tesseract_text)} 字符")
                logger.info(f"【Tesseract识别内容】 {os.path.basename(image_path)}:\n{tesseract_text}")
                return tesseract_text, "Tesseract OCR (用户配置)"
            else:
                logger.info("⚠️  Tesseract识别结果为空或过短")
                return "", ""
                
        except Exception as e:
            logger.warning(f"❌ Tesseract识别失败: {str(e)}")
            return "", ""
        finally:
            # 🚀 优化：清理临时预处理文件
            if processed_image_path and processed_image_path != image_path:
                try:
                    if os.path.exists(processed_image_path):
                        os.remove(processed_image_path)
                        logger.debug(f"成功删除临时预处理文件: {processed_image_path}")
                except Exception as cleanup_e:
                    logger.warning(f"清理临时预处理文件失败: {str(cleanup_e)}")
    
    def _try_multimodal_ai(self, image_path: str) -> Tuple[str, str]:
        """
        尝试使用多模态AI
        
        🔧 修复：添加超时保护和异常处理
        """
        logger.info("🔍 开始尝试使用多模态AI服务")
        
        # 🔧 2026-02-08 修复：去掉 _OCR_GLOBAL_LOCK 和 ThreadPoolExecutor
        # 因为已使用 max_workers=1 保证单线程执行，不需要额外的锁
        # 额外的 ThreadPoolExecutor 会创建新线程，导致并发网络请求崩溃
        try:
            from lib.multimodal_ai import multimodal_service
            
            # 直接检查可用性（不再用 ThreadPoolExecutor 包装）
            _debug_log_ocr("检查多模态AI服务可用性...")
            is_available = False
            try:
                is_available = multimodal_service.is_available()
            except Exception as check_e:
                logger.warning(f"⚠️  多模态AI可用性检查异常: {str(check_e)}")
                return "", ""
            
            logger.info(f"📊 多模态AI服务可用性检查结果: {is_available}")
            
            if not is_available:
                logger.warning("❌ 多模态AI服务未配置或不可用，尝试重新加载配置")
                
                # 尝试重新加载配置
                reload_result = multimodal_service.reload_config()
                logger.info(f"🔄 配置重新加载结果: {reload_result}")
                
                # 重新检查是否可用
                is_available_after_reload = multimodal_service.is_available()
                logger.info(f"📊 重新加载后多模态AI服务可用性: {is_available_after_reload}")
                
                if not is_available_after_reload:
                    logger.warning("❌ 重新加载后多模态AI服务仍不可用")
                    return "", ""
            
            logger.info("🤖 尝试使用多模态AI进行图片识别")
            
            # 🔧 直接调用（不再用 ThreadPoolExecutor 包装）
            success, content, cost, provider = False, "", 0.0, ""
            try:
                result = multimodal_service.recognize_image_smart(
                    image_path=image_path,
                    prompt="请详细识别这张图片中的所有文字内容，包括中文和英文。"
                )
                success, content, cost, provider = result
            except Exception as recognize_e:
                logger.warning(f"⚠️  多模态AI识别异常: {str(recognize_e)}")
                return "", ""
            
            if success and content and len(content.strip()) > 10:
                logger.info(f"✅ 多模态AI识别成功，服务商: {provider}，成本: ${cost:.4f}")
                logger.info(f"【多模态AI识别内容】 {os.path.basename(image_path)}:\n{content}")
                return content, f"多模态AI ({provider})"
            else:
                logger.error(f"❌ 多模态AI识别失败: {content}")
                return "", ""
                
        except ImportError as e:
            logger.error(f"❌ 多模态AI服务导入失败: {str(e)}")
            return "", ""
        except MemoryError as me:
            logger.error(f"❌ 内存不足: {str(me)}")
            return "", ""
        except Exception as e:
            logger.error(f"❌ 多模态AI识别过程中出现异常: {str(e)}")
            import traceback
            logger.error(f"  - 详细堆栈: {traceback.format_exc()}")
            return "", ""
    
    def _try_local_model_recognition(self, image_path: str, api_key: str, api_base: str, model_name: str, max_chars: int) -> Tuple[str, str]:
        """
        尝试使用本地模式的用户自定义API服务进行图片识别
        
        🔧 2026-02-08 修复：去掉 _OCR_GLOBAL_LOCK 和 ThreadPoolExecutor
        直接调用，因为 max_workers=1 已保证单线程，内部 API 调用已有全局网络锁
        """
        if not api_key or not api_base or not model_name:
            logger.info("⚠️  本地模式缺少API配置信息，跳过自定义API服务识别")
            return "", ""
        
        # 🤖 内置本地模型（builtin://）是纯文本模型，不支持直接图片识别
        if api_base and api_base.startswith('builtin://'):
            logger.info("⚠️  内置本地模型为纯文本模型，跳过在线API图片识别")
            return "", ""
        
        try:
            from lib.ai import recognize_image_with_online_model
            _debug_log_ocr(f"本地模式：使用用户自定义API服务识别图片")
            _debug_log_ocr(f"API服务: {api_base}, 模型: {model_name}")
            
            # 🔧 直接调用（不再用 ThreadPoolExecutor 包装）
            result_text = ""
            try:
                result_text = recognize_image_with_online_model(
                    image_path=image_path,
                    api_key=api_key,
                    api_base=api_base,
                    model_name=model_name,
                    max_chars=max_chars
                )
            except Exception as e:
                logger.warning(f"⚠️  本地自定义API服务识别异常: {str(e)}")
                return "", ""
            
            if result_text and len(result_text.strip()) > 10:
                logger.info(f"✅ 本地自定义API服务识别成功，内容长度: {len(result_text)} 字符")
                logger.info(f"📄 识别结果：\n{result_text}")
                return result_text, f"本地自定义API服务 ({model_name})"
            else:
                logger.info(f"⚠️  本地自定义API服务识别结果为空或过短")
                return "", ""
                
        except ImportError as ie:
            logger.warning(f"❌ 导入 recognize_image_with_online_model 失败: {str(ie)}")
            return "", ""
        except MemoryError as me:
            logger.error(f"❌ 内存不足: {str(me)}")
            return "", ""
        except Exception as e:
            logger.warning(f"❌ 本地自定义API服务识别失败: {str(e)}")
            import traceback
            logger.debug(f"详细错误: {traceback.format_exc()}")
            return "", ""
    
    # ❌ 已移除 _try_online_model 方法
    # ChatGPT/DeepSeek等文本模型不适合图片识别，已改为使用多模态AI或Moonshot API
    
    def _try_moonshot_api(self, image_path: str, api_key: str, max_chars: int) -> Tuple[str, str]:
        """尝试使用Moonshot API - VIP用户优先,非VIP用户作为首选方案"""
        
        # 🚀 优先使用传入的API密钥，如果没有则从配置文件中获取Moonshot专用密钥
        moonshot_api_key = api_key  # 优先使用传入的密钥
        
        # 如果传入的密钥为空，尝试从多模态设置中获取专门的Moonshot密钥
        if not moonshot_api_key:
            moonshot_api_key = self._get_moonshot_api_key()
            
        if not moonshot_api_key:
            logger.info("⚠️  未提供API密钥且未配置Moonshot专用API密钥，跳过Moonshot API")
            logger.info("💡 提示：请传入API密钥或在多模态设置中配置Moonshot API密钥")
            return "", ""
        
        try:
            logger.info("🌙 使用 Moonshot API 提取图片文字")
            logger.info(f"🔑 使用的Moonshot API密钥: {moonshot_api_key[:20]}..." if moonshot_api_key and len(moonshot_api_key) > 20 else f"🔑 使用的Moonshot API密钥: {moonshot_api_key}")
            
            # 🔧 2026-02-08 修复：直接调用，不再通过 ThreadPoolExecutor 创建额外线程
            # recognize_image_with_moonshot 内部已通过 safe_moonshot_recognize 使用全局锁保护
            # 额外的线程包装会导致并发网络请求，引发 access violation 崩溃
            moonshot_text = ""
            try:
                moonshot_text = recognize_image_with_moonshot(image_path, moonshot_api_key, max_chars)
            except Exception as e:
                logger.warning(f"⚠️ Moonshot API 请求异常: {str(e)}")
                return "", ""
            
            if moonshot_text and len(moonshot_text.strip()) > 10:
                logger.info(f"✅ Moonshot API 文字提取成功，内容长度: {len(moonshot_text)} 字符")
                logger.info(f"📄 Moonshot API 提取的文字内容：\n{moonshot_text}")
                return moonshot_text, "Moonshot API"
            else:
                logger.info("⚠️  Moonshot API 未提取到有效文字内容")
                return "", ""
                
        except FuturesTimeoutError:
            logger.warning(f"⚠️ Moonshot API 请求超时（{MOONSHOT_API_TIMEOUT}秒），跳过此图片")
            return "", ""
        except Exception as e:
            error_msg = str(e)
            if "401" in error_msg or "Invalid Authentication" in error_msg:
                logger.error(f"❌ Moonshot API 认证失败，请检查API密钥是否正确: {error_msg}")
                logger.info("💡 解决方案：请检查传入的API密钥是否为有效的Moonshot API密钥")
            else:
                logger.warning(f"❌ Moonshot API 文字提取失败: {error_msg}")
            return "", ""
    
    def _try_doubao_api(self, image_path: str, doubao_info: dict, max_chars: int) -> Tuple[str, str]:
        """尝试使用豆包API - 非VIP用户的备用方案
        
        🔧 2026-01-23 修改：使用安全的 API 调用，通过 api_manager
        """
        
        if not doubao_info or not doubao_info.get('key'):
            logger.info("⚠️  未配置豆包API密钥，跳过豆包API")
            return "", ""
        
        _debug_log_ocr("开始豆包API调用...")
        
        try:
            # 🔧 2026-02-08 修复：直接调用 safe_online_model_recognize，不再使用额外的 _OCR_GLOBAL_LOCK
            # safe_online_model_recognize 内部已通过 _GLOBAL_NETWORK_LOCK 保护网络请求
            # 双重锁会导致死锁风险
            from lib.api_manager import safe_online_model_recognize
            
            doubao_key = doubao_info.get('key', '')
            doubao_url = doubao_info.get('url', 'https://ark.cn-beijing.volces.com/api/v3')
            doubao_model = doubao_info.get('model', 'doubao-seed-1-6-vision-250815')
            
            _debug_log_ocr(f"豆包API: {doubao_url}, 模型: {doubao_model}")
            
            # 🔧 直接调用安全的 API 函数（内部已有全局网络锁保护）
            doubao_text = safe_online_model_recognize(
                image_path=image_path,
                api_key=doubao_key,
                api_base=doubao_url,
                model_name=doubao_model,
                max_chars=max_chars
            )
            
            if doubao_text and len(doubao_text.strip()) > 10:
                _debug_log_ocr(f"✅ 豆包API 成功，内容长度: {len(doubao_text)}")
                return doubao_text, "豆包API"
            else:
                _debug_log_ocr("⚠️ 豆包API 未提取到有效内容")
                return "", ""
        
        except Exception as e:
            error_msg = str(e)
            _debug_log_ocr(f"❌ 豆包API 失败: {error_msg}")
            if "401" in error_msg or "authentication" in error_msg.lower():
                logger.error(f"❌ 豆包API 认证失败: {error_msg}")
            else:
                logger.warning(f"❌ 豆包API 文字提取失败: {error_msg}")
            return "", ""
    
    def _get_moonshot_api_key(self) -> str:
        """获取Moonshot专用API密钥"""
        try:
            import json
            
            # 从多模态设置文件中读取Moonshot密钥
            config_file = os.path.join(os.path.expanduser('~'), '.fileneatai', 'multimodal_settings.json')
            if os.path.exists(config_file):
                with open(config_file, 'r', encoding='utf-8') as f:
                    settings = json.load(f)
                
                # 检查是否配置了Moonshot API
                multimodal_config = settings.get('multimodal', {})
                if multimodal_config.get('enabled', False):
                    api_base = multimodal_config.get('api_base', '').strip()
                    api_key = multimodal_config.get('api_key', '').strip()
                    
                    # 判断是否是Moonshot API配置
                    if 'moonshot.cn' in api_base.lower() and api_key:
                        logger.info("✅ 找到Moonshot专用API密钥配置")
                        return api_key
                    elif api_key and not api_base:
                        # 如果只配置了密钥没有配置地址，假设是Moonshot
                        logger.info("✅ 使用多模态设置中的API密钥作为Moonshot密钥")
                        return api_key
            
            logger.info("⚠️  未找到Moonshot专用API密钥配置")
            return ""
            
        except Exception as e:
            logger.warning(f"读取Moonshot API密钥配置失败: {str(e)}")
            return ""


# 全局实例
ocr_optimizer = OCROptimizer()
