import logging.config
import os
import tempfile
import sys
import datetime

# Determine the log directory
def get_log_dir():
    """获取日志目录"""
    if getattr(sys, 'frozen', False):
        # 打包后的exe
        base_dir = os.path.dirname(sys.executable)
    else:
        # 开发环境
        base_dir = os.path.dirname(os.path.abspath(__file__))
    
    log_dir = os.path.join(base_dir, 'logs')
    try:
        if not os.path.exists(log_dir):
            os.makedirs(log_dir)
        return log_dir
    except:
        return tempfile.gettempdir()

# Determine the appropriate log file path
def get_log_file_path():
    # If running as a packaged exe
    if getattr(sys, 'frozen', False):
        # First try to use a directory next to the executable
        try:
            base_dir = os.path.dirname(sys.executable)
            log_dir = os.path.join(base_dir, 'logs')
            if not os.path.exists(log_dir):
                os.makedirs(log_dir)
            return os.path.join(log_dir, 'app.log')
        except:
            # If that fails, use the user's temp directory
            return os.path.join(tempfile.gettempdir(), 'app.log')
    else:
        # When running in development, use current directory
        return 'app.log'

log_file_path = get_log_file_path()
LOG_DIR = get_log_dir()

LOGGING_CONFIG = {
    'version': 1,
    'disable_existing_loggers': False,  # 不禁用已有的日志器
    'formatters': {
        'standard': {
            'format': '%(asctime)s - %(name)s - %(levelname)s - %(message)s'
        },
        'detailed': {
            'format': '%(asctime)s - %(name)s - %(levelname)s - %(pathname)s:%(lineno)d - %(message)s'
        },
    },
    'handlers': {
        'console': {
            'level': 'DEBUG',
            'class': 'logging.StreamHandler',
            'formatter': 'standard',
        },
        'file': {
            'level': 'INFO',
            'class': 'logging.FileHandler',
            'filename': log_file_path,
            'mode': 'a',
            'encoding': 'utf-8',
            'formatter': 'detailed',
        },
        'error_file': {
            'level': 'ERROR',
            'class': 'logging.FileHandler',
            'filename': log_file_path,
            'mode': 'a',
            'encoding': 'utf-8',
            'formatter': 'detailed',
        },
    },
    'loggers': {
        'fileneatai': {
            'level': 'DEBUG',
            'handlers': ['console', 'file', 'error_file'],
            'propagate': False
        }
    },
    'root': {
        'level': 'WARNING',
        'handlers': ['console']
    }
}

# 配置日志
try:
    logging.config.dictConfig(LOGGING_CONFIG)
except ValueError as e:
    # Fallback configuration if we can't write to the desired location
    print(f"Warning: Could not configure logger with file handlers: {e}")
    # Configure with console only
    console_only_config = LOGGING_CONFIG.copy()
    for logger_name in console_only_config['loggers']:
        console_only_config['loggers'][logger_name]['handlers'] = ['console']
    logging.config.dictConfig(console_only_config)

# 示例使用日志
logger = logging.getLogger('fileneatai')

# 记录日志文件位置（便于用户找到）
def get_log_paths():
    """获取所有日志文件路径"""
    return {
        'app_log': log_file_path,
        'crash_log': os.path.join(LOG_DIR, 'crash.log'),
        'log_dir': LOG_DIR
    }

# 启动时记录
try:
    logger.info(f"========== 程序启动 {datetime.datetime.now().strftime('%Y-%m-%d %H:%M:%S')} ==========")
    logger.info(f"日志文件位置: {log_file_path}")
    logger.info(f"崩溃日志位置: {os.path.join(LOG_DIR, 'crash.log')}")
except:
    pass
