# -*- coding: utf-8 -*-
"""
FileNeatAI 独立License生成工具
可以在任何Python环境中运行，不依赖项目其他模块

依赖：
  pip install pycryptodome

使用：
  python standalone_license_generator.py
"""
import os
import json
import base64
import hashlib
from datetime import datetime, timedelta

try:
    from Crypto.Cipher import AES
except ImportError:
    print("❌ 缺少依赖库，请先安装：")
    print("   pip install pycryptodome")
    exit(1)

# ========== 密钥配置（与主程序保持一致） ==========
# 重要：这些密钥必须与 lib/api.py 中的密钥完全一致！
LICENSE_AES_KEY = 'A5GX7BLS6k7exu7D'  # 32字节密钥
LICENSE_AES_IV = 'A5GX7BLS6k7exu7D'  # 16字节IV
# =================================================


class StandaloneLicenseGenerator:
    """独立的License生成器"""
    
    def __init__(self, output_dir='.'):
        """
        初始化
        :param output_dir: License文件输出目录
        """
        self.output_dir = output_dir
        
    def encrypt_license(self, data: dict) -> str:
        """加密License数据"""
        try:
            # 转为JSON字符串并编码为UTF-8字节
            json_str = json.dumps(data, ensure_ascii=False)
            json_bytes = json_str.encode('utf-8')
            
            # PKCS7填充（字节级别）
            block_size = AES.block_size
            padding = block_size - len(json_bytes) % block_size
            padded_data = json_bytes + bytes([padding] * padding)
            
            # AES加密
            key = LICENSE_AES_KEY.encode('utf-8')[:32].ljust(32, b'\0')
            iv = LICENSE_AES_IV.encode('utf-8')[:16].ljust(16, b'\0')
            cipher = AES.new(key, AES.MODE_CBC, iv)
            encrypted = cipher.encrypt(padded_data)
            
            # Base64编码
            return base64.b64encode(encrypted).decode('utf-8')
        except Exception as e:
            print(f"❌ 加密失败: {str(e)}")
            return None
    
    def calculate_signature(self, data: dict) -> str:
        """计算License签名（防篡改）"""
        # 按key排序后连接成字符串
        sorted_keys = sorted([k for k in data.keys() if k != 'signature'])
        content = ''.join([f"{k}:{data[k]}" for k in sorted_keys])
        
        # 加盐后计算SHA256
        salt = "FileNeatAI_Offline_Salt_2024"
        signature_content = content + salt
        return hashlib.sha256(signature_content.encode('utf-8')).hexdigest()
    
    def generate_license(self, machine_code: str, local_unlock: int, 
                        duration_days: int = 365, note: str = '',
                        online_limit: int = 35000,
                        output_filename: str = 'license.dat') -> bool:
        """
        生成License文件
        
        Args:
            machine_code: 机器码（必须）
            local_unlock: 本地解锁权限 (0=无权限, 1=有权限)
            duration_days: 有效期天数（-1表示永久）
            note: 备注信息（可选，如客户名称等）
            online_limit: 在线模型使用次数限制（默认35000）
            output_filename: 输出文件名
        """
        try:
            # 计算到期时间
            issue_time = datetime.now()
            if duration_days == -1:
                expire_time = datetime(2099, 12, 31, 23, 59, 59)
            else:
                expire_time = issue_time + timedelta(days=duration_days)
            
            # License数据（简化版，只包含核心字段）
            license_data = {
                'machine_code': machine_code,
                'local_unlock': local_unlock,  # 0或1
                'online_limit': online_limit,  # 在线模型使用次数限制
                'issue_time': issue_time.strftime('%Y-%m-%d %H:%M:%S'),
                'expire_time': expire_time.strftime('%Y-%m-%d %H:%M:%S'),
                'note': note,
                'version': '3.0'
            }
            
            # 计算签名（不包含signature字段本身）
            license_data['signature'] = self.calculate_signature(license_data)
            
            # 加密并保存
            encrypted = self.encrypt_license(license_data)
            if not encrypted:
                return False
            
            # 确保输出目录存在
            os.makedirs(self.output_dir, exist_ok=True)
            output_path = os.path.join(self.output_dir, output_filename)
            
            with open(output_path, 'w', encoding='utf-8') as f:
                f.write(encrypted)
            
            print(f"✅ License生成成功!")
            print(f"📁 文件位置: {os.path.abspath(output_path)}")
            print(f"🔓 本地解锁: {local_unlock}")
            print(f"📊 在线模型限制: {online_limit}次")
            print(f"⏰ 到期时间: {expire_time}")
            print(f"📅 有效期: {'永久' if duration_days == -1 else f'{duration_days}天'}")
            if note:
                print(f"📝 备注: {note}")
            
            return True
            
        except Exception as e:
            print(f"❌ License生成失败: {str(e)}")
            return False
    
    def batch_generate(self, licenses_config: list) -> int:
        """
        批量生成License
        
        Args:
            licenses_config: License配置列表
            [
                {
                    'machine_code': 'xxx',
                    'local_unlock': 1,
                    'online_limit': 35000,
                    'duration_days': 365,
                    'note': '客户名称',
                    'filename': 'customer1.dat'
                },
                ...
            ]
        
        Returns:
            成功生成的数量
        """
        success_count = 0
        
        for i, config in enumerate(licenses_config, 1):
            print(f"\n[{i}/{len(licenses_config)}] 生成License...")
            
            if 'machine_code' not in config:
                print("❌ 缺少machine_code，跳过")
                continue
            
            filename = config.get('filename', f'license_{i}.dat')
            
            if self.generate_license(
                machine_code=config['machine_code'],
                local_unlock=config.get('local_unlock', 1),
                online_limit=config.get('online_limit', 35000),
                duration_days=config.get('duration_days', 365),
                note=config.get('note', ''),
                output_filename=filename
            ):
                success_count += 1
        
        return success_count


def interactive_generate():
    """交互式生成License"""
    print("=" * 70)
    print("FileNeatAI 独立License生成工具")
    print("=" * 70)
    
    # 获取机器码
    print("\n【步骤1】输入客户的机器码")
    machine_code = input("请输入机器码: ").strip()
    
    if not machine_code:
        print("❌ 机器码不能为空")
        return
    
    # 设置本地解锁权限
    print("\n【步骤2】设置本地解锁权限")
    print("0 = 无权限（只能在线使用，需要联网）")
    print("1 = 有权限（可完全离线使用本地模型）")
    
    local_unlock_input = input("请选择 (0/1，默认1): ").strip()
    local_unlock = 1 if local_unlock_input in ['', '1'] else 0
    
    # 设置在线模型限制
    print("\n【步骤3】设置在线模型使用次数")
    print("1. 10,000次")
    print("2. 35,000次 ★推荐")
    print("3. 50,000次")
    print("4. 100,000次")
    print("5. 无限制 (1,000,000次)")
    print("6. 自定义次数")
    
    limit_map = {
        '1': 10000,
        '2': 35000,
        '3': 50000,
        '4': 100000,
        '5': 1000000
    }
    
    limit_choice = input("请选择 (1-6，默认2): ").strip()
    if not limit_choice:
        limit_choice = '2'
    
    if limit_choice == '6':
        online_limit = int(input("请输入自定义次数: "))
    else:
        online_limit = limit_map.get(limit_choice, 35000)
    
    # 设置有效期
    print("\n【步骤4】设置有效期")
    print("1. 7天试用")
    print("2. 1个月 (30天)")
    print("3. 3个月 (90天)")
    print("4. 6个月 (180天)")
    print("5. 1年 (365天) ★推荐")
    print("6. 2年 (730天)")
    print("7. 3年 (1095天)")
    print("8. 永久 (到2099年)")
    print("9. 自定义天数")
    
    duration_map = {
        '1': 7,
        '2': 30,
        '3': 90,
        '4': 180,
        '5': 365,
        '6': 730,
        '7': 1095,
        '8': -1
    }
    
    duration_choice = input("请选择 (1-9，默认5): ").strip()
    if not duration_choice:
        duration_choice = '5'
    
    if duration_choice == '9':
        duration_days = int(input("请输入自定义天数: "))
    else:
        duration_days = duration_map.get(duration_choice, 365)
    
    # 备注信息
    print("\n【步骤5】备注信息（可选）")
    note = input("请输入备注（如客户名称，可直接回车跳过）: ").strip()
    
    # 输出目录
    print("\n【步骤6】输出目录")
    output_dir = input("请输入输出目录（默认当前目录）: ").strip()
    if not output_dir:
        output_dir = '.'
    
    # 确认信息
    print("\n" + "=" * 70)
    print("【确认信息】")
    print("=" * 70)
    print(f"📱 机器码: {machine_code}")
    print(f"🔓 本地解锁: {'是 (1) - 可完全离线使用' if local_unlock == 1 else '否 (0) - 需要联网'}")
    print(f"📊 在线模型限制: {online_limit:,}次")
    print(f"⏰ 有效期: {'永久' if duration_days == -1 else f'{duration_days}天'}")
    if note:
        print(f"📝 备注: {note}")
    print(f"📁 输出目录: {os.path.abspath(output_dir)}")
    print("=" * 70)
    
    confirm = input("\n确认生成? (y/n，默认y): ").strip().lower()
    if confirm and confirm != 'y':
        print("❌ 已取消")
        return
    
    # 生成License
    print("\n⏳ 正在生成License...")
    generator = StandaloneLicenseGenerator(output_dir=output_dir)
    generator.generate_license(
        machine_code=machine_code,
        local_unlock=local_unlock,
        online_limit=online_limit,
        duration_days=duration_days,
        note=note
    )


def batch_generate_from_json(json_file: str):
    """从JSON文件批量生成License"""
    try:
        with open(json_file, 'r', encoding='utf-8') as f:
            config = json.load(f)
        
        licenses = config.get('licenses', [])
        output_dir = config.get('output_dir', './licenses')
        
        print(f"📋 从配置文件读取到 {len(licenses)} 个License待生成")
        print(f"📁 输出目录: {output_dir}")
        
        generator = StandaloneLicenseGenerator(output_dir=output_dir)
        success_count = generator.batch_generate(licenses)
        
        print("\n" + "=" * 70)
        print(f"✅ 批量生成完成: {success_count}/{len(licenses)} 个成功")
        print("=" * 70)
        
    except FileNotFoundError:
        print(f"❌ 文件不存在: {json_file}")
    except json.JSONDecodeError:
        print(f"❌ JSON格式错误: {json_file}")
    except Exception as e:
        print(f"❌ 批量生成失败: {str(e)}")


def create_sample_config():
    """创建示例配置文件"""
    sample_config = {
        "output_dir": "./licenses",
        "licenses": [
            {
                "machine_code": "EXAMPLE-MACHINE-CODE-001",
                "local_unlock": 1,
                "online_limit": 35000,
                "duration_days": 365,
                "note": "客户A - 2024年采购",
                "filename": "customer_a.dat"
            },
            {
                "machine_code": "EXAMPLE-MACHINE-CODE-002",
                "local_unlock": 1,
                "online_limit": 100000,
                "duration_days": 730,
                "note": "客户B - 2年版本 高级套餐",
                "filename": "customer_b.dat"
            }
        ]
    }
    
    with open('license_config_sample.json', 'w', encoding='utf-8') as f:
        json.dump(sample_config, f, ensure_ascii=False, indent=2)
    
    print("✅ 示例配置文件已创建: license_config_sample.json")
    print("   请修改后使用: python standalone_license_generator.py --batch license_config_sample.json")


def main():
    """主函数"""
    import sys
    
    print("\n" + "=" * 70)
    print("FileNeatAI 独立License生成工具 v1.0")
    print("=" * 70)
    
    # 命令行参数处理
    if len(sys.argv) > 1:
        # 快速模式：直接传入机器码生成永久License
        if not sys.argv[1].startswith('--'):
            machine_code = sys.argv[1]
            
            # 可选参数：天数（默认永久）
            duration_days = -1
            if len(sys.argv) > 2 and sys.argv[2].isdigit():
                duration_days = int(sys.argv[2])
            
            # 可选参数：备注
            note = sys.argv[3] if len(sys.argv) > 3 else ''
            
            print("\n🚀 快速生成模式")
            print(f"📱 机器码: {machine_code}")
            print(f"⏰ 有效期: {'永久' if duration_days == -1 else f'{duration_days}天'}")
            if note:
                print(f"📝 备注: {note}")
            print()
            
            generator = StandaloneLicenseGenerator(output_dir='.')
            success = generator.generate_license(
                machine_code=machine_code,
                local_unlock=1,  # 默认开启本地解锁
                duration_days=duration_days,
                note=note,
                output_filename='license.dat'
            )
            
            if success:
                print("\n💡 提示：请将 license.dat 发送给客户")
            return
        
        if sys.argv[1] == '--batch' and len(sys.argv) > 2:
            # 批量生成模式
            batch_generate_from_json(sys.argv[2])
            return
        elif sys.argv[1] == '--sample':
            # 创建示例配置
            create_sample_config()
            return
        elif sys.argv[1] == '--help':
            print("\n使用方法:")
            print("  1. 快速生成（永久License）:")
            print("     python standalone_license_generator.py <机器码>")
            print("     示例: python standalone_license_generator.py 92BB5DF5-C882-775E-AB18-E6115ABF1F38")
            print("\n  2. 快速生成（指定天数）:")
            print("     python standalone_license_generator.py <机器码> <天数> [备注]")
            print("     示例: python standalone_license_generator.py 92BB5DF5-C882-775E-AB18-E6115ABF1F38 365 客户A")
            print("\n  3. 交互式生成:")
            print("     python standalone_license_generator.py")
            print("\n  4. 批量生成:")
            print("     python standalone_license_generator.py --batch config.json")
            print("\n  5. 创建示例配置:")
            print("     python standalone_license_generator.py --sample")
            return
    
    # 交互式生成
    while True:
        print("\n请选择操作：")
        print("1. 生成单个License")
        print("2. 批量生成License（从JSON配置文件）")
        print("3. 创建示例配置文件")
        print("4. 退出")
        print("=" * 70)
        
        choice = input("请输入选项 (1-4): ").strip()
        
        if choice == '1':
            interactive_generate()
        elif choice == '2':
            json_file = input("请输入配置文件路径: ").strip()
            if json_file:
                batch_generate_from_json(json_file)
        elif choice == '3':
            create_sample_config()
        elif choice == '4':
            print("\n再见！")
            break
        else:
            print("❌ 无效选项，请重新选择")


if __name__ == '__main__':
    main()

