# -*- coding: utf-8 -*-
"""
优化的UI组件 - 解决大量文件显示时的性能问题
"""

import os
import time
import threading
from typing import List, Dict, Optional, Callable
from PySide6.QtCore import Qt, QTimer, QObject, Signal, QAbstractItemModel, QModelIndex, QSize
from PySide6.QtWidgets import (QWidget, QVBoxLayout, QHBoxLayout, QLabel, QPushButton, 
                              QProgressBar, QListView, QStyledItemDelegate, QTreeView,
                              QAbstractItemView, QFrame, QScrollArea, QSizePolicy, QDialog)
from PySide6.QtGui import QIcon, QPixmap, QStandardItemModel, QStandardItem
from logger import logger


class VirtualFileListModel(QAbstractItemModel):
    """虚拟文件列表模型 - 支持大量文件的高效显示"""
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self.file_data = []
        self.headers = ['文件名', '大小', '类型', '状态']
        
    def set_file_data(self, file_data: List[Dict]):
        """设置文件数据"""
        self.beginResetModel()
        self.file_data = file_data
        self.endResetModel()
    
    def add_files_batch(self, new_files: List[Dict]):
        """批量添加文件"""
        if not new_files:
            return
            
        start_row = len(self.file_data)
        end_row = start_row + len(new_files) - 1
        
        self.beginInsertRows(QModelIndex(), start_row, end_row)
        self.file_data.extend(new_files)
        self.endInsertRows()
    
    def rowCount(self, parent=QModelIndex()):
        return len(self.file_data)
    
    def columnCount(self, parent=QModelIndex()):
        return len(self.headers)
    
    def data(self, index, role=Qt.DisplayRole):
        if not index.isValid() or index.row() >= len(self.file_data):
            return None
            
        file_info = self.file_data[index.row()]
        column = index.column()
        
        if role == Qt.DisplayRole:
            if column == 0:  # 文件名
                return file_info.get('file_name', '')
            elif column == 1:  # 大小
                size = file_info.get('file_size', 0)
                return self._format_file_size(size)
            elif column == 2:  # 类型
                return file_info.get('file_type', '')
            elif column == 3:  # 状态
                return file_info.get('status', '待处理')
        elif role == Qt.UserRole:
            return file_info
            
        return None
    
    def headerData(self, section, orientation, role=Qt.DisplayRole):
        if orientation == Qt.Horizontal and role == Qt.DisplayRole:
            return self.headers[section]
        return None
    
    def index(self, row, column, parent=QModelIndex()):
        if not self.hasIndex(row, column, parent):
            return QModelIndex()
        return self.createIndex(row, column)
    
    def parent(self, index):
        return QModelIndex()
    
    def _format_file_size(self, size: int) -> str:
        """格式化文件大小"""
        if size < 1024:
            return f"{size} B"
        elif size < 1024 * 1024:
            return f"{size / 1024:.1f} KB"
        elif size < 1024 * 1024 * 1024:
            return f"{size / (1024 * 1024):.1f} MB"
        else:
            return f"{size / (1024 * 1024 * 1024):.1f} GB"


class OptimizedFileListWidget(QWidget):
    """优化的文件列表组件"""
    
    # 信号定义
    file_selected = Signal(str)  # 文件被选中
    files_deleted = Signal(list)  # 文件被删除
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setup_ui()
        self.model = VirtualFileListModel()
        self.list_view.setModel(self.model)
        
        # 批量更新定时器
        self.update_timer = QTimer()
        self.update_timer.timeout.connect(self._process_pending_updates)
        self.update_timer.setSingleShot(False)
        self.pending_updates = []
        self.update_lock = threading.Lock()
        
    def setup_ui(self):
        """设置UI"""
        layout = QVBoxLayout(self)
        layout.setContentsMargins(0, 0, 0, 0)
        
        # 顶部信息栏
        self.info_frame = QFrame()
        self.info_frame.setStyleSheet("""
            QFrame {
                background-color: #2a2a2a;
                border-radius: 4px;
                padding: 8px;
            }
        """)
        info_layout = QHBoxLayout(self.info_frame)
        
        self.file_count_label = QLabel("文件数量: 0")
        self.file_count_label.setStyleSheet("color: #ffffff; font-size: 14px;")
        
        self.total_size_label = QLabel("总大小: 0 B")
        self.total_size_label.setStyleSheet("color: #ffffff; font-size: 14px;")
        
        self.status_label = QLabel("就绪")
        self.status_label.setStyleSheet("color: #19AF39; font-size: 14px;")
        
        info_layout.addWidget(self.file_count_label)
        info_layout.addWidget(self.total_size_label)
        info_layout.addStretch()
        info_layout.addWidget(self.status_label)
        
        layout.addWidget(self.info_frame)
        
        # 文件列表视图
        self.list_view = QListView()
        self.list_view.setStyleSheet("""
            QListView {
                background-color: #1e1e1e;
                border: 1px solid #3a3a3a;
                border-radius: 4px;
                color: #ffffff;
                selection-background-color: #19AF39;
            }
            QListView::item {
                padding: 8px;
                border-bottom: 1px solid #3a3a3a;
            }
            QListView::item:hover {
                background-color: #2a2a2a;
            }
        """)
        
        # 设置视图属性
        self.list_view.setAlternatingRowColors(True)
        self.list_view.setSelectionMode(QAbstractItemView.ExtendedSelection)
        
        layout.addWidget(self.list_view)
        
        # 底部操作栏
        self.action_frame = QFrame()
        action_layout = QHBoxLayout(self.action_frame)
        
        self.clear_button = QPushButton("清空列表")
        self.clear_button.setStyleSheet("""
            QPushButton {
                background-color: #dc3545;
                color: white;
                border: none;
                padding: 8px 16px;
                border-radius: 4px;
                font-size: 14px;
            }
            QPushButton:hover {
                background-color: #c82333;
            }
        """)
        self.clear_button.clicked.connect(self.clear_files)
        
        self.select_all_button = QPushButton("全选")
        self.select_all_button.setStyleSheet("""
            QPushButton {
                background-color: #6c757d;
                color: white;
                border: none;
                padding: 8px 16px;
                border-radius: 4px;
                font-size: 14px;
            }
            QPushButton:hover {
                background-color: #5a6268;
            }
        """)
        self.select_all_button.clicked.connect(self.select_all_files)
        
        action_layout.addWidget(self.clear_button)
        action_layout.addWidget(self.select_all_button)
        action_layout.addStretch()
        
        layout.addWidget(self.action_frame)
    
    def add_files_async(self, file_list: List[Dict]):
        """异步添加文件（批量）"""
        with self.update_lock:
            self.pending_updates.extend(file_list)
        
        # 启动更新定时器
        if not self.update_timer.isActive():
            self.update_timer.start(200)  # 每200ms处理一次批量更新
    
    def _process_pending_updates(self):
        """处理待更新的文件"""
        with self.update_lock:
            if not self.pending_updates:
                self.update_timer.stop()
                return
            
            # 取出一批进行更新
            batch_size = 100
            current_batch = self.pending_updates[:batch_size]
            self.pending_updates = self.pending_updates[batch_size:]
        
        # 更新模型
        self.model.add_files_batch(current_batch)
        
        # 更新信息栏
        self._update_info_bar()
        
        # 如果还有待更新的文件，继续定时器
        if not self.pending_updates:
            self.update_timer.stop()
    
    def _update_info_bar(self):
        """更新信息栏"""
        file_count = len(self.model.file_data)
        total_size = sum(file_info.get('file_size', 0) for file_info in self.model.file_data)
        
        self.file_count_label.setText(f"文件数量: {file_count}")
        self.total_size_label.setText(f"总大小: {self._format_file_size(total_size)}")
        
        if file_count > 0:
            self.status_label.setText("就绪")
            self.status_label.setStyleSheet("color: #19AF39; font-size: 14px;")
        else:
            self.status_label.setText("无文件")
            self.status_label.setStyleSheet("color: #6c757d; font-size: 14px;")
    
    def clear_files(self):
        """清空文件列表"""
        self.model.set_file_data([])
        self._update_info_bar()
    
    def select_all_files(self):
        """全选文件"""
        self.list_view.selectAll()
    
    def _format_file_size(self, size: int) -> str:
        """格式化文件大小"""
        if size < 1024:
            return f"{size} B"
        elif size < 1024 * 1024:
            return f"{size / 1024:.1f} KB"
        elif size < 1024 * 1024 * 1024:
            return f"{size / (1024 * 1024):.1f} MB"
        else:
            return f"{size / (1024 * 1024 * 1024):.1f} GB"


class MultimodalConfigWidget(QWidget):
    """多模态AI配置组件"""
    
    config_changed = Signal(dict)  # 配置变更信号
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setup_ui()
        self.load_config()
        
    def setup_ui(self):
        """设置UI"""
        layout = QVBoxLayout(self)
        
        # 标题
        title_label = QLabel("多模态AI图片识别配置")
        title_label.setStyleSheet("""
            QLabel {
                font-size: 18px;
                font-weight: bold;
                color: #ffffff;
                padding: 10px 0;
            }
        """)
        layout.addWidget(title_label)
        
        # 服务商选择区域
        self.create_provider_section(layout)
        
        # 成本控制区域
        self.create_cost_control_section(layout)
        
        # 高级设置区域
        self.create_advanced_settings_section(layout)
        
        # 测试区域
        self.create_test_section(layout)
        
        layout.addStretch()
    
    def create_provider_section(self, parent_layout):
        """创建服务商选择区域"""
        from PySide6.QtWidgets import QGroupBox, QRadioButton, QLineEdit, QCheckBox
        
        group = QGroupBox("AI服务商配置")
        group.setStyleSheet("""
            QGroupBox {
                font-size: 16px;
                font-weight: bold;
                color: #ffffff;
                border: 2px solid #3a3a3a;
                border-radius: 8px;
                margin-top: 10px;
                padding-top: 10px;
            }
            QGroupBox::title {
                subcontrol-origin: margin;
                left: 10px;
                padding: 0 5px 0 5px;
            }
        """)
        
        group_layout = QVBoxLayout(group)
        
        # Claude配置
        claude_frame = QFrame()
        claude_layout = QHBoxLayout(claude_frame)
        
        self.claude_radio = QRadioButton("Claude 3.5 Sonnet")
        self.claude_radio.setChecked(True)  # 默认选择
        self.claude_radio.setStyleSheet("color: #ffffff; font-size: 14px;")
        
        self.claude_key_input = QLineEdit()
        self.claude_key_input.setPlaceholderText("输入Claude API密钥...")
        self.claude_key_input.setEchoMode(QLineEdit.Password)
        self.claude_key_input.setStyleSheet("""
            QLineEdit {
                background-color: #2a2a2a;
                border: 1px solid #3a3a3a;
                border-radius: 4px;
                padding: 8px;
                color: #ffffff;
                font-size: 14px;
            }
        """)
        
        claude_layout.addWidget(self.claude_radio)
        claude_layout.addWidget(QLabel("API密钥:"))
        claude_layout.addWidget(self.claude_key_input)
        
        group_layout.addWidget(claude_frame)
        
        # GPT-4 Vision配置
        gpt4_frame = QFrame()
        gpt4_layout = QHBoxLayout(gpt4_frame)
        
        self.gpt4_radio = QRadioButton("GPT-4 Vision")
        self.gpt4_radio.setStyleSheet("color: #ffffff; font-size: 14px;")
        
        self.gpt4_key_input = QLineEdit()
        self.gpt4_key_input.setPlaceholderText("输入OpenAI API密钥...")
        self.gpt4_key_input.setEchoMode(QLineEdit.Password)
        self.gpt4_key_input.setStyleSheet(self.claude_key_input.styleSheet())
        
        gpt4_layout.addWidget(self.gpt4_radio)
        gpt4_layout.addWidget(QLabel("API密钥:"))
        gpt4_layout.addWidget(self.gpt4_key_input)
        
        group_layout.addWidget(gpt4_frame)
        
        # Gemini配置
        gemini_frame = QFrame()
        gemini_layout = QHBoxLayout(gemini_frame)
        
        self.gemini_radio = QRadioButton("Gemini Pro Vision")
        self.gemini_radio.setStyleSheet("color: #ffffff; font-size: 14px;")
        
        self.gemini_key_input = QLineEdit()
        self.gemini_key_input.setPlaceholderText("输入Google API密钥...")
        self.gemini_key_input.setEchoMode(QLineEdit.Password)
        self.gemini_key_input.setStyleSheet(self.claude_key_input.styleSheet())
        
        gemini_layout.addWidget(self.gemini_radio)
        gemini_layout.addWidget(QLabel("API密钥:"))
        gemini_layout.addWidget(self.gemini_key_input)
        
        group_layout.addWidget(gemini_frame)
        
        # 文心一言配置
        wenxin_frame = QFrame()
        wenxin_layout = QHBoxLayout(wenxin_frame)
        
        self.wenxin_radio = QRadioButton("文心一言4.0")
        self.wenxin_radio.setStyleSheet("color: #ffffff; font-size: 14px;")
        
        self.wenxin_key_input = QLineEdit()
        self.wenxin_key_input.setPlaceholderText("输入百度API密钥...")
        self.wenxin_key_input.setEchoMode(QLineEdit.Password)
        self.wenxin_key_input.setStyleSheet(self.claude_key_input.styleSheet())
        
        wenxin_layout.addWidget(self.wenxin_radio)
        wenxin_layout.addWidget(QLabel("API密钥:"))
        wenxin_layout.addWidget(self.wenxin_key_input)
        
        group_layout.addWidget(wenxin_frame)
        
        parent_layout.addWidget(group)
    
    def create_cost_control_section(self, parent_layout):
        """创建成本控制区域"""
        from PySide6.QtWidgets import QGroupBox, QSpinBox, QDoubleSpinBox, QSlider
        
        group = QGroupBox("成本控制")
        group.setStyleSheet("""
            QGroupBox {
                font-size: 16px;
                font-weight: bold;
                color: #ffffff;
                border: 2px solid #3a3a3a;
                border-radius: 8px;
                margin-top: 10px;
                padding-top: 10px;
            }
        """)
        
        group_layout = QVBoxLayout(group)
        
        # 预算设置
        budget_frame = QFrame()
        budget_layout = QHBoxLayout(budget_frame)
        
        budget_label = QLabel("月度预算限制:")
        budget_label.setStyleSheet("color: #ffffff; font-size: 14px;")
        
        self.budget_spinbox = QDoubleSpinBox()
        self.budget_spinbox.setRange(1.0, 1000.0)
        self.budget_spinbox.setValue(50.0)
        self.budget_spinbox.setSuffix(" USD")
        self.budget_spinbox.setStyleSheet("""
            QDoubleSpinBox {
                background-color: #2a2a2a;
                border: 1px solid #3a3a3a;
                border-radius: 4px;
                padding: 8px;
                color: #ffffff;
                font-size: 14px;
            }
        """)
        
        budget_layout.addWidget(budget_label)
        budget_layout.addWidget(self.budget_spinbox)
        budget_layout.addStretch()
        
        group_layout.addWidget(budget_frame)
        
        # 单次处理限制
        limit_frame = QFrame()
        limit_layout = QHBoxLayout(limit_frame)
        
        limit_label = QLabel("单次最大处理图片数:")
        limit_label.setStyleSheet("color: #ffffff; font-size: 14px;")
        
        self.limit_spinbox = QSpinBox()
        self.limit_spinbox.setRange(1, 1000)
        self.limit_spinbox.setValue(100)
        self.limit_spinbox.setStyleSheet(self.budget_spinbox.styleSheet())
        
        limit_layout.addWidget(limit_label)
        limit_layout.addWidget(self.limit_spinbox)
        limit_layout.addStretch()
        
        group_layout.addWidget(limit_frame)
        
        # 使用统计显示
        self.usage_label = QLabel("本月已使用: $0.00 / $50.00")
        self.usage_label.setStyleSheet("color: #19AF39; font-size: 14px;")
        group_layout.addWidget(self.usage_label)
        
        parent_layout.addWidget(group)
    
    def create_advanced_settings_section(self, parent_layout):
        """创建高级设置区域"""
        from PySide6.QtWidgets import QGroupBox, QComboBox, QCheckBox
        
        group = QGroupBox("高级设置")
        group.setStyleSheet("""
            QGroupBox {
                font-size: 16px;
                font-weight: bold;
                color: #ffffff;
                border: 2px solid #3a3a3a;
                border-radius: 8px;
                margin-top: 10px;
                padding-top: 10px;
            }
        """)
        
        group_layout = QVBoxLayout(group)
        
        # 识别质量设置
        quality_frame = QFrame()
        quality_layout = QHBoxLayout(quality_frame)
        
        quality_label = QLabel("识别质量:")
        quality_label.setStyleSheet("color: #ffffff; font-size: 14px;")
        
        self.quality_combo = QComboBox()
        self.quality_combo.addItems(["快速", "标准", "精确"])
        self.quality_combo.setCurrentText("标准")
        self.quality_combo.setStyleSheet("""
            QComboBox {
                background-color: #2a2a2a;
                border: 1px solid #3a3a3a;
                border-radius: 4px;
                padding: 8px;
                color: #ffffff;
                font-size: 14px;
                min-width: 120px;
            }
            QComboBox::drop-down {
                border: none;
            }
            QComboBox::down-arrow {
                width: 20px;
                height: 20px;
            }
        """)
        
        quality_layout.addWidget(quality_label)
        quality_layout.addWidget(self.quality_combo)
        quality_layout.addStretch()
        
        group_layout.addWidget(quality_frame)
        
        # 智能路由设置
        self.smart_routing_checkbox = QCheckBox("启用智能路由（自动选择最佳服务商）")
        self.smart_routing_checkbox.setChecked(True)
        self.smart_routing_checkbox.setStyleSheet("color: #ffffff; font-size: 14px;")
        group_layout.addWidget(self.smart_routing_checkbox)
        
        # 缓存设置
        self.enable_cache_checkbox = QCheckBox("启用结果缓存（提高重复处理速度）")
        self.enable_cache_checkbox.setChecked(True)
        self.enable_cache_checkbox.setStyleSheet("color: #ffffff; font-size: 14px;")
        group_layout.addWidget(self.enable_cache_checkbox)
        
        # 降级策略
        self.fallback_ocr_checkbox = QCheckBox("AI失败时自动降级到OCR")
        self.fallback_ocr_checkbox.setChecked(True)
        self.fallback_ocr_checkbox.setStyleSheet("color: #ffffff; font-size: 14px;")
        group_layout.addWidget(self.fallback_ocr_checkbox)
        
        parent_layout.addWidget(group)
    
    def create_test_section(self, parent_layout):
        """创建测试区域"""
        from PySide6.QtWidgets import QGroupBox, QTextEdit
        
        group = QGroupBox("服务测试")
        group.setStyleSheet("""
            QGroupBox {
                font-size: 16px;
                font-weight: bold;
                color: #ffffff;
                border: 2px solid #3a3a3a;
                border-radius: 8px;
                margin-top: 10px;
                padding-top: 10px;
            }
        """)
        
        group_layout = QVBoxLayout(group)
        
        # 测试按钮
        test_frame = QFrame()
        test_layout = QHBoxLayout(test_frame)
        
        self.test_button = QPushButton("测试当前配置")
        self.test_button.setStyleSheet("""
            QPushButton {
                background-color: #19AF39;
                color: white;
                border: none;
                padding: 10px 20px;
                border-radius: 4px;
                font-size: 14px;
                font-weight: bold;
            }
            QPushButton:hover {
                background-color: #16973a;
            }
            QPushButton:disabled {
                background-color: #666666;
                color: #999999;
            }
        """)
        self.test_button.clicked.connect(self.test_current_config)
        
        self.test_status_label = QLabel("未测试")
        self.test_status_label.setStyleSheet("color: #6c757d; font-size: 14px;")
        
        test_layout.addWidget(self.test_button)
        test_layout.addWidget(self.test_status_label)
        test_layout.addStretch()
        
        group_layout.addWidget(test_frame)
        
        # 测试结果显示
        self.test_result_text = QTextEdit()
        self.test_result_text.setMaximumHeight(150)
        self.test_result_text.setStyleSheet("""
            QTextEdit {
                background-color: #1e1e1e;
                border: 1px solid #3a3a3a;
                border-radius: 4px;
                color: #ffffff;
                font-size: 12px;
                font-family: 'Courier New', monospace;
            }
        """)
        self.test_result_text.setPlaceholderText("测试结果将显示在这里...")
        
        group_layout.addWidget(self.test_result_text)
        
        parent_layout.addWidget(group)
    
    def test_current_config(self):
        """测试当前配置"""
        self.test_button.setEnabled(False)
        self.test_status_label.setText("测试中...")
        self.test_status_label.setStyleSheet("color: #ffc107; font-size: 14px;")
        self.test_result_text.clear()
        
        # 这里应该启动一个测试线程
        # 为了演示，我们模拟一个测试过程
        QTimer.singleShot(2000, self._simulate_test_complete)
    
    def _simulate_test_complete(self):
        """模拟测试完成"""
        self.test_button.setEnabled(True)
        self.test_status_label.setText("测试完成")
        self.test_status_label.setStyleSheet("color: #19AF39; font-size: 14px;")
        
        # 模拟测试结果
        test_result = """测试结果：
✓ Claude 3.5 Sonnet - 连接成功
✓ 图片识别功能正常
✓ 成本估算: $0.015/张
⚠ 建议: 当前配置适合中等规模使用

测试用时: 1.2秒
预计处理速度: 2-3张/分钟"""
        
        self.test_result_text.setText(test_result)
    
    def load_config(self):
        """加载配置"""
        # 这里应该从配置文件加载设置
        pass
    
    def save_config(self):
        """保存配置"""
        config = self.get_current_config()
        # 这里应该保存到配置文件
        self.config_changed.emit(config)
    
    def get_current_config(self) -> dict:
        """获取当前配置"""
        provider = "claude"
        if self.gpt4_radio.isChecked():
            provider = "gpt4_vision"
        elif self.gemini_radio.isChecked():
            provider = "gemini_pro"
        elif self.wenxin_radio.isChecked():
            provider = "wenxin"
        
        return {
            'provider': provider,
            'api_keys': {
                'claude': self.claude_key_input.text(),
                'gpt4_vision': self.gpt4_key_input.text(),
                'gemini_pro': self.gemini_key_input.text(),
                'wenxin': self.wenxin_key_input.text() if hasattr(self, 'wenxin_key_input') else ''
            },
            'budget_limit': self.budget_spinbox.value(),
            'max_images_per_batch': self.limit_spinbox.value(),
            'quality_level': self.quality_combo.currentText(),
            'smart_routing': self.smart_routing_checkbox.isChecked(),
            'enable_cache': self.enable_cache_checkbox.isChecked(),
            'fallback_ocr': self.fallback_ocr_checkbox.isChecked()
        }


class PerformanceIndicator(QWidget):
    """性能指示器组件"""
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setup_ui()
        
        # 性能数据
        self.metrics = {
            'cpu_percent': 0,
            'memory_percent': 0,
            'processing_rate': 0,
            'error_rate': 0
        }
    
    def setup_ui(self):
        """设置UI"""
        layout = QHBoxLayout(self)
        layout.setContentsMargins(5, 5, 5, 5)
        
        # CPU使用率
        self.cpu_label = QLabel("CPU: 0%")
        self.cpu_label.setStyleSheet("color: #ffffff; font-size: 12px;")
        
        # 内存使用率
        self.memory_label = QLabel("内存: 0%")
        self.memory_label.setStyleSheet("color: #ffffff; font-size: 12px;")
        
        # 处理速度
        self.speed_label = QLabel("速度: 0 文件/秒")
        self.speed_label.setStyleSheet("color: #ffffff; font-size: 12px;")
        
        # 错误率
        self.error_label = QLabel("错误率: 0%")
        self.error_label.setStyleSheet("color: #19AF39; font-size: 12px;")
        
        layout.addWidget(self.cpu_label)
        layout.addWidget(QLabel("|"))
        layout.addWidget(self.memory_label)
        layout.addWidget(QLabel("|"))
        layout.addWidget(self.speed_label)
        layout.addWidget(QLabel("|"))
        layout.addWidget(self.error_label)
        layout.addStretch()
    
    def update_metrics(self, metrics: dict):
        """更新性能指标"""
        self.metrics.update(metrics)
        
        # 更新显示
        self.cpu_label.setText(f"CPU: {metrics.get('cpu_percent', 0):.1f}%")
        self.memory_label.setText(f"内存: {metrics.get('memory_percent', 0):.1f}%")
        self.speed_label.setText(f"速度: {metrics.get('processing_rate', 0):.1f} 文件/秒")
        
        # 错误率颜色变化
        error_rate = metrics.get('error_rate', 0) * 100
        self.error_label.setText(f"错误率: {error_rate:.1f}%")
        
        if error_rate < 5:
            color = "#19AF39"  # 绿色
        elif error_rate < 15:
            color = "#ffc107"  # 黄色
        else:
            color = "#dc3545"  # 红色
        
        self.error_label.setStyleSheet(f"color: {color}; font-size: 12px;")


class SmartProgressDialog(QDialog):
    """智能进度对话框"""
    
    cancel_requested = Signal()
    
    def __init__(self, title: str = "处理中", parent=None):
        super().__init__(parent)
        self.setWindowTitle(title)
        self.setWindowFlags(Qt.Dialog | Qt.WindowCloseButtonHint)
        self.setModal(True)
        self.resize(500, 300)
        
        self.setup_ui()
        self.start_time = time.time()
        
        # 居中显示
        self.center_on_screen()
    
    def setup_ui(self):
        """设置UI"""
        layout = QVBoxLayout(self)
        
        # 主要进度信息
        self.main_label = QLabel("正在处理文件...")
        self.main_label.setStyleSheet("font-size: 16px; color: #ffffff; padding: 10px;")
        layout.addWidget(self.main_label)
        
        # 进度条
        self.progress_bar = QProgressBar()
        self.progress_bar.setStyleSheet("""
            QProgressBar {
                border: 2px solid #3a3a3a;
                border-radius: 8px;
                background-color: #1e1e1e;
                text-align: center;
                color: #ffffff;
                font-size: 14px;
            }
            QProgressBar::chunk {
                background-color: #19AF39;
                border-radius: 6px;
            }
        """)
        layout.addWidget(self.progress_bar)
        
        # 详细信息
        self.detail_label = QLabel("准备中...")
        self.detail_label.setStyleSheet("font-size: 14px; color: #cccccc; padding: 5px;")
        layout.addWidget(self.detail_label)
        
        # 性能指示器
        self.performance_indicator = PerformanceIndicator()
        layout.addWidget(self.performance_indicator)
        
        # 统计信息
        self.stats_label = QLabel("已处理: 0 | 成功: 0 | 失败: 0 | 剩余时间: --")
        self.stats_label.setStyleSheet("font-size: 12px; color: #999999; padding: 5px;")
        layout.addWidget(self.stats_label)
        
        # 按钮区域
        button_frame = QFrame()
        button_layout = QHBoxLayout(button_frame)
        
        self.cancel_button = QPushButton("取消")
        self.cancel_button.setStyleSheet("""
            QPushButton {
                background-color: #dc3545;
                color: white;
                border: none;
                padding: 10px 20px;
                border-radius: 4px;
                font-size: 14px;
            }
            QPushButton:hover {
                background-color: #c82333;
            }
        """)
        self.cancel_button.clicked.connect(self.cancel_requested.emit)
        
        button_layout.addStretch()
        button_layout.addWidget(self.cancel_button)
        
        layout.addWidget(button_frame)
    
    def update_progress(self, current: int, total: int, current_file: str = ""):
        """更新进度"""
        if total > 0:
            progress = int((current / total) * 100)
            self.progress_bar.setValue(progress)
            
            # 更新主标签
            self.main_label.setText(f"正在处理文件 ({current}/{total})")
            
            # 更新详细信息
            if current_file:
                self.detail_label.setText(f"当前文件: {os.path.basename(current_file)}")
            
            # 计算剩余时间
            elapsed = time.time() - self.start_time
            if current > 0:
                avg_time_per_file = elapsed / current
                remaining_files = total - current
                remaining_time = avg_time_per_file * remaining_files
                
                self.stats_label.setText(
                    f"已处理: {current} | 总计: {total} | "
                    f"剩余时间: {self._format_time(remaining_time)}"
                )
    
    def update_stats(self, processed: int, success: int, failed: int):
        """更新统计信息"""
        elapsed = time.time() - self.start_time
        self.stats_label.setText(
            f"已处理: {processed} | 成功: {success} | 失败: {failed} | "
            f"用时: {self._format_time(elapsed)}"
        )
    
    def _format_time(self, seconds: float) -> str:
        """格式化时间"""
        if seconds < 60:
            return f"{seconds:.0f}秒"
        elif seconds < 3600:
            return f"{seconds // 60:.0f}分{seconds % 60:.0f}秒"
        else:
            hours = seconds // 3600
            minutes = (seconds % 3600) // 60
            return f"{hours:.0f}时{minutes:.0f}分"
    
    def center_on_screen(self):
        """居中显示"""
        from PySide6.QtWidgets import QApplication
        screen = QApplication.primaryScreen()
        screen_geometry = screen.geometry()
        x = (screen_geometry.width() - self.width()) // 2
        y = (screen_geometry.height() - self.height()) // 2
        self.move(x, y)
