"""
🔧 API 调用管理器 - 线程安全的网络请求管理

🔧 2026-02-12 重写：
- 移除全局互斥锁，不再串行化网络请求
- 每个线程使用独立的 requests.Session（通过 threading.local）
- requests 库本身是线程安全的，独立 Session 之间不会冲突
- 4线程并行请求，大幅提升处理速度

核心原则：
- 每个线程独立 Session，无共享状态
- 不需要全局锁，并行请求安全
- 详细记录每一步操作
"""

import threading
import time
import gc
import sys
import os
import traceback
from typing import Optional, Callable, Any, Dict
from logger import logger

# ============================================================================
# 🔧 2026-02-12: 移除全局锁，使用空操作锁保持接口兼容
# ============================================================================
class _NoOpLock:
    """空操作锁 - 保持旧接口兼容但不阻塞"""
    def acquire(self, timeout=None):
        return True
    def release(self):
        pass
    def __enter__(self):
        return self
    def __exit__(self, *args):
        pass

_GLOBAL_NETWORK_LOCK = _NoOpLock()  # 不再阻塞，保持接口兼容

# 🔧 API 调用计数器 - 用于监控和调试
_api_call_count = 0
_api_call_count_lock = threading.Lock()

# 🔧 API 调用超时设置（秒）
API_CALL_TIMEOUT = 30
API_CONNECT_TIMEOUT = 10

# 🔧 调试模式 - 输出详细日志
DEBUG_MODE = True


def _debug_log(msg: str):
    """调试日志输出"""
    if DEBUG_MODE:
        thread_id = threading.current_thread().ident
        logger.info(f"[API-DEBUG][Thread-{thread_id}] {msg}")


def get_api_call_count() -> int:
    """获取 API 调用总次数"""
    with _api_call_count_lock:
        return _api_call_count


def _increment_api_call_count():
    """增加 API 调用计数"""
    global _api_call_count
    with _api_call_count_lock:
        _api_call_count += 1
        return _api_call_count


# ============================================================================
# 🔧 基于 requests 的安全网络请求
# ============================================================================

def _create_requests_session(timeout: float = API_CALL_TIMEOUT):
    """
    🔧 创建独立的 requests Session
    
    每次请求都创建新的 Session，避免连接池共享问题
    """
    import requests
    from requests.adapters import HTTPAdapter
    from urllib3.util.retry import Retry
    
    session = requests.Session()
    
    # 配置重试策略
    retry_strategy = Retry(
        total=2,
        backoff_factor=0.5,
        status_forcelist=[500, 502, 503, 504],
    )
    
    adapter = HTTPAdapter(
        max_retries=retry_strategy,
        pool_connections=1,  # 每个 Session 只用一个连接
        pool_maxsize=1,
        pool_block=False
    )
    
    session.mount("http://", adapter)
    session.mount("https://", adapter)
    
    return session


def safe_requests_post(url: str, headers: dict = None, json_data: dict = None,
                       timeout: float = API_CALL_TIMEOUT) -> dict:
    """
    🔧 安全的 POST 请求（使用全局锁）
    
    Returns:
        dict: {"success": bool, "data": response_json or None, "error": str or None}
    """
    call_id = _increment_api_call_count()
    _debug_log(f"[Call-{call_id}] 准备发送 POST 请求到 {url[:50]}...")
    
    # 🔧 获取全局锁
    _debug_log(f"[Call-{call_id}] 等待获取全局网络锁...")
    lock_acquired = _GLOBAL_NETWORK_LOCK.acquire(timeout=60)
    if not lock_acquired:
        _debug_log(f"[Call-{call_id}] ❌ 获取全局网络锁超时！")
        return {"success": False, "data": None, "error": "获取网络锁超时"}
    
    _debug_log(f"[Call-{call_id}] ✅ 已获取全局网络锁")
    
    session = None
    try:
        import requests
        
        _debug_log(f"[Call-{call_id}] 创建 requests Session...")
        session = _create_requests_session(timeout)
        
        _debug_log(f"[Call-{call_id}] 发送请求...")
        response = session.post(
            url,
            headers=headers,
            json=json_data,
            timeout=(API_CONNECT_TIMEOUT, timeout)
        )
        
        _debug_log(f"[Call-{call_id}] 收到响应，状态码: {response.status_code}")
        
        if response.status_code == 200:
            data = response.json()
            _debug_log(f"[Call-{call_id}] ✅ 请求成功")
            return {"success": True, "data": data, "error": None}
        else:
            error_msg = f"HTTP {response.status_code}: {response.text[:200]}"
            _debug_log(f"[Call-{call_id}] ❌ 请求失败: {error_msg}")
            return {"success": False, "data": None, "error": error_msg}
    except Exception as e:
        error_msg = f"{type(e).__name__}: {str(e)}"
        _debug_log(f"[Call-{call_id}] ❌ 请求异常: {error_msg}")
        logger.error(f"[Call-{call_id}] 详细异常:\n{traceback.format_exc()}")
        return {"success": False, "data": None, "error": error_msg}
    finally:
        # 🔧 清理资源
        if session:
            try:
                session.close()
            except Exception:
                pass
        
        # 🔧 释放全局锁
        try:
            _GLOBAL_NETWORK_LOCK.release()
            _debug_log(f"[Call-{call_id}] 🔓 已释放全局网络锁")
        except RuntimeError:
            pass
        
        # 🔧 垃圾回收
        gc.collect()


# ============================================================================
# 🔧 安全的 OpenAI API 调用（使用 requests 直接调用，不用 SDK）
# ============================================================================

def safe_openai_chat_completion(api_key: str, api_base: str, model: str,
                                messages: list, max_tokens: int = 200,
                                timeout: float = API_CALL_TIMEOUT) -> str:
    """
    🔧 安全的 OpenAI 聊天完成调用（直接用 requests，不用 SDK）
    
    这是最稳定的方案，完全避免 httpx 的问题
    """
    call_id = _increment_api_call_count()
    _debug_log(f"[Chat-{call_id}] 开始 OpenAI Chat 调用，模型: {model}")
    
    # 🤖 内置本地模型检查：builtin:// 协议不能走 HTTP 请求
    if api_base and api_base.startswith('builtin://'):
        _debug_log(f"[Chat-{call_id}] 🤖 检测到内置本地模型 ({api_base})，使用本地推理")
        try:
            from lib.local_model import get_local_model
            local_model = get_local_model()
            if not local_model.is_loaded():
                _debug_log(f"[Chat-{call_id}] ❌ 内置本地模型未加载")
                return ""
            
            # 将 OpenAI 消息格式转换为纯文本提示
            prompt_parts = []
            for msg in messages:
                role = msg.get('role', 'user')
                content = msg.get('content', '')
                if isinstance(content, list):
                    # 多模态消息（图文混合），只提取文本部分
                    text_parts = [item.get('text', '') for item in content if isinstance(item, dict) and item.get('type') == 'text']
                    content = ' '.join(text_parts)
                if content:
                    prompt_parts.append(content)
            
            prompt = "\n".join(prompt_parts)
            
            result = local_model.generate(prompt, max_tokens=max_tokens)
            if result:
                _debug_log(f"[Chat-{call_id}] ✅ 内置本地模型调用成功，返回 {len(result)} 字符")
                return result
            else:
                _debug_log(f"[Chat-{call_id}] ⚠️ 内置本地模型返回空结果")
                return ""
        except Exception as e:
            _debug_log(f"[Chat-{call_id}] ❌ 内置本地模型调用失败: {str(e)}")
            logger.error(f"[Chat-{call_id}] 内置本地模型异常:\n{traceback.format_exc()}")
            return ""
    
    # 构建请求 URL
    url = f"{api_base.rstrip('/')}/chat/completions"
    
    headers = {
        "Authorization": f"Bearer {api_key}",
        "Content-Type": "application/json"
    }
    
    json_data = {
        "model": model,
        "messages": messages,
        "max_tokens": max_tokens,
        "temperature": 0.1
    }
    
    result = safe_requests_post(url, headers=headers, json_data=json_data, timeout=timeout)
    
    if result["success"] and result["data"]:
        try:
            content = result["data"]["choices"][0]["message"]["content"]
            _debug_log(f"[Chat-{call_id}] ✅ 调用成功，返回 {len(content)} 字符")
            return content
        except (KeyError, IndexError) as e:
            _debug_log(f"[Chat-{call_id}] ❌ 解析响应失败: {str(e)}")
            return ""
    else:
        _debug_log(f"[Chat-{call_id}] ❌ 调用失败: {result['error']}")
        return ""


# ============================================================================
# 🔧 安全的图片识别（豆包等在线模型）
# ============================================================================

def safe_online_model_recognize(image_path: str, api_key: str, api_base: str,
                                model_name: str, max_chars: int = 5000) -> str:
    """
    🔧 安全的在线模型图片识别
    
    使用 requests 直接调用 API，完全避免 httpx 的多线程问题
    """
    import base64
    
    call_id = _increment_api_call_count()
    _debug_log(f"[Image-{call_id}] 开始图片识别: {os.path.basename(image_path)}")
    _debug_log(f"[Image-{call_id}] 模型: {model_name}, API: {api_base[:30]}...")
    
    try:
        # 🔧 压缩图片（在锁外面进行，减少锁持有时间）
        _debug_log(f"[Image-{call_id}] 开始压缩图片...")
        
        from lib.ai import compress_image_for_api
        compressed_image_data = compress_image_for_api(image_path, max_size=1024, quality=70)
        base64_image = base64.b64encode(compressed_image_data).decode('utf-8')
        
        _debug_log(f"[Image-{call_id}] 图片压缩完成，Base64 大小: {len(base64_image)/1024:.1f}KB")
        
        # 构建消息
        messages = [
            {
                "role": "user",
                "content": [
                    {
                        "type": "text",
                        "text": "简要描述图片：1.文字内容 2.类型 3.主题（限50字）"
                    },
                    {
                        "type": "image_url",
                        "image_url": {
                            "url": f"data:image/jpeg;base64,{base64_image}"
                        }
                    }
                ]
            }
        ]
        
        # 🔧 使用安全的 API 调用
        result = safe_openai_chat_completion(
            api_key=api_key,
            api_base=api_base,
            model=model_name,
            messages=messages,
            max_tokens=200,
            timeout=API_CALL_TIMEOUT
        )
        
        if result:
            _debug_log(f"[Image-{call_id}] ✅ 图片识别成功，返回 {len(result)} 字符")
            return result[:max_chars]
        else:
            _debug_log(f"[Image-{call_id}] ❌ 图片识别返回空结果")
            return ""
    except MemoryError as me:
        _debug_log(f"[Image-{call_id}] ❌ 内存不足: {str(me)}")
        gc.collect()
        return ""
    except Exception as e:
        _debug_log(f"[Image-{call_id}] ❌ 图片识别失败: {str(e)}")
        logger.error(f"[Image-{call_id}] 详细异常:\n{traceback.format_exc()}")
        return ""


# ============================================================================
# 🔧 安全的 Moonshot API 调用
# ============================================================================

def safe_moonshot_recognize(image_path: str, api_key: str, max_chars: int = 5000) -> str:
    """
    🔧 安全的 Moonshot 图片识别
    
    使用 requests 直接调用 Moonshot API
    """
    import json
    
    call_id = _increment_api_call_count()
    _debug_log(f"[Moonshot-{call_id}] 开始 Moonshot 图片识别: {os.path.basename(image_path)}")
    
    api_base = "https://api.moonshot.cn/v1"
    
    try:
        # ==============================
        # 步骤1: 上传文件
        # ==============================
        _debug_log(f"[Moonshot-{call_id}] 步骤1: 上传文件...")
        
        # 🔧 获取全局锁
        lock_acquired = _GLOBAL_NETWORK_LOCK.acquire(timeout=60)
        if not lock_acquired:
            _debug_log(f"[Moonshot-{call_id}] ❌ 获取网络锁超时（上传）")
            return ""
        
        file_id = None
        try:
            import requests
            
            session = _create_requests_session(API_CALL_TIMEOUT)
            
            with open(image_path, 'rb') as f:
                files = {'file': (os.path.basename(image_path), f)}
                headers = {"Authorization": f"Bearer {api_key}"}
                
                response = session.post(
                    f"{api_base}/files",
                    headers=headers,
                    files=files,
                    data={"purpose": "file-extract"},
                    timeout=(API_CONNECT_TIMEOUT, API_CALL_TIMEOUT)
                )
            
            if response.status_code != 200:
                _debug_log(f"[Moonshot-{call_id}] ❌ 文件上传失败: {response.status_code}")
                return ""
            
            file_data = response.json()
            file_id = file_data.get("id")
            _debug_log(f"[Moonshot-{call_id}] ✅ 文件上传成功，ID: {file_id}")
            
            try:
                session.close()
            except Exception:
                pass
        finally:
            try:
                _GLOBAL_NETWORK_LOCK.release()
            except RuntimeError:
                pass
            _debug_log(f"[Moonshot-{call_id}] 🔓 释放网络锁（上传完成）")
        
        if not file_id:
            return ""
        
        # ==============================
        # 步骤2: 获取文件内容
        # ==============================
        _debug_log(f"[Moonshot-{call_id}] 步骤2: 获取文件内容...")
        
        lock_acquired = _GLOBAL_NETWORK_LOCK.acquire(timeout=60)
        if not lock_acquired:
            _debug_log(f"[Moonshot-{call_id}] ❌ 获取网络锁超时（获取内容）")
            return ""
        
        extracted_text = ""
        try:
            import requests
            
            session = _create_requests_session(API_CALL_TIMEOUT)
            headers = {"Authorization": f"Bearer {api_key}"}
            
            response = session.get(
                f"{api_base}/files/{file_id}/content",
                headers=headers,
                timeout=(API_CONNECT_TIMEOUT, API_CALL_TIMEOUT)
            )
            
            if response.status_code == 200:
                content_text = response.text
                _debug_log(f"[Moonshot-{call_id}] ✅ 获取内容成功，长度: {len(content_text)}")
                
                # 尝试解析 JSON
                try:
                    content_data = json.loads(content_text)
                    extracted_text = content_data.get("content", "")
                except Exception:
                    extracted_text = content_text.strip()
            
            try:
                session.close()
            except Exception:
                pass
        finally:
            try:
                _GLOBAL_NETWORK_LOCK.release()
            except RuntimeError:
                pass
            _debug_log(f"[Moonshot-{call_id}] 🔓 释放网络锁（获取内容完成）")
        
        # 如果提取到了内容，直接返回
        if extracted_text and len(extracted_text.strip()) > 5:
            return extracted_text[:max_chars]
        
        # ==============================
        # 步骤3: 使用对话模式（备用）
        # ==============================
        _debug_log(f"[Moonshot-{call_id}] 步骤3: 使用对话模式...")
        
        messages = [
            {
                "role": "user",
                "content": f"图片文件 {os.path.basename(image_path)} 无法直接提取文字。请简要描述：1.可能的图片类型 2.建议的分类。限50字内。"
            }
        ]
        
        description = safe_openai_chat_completion(
            api_key=api_key,
            api_base=api_base,
            model="moonshot-v1-8k",
            messages=messages,
            max_tokens=100,
            timeout=API_CALL_TIMEOUT
        )
        
        return description[:max_chars] if description else ""
    
    except Exception as e:
        _debug_log(f"[Moonshot-{call_id}] ❌ Moonshot 识别失败: {str(e)}")
        logger.error(f"[Moonshot-{call_id}] 详细异常:\n{traceback.format_exc()}")
        return ""


# ============================================================================
# 🔧 为 LangChain 提供的安全 HTTP 客户端
# ============================================================================

def create_safe_langchain_http_client(timeout: float = API_CALL_TIMEOUT):
    """
    🔧 为 LangChain 创建安全的 HTTP 客户端
    
    注意：这仍然使用 httpx（因为 LangChain 需要），但配置更保守
    """
    import httpx
    
    _debug_log("创建 LangChain 用的 httpx 客户端...")
    
    # 🔧 创建最保守的配置
    transport = httpx.HTTPTransport(
        retries=0,
        http2=False,  # 禁用 HTTP/2
    )
    
    return httpx.Client(
        transport=transport,
        timeout=httpx.Timeout(timeout, connect=API_CONNECT_TIMEOUT),
        limits=httpx.Limits(
            max_connections=1,
            max_keepalive_connections=0
        )
    )


# ============================================================================
# 🔧 兼容旧接口
# ============================================================================

class SafeOpenAIClient:
    """
    🔧 安全的 OpenAI 客户端包装器（兼容旧接口）
    """
    
    def __init__(self, api_key: str, api_base: str, timeout: float = API_CALL_TIMEOUT):
        self.api_key = api_key
        self.api_base = api_base
        self.timeout = timeout
    
    def chat_completion(self, messages: list, model: str, max_tokens: int = 200, **kwargs) -> str:
        """安全的聊天完成调用"""
        return safe_openai_chat_completion(
            api_key=self.api_key,
            api_base=self.api_base,
            model=model,
            messages=messages,
            max_tokens=max_tokens,
            timeout=self.timeout
        )


def safe_api_call(func: Callable, *args, timeout: float = API_CALL_TIMEOUT, **kwargs) -> Any:
    """
    🔧 安全的 API 调用包装器（兼容旧接口）
    
    使用全局锁确保同一时间只有一个调用
    """
    call_id = _increment_api_call_count()
    _debug_log(f"[SafeCall-{call_id}] 开始安全 API 调用...")
    
    lock_acquired = _GLOBAL_NETWORK_LOCK.acquire(timeout=timeout + 30)
    if not lock_acquired:
        _debug_log(f"[SafeCall-{call_id}] ❌ 获取全局锁超时")
        raise TimeoutError("等待 API 锁超时")
    
    try:
        _debug_log(f"[SafeCall-{call_id}] ✅ 开始执行函数...")
        result = func(*args, **kwargs)
        _debug_log(f"[SafeCall-{call_id}] ✅ 函数执行完成")
        return result
    except Exception as e:
        _debug_log(f"[SafeCall-{call_id}] ❌ 函数执行失败: {str(e)}")
        raise
    finally:
        try:
            _GLOBAL_NETWORK_LOCK.release()
        except RuntimeError:
            pass
        _debug_log(f"[SafeCall-{call_id}] 🔓 释放全局锁")
        gc.collect()


def create_safe_openai_client(api_key: str, api_base: str, timeout: float = API_CALL_TIMEOUT):
    """
    🔧 创建安全的 OpenAI 客户端（兼容旧接口）
    
    注意：为了兼容性仍返回 OpenAI SDK 客户端，但配置更保守
    """
    import httpx
    from openai import OpenAI
    
    _debug_log(f"创建 OpenAI 客户端: {api_base[:30]}...")
    
    transport = httpx.HTTPTransport(
        retries=0,
        http2=False,
    )
    
    http_client = httpx.Client(
        transport=transport,
        timeout=httpx.Timeout(timeout, connect=API_CONNECT_TIMEOUT),
        limits=httpx.Limits(
            max_connections=1,
            max_keepalive_connections=0
        )
    )
    
    return OpenAI(
        base_url=api_base,
        api_key=api_key,
        timeout=httpx.Timeout(timeout, connect=API_CONNECT_TIMEOUT),
        max_retries=1,
        http_client=http_client
    )


# ============================================================================
# 🔧 导出
# ============================================================================

__all__ = [
    'safe_api_call',
    'safe_requests_post',
    'safe_openai_chat_completion',
    'safe_online_model_recognize',
    'safe_moonshot_recognize',
    'create_safe_openai_client',
    'create_safe_langchain_http_client',
    'SafeOpenAIClient',
    'get_api_call_count',
    'API_CALL_TIMEOUT',
    'API_CONNECT_TIMEOUT',
    '_GLOBAL_NETWORK_LOCK',  # 导出全局锁，让其他模块也能使用
]
