# -*- coding: utf-8 -*-
"""
离线License验证模块 - 简化版
只验证 local_unlock 权限和到期时间
"""
import os
import sys
import json
import base64
import hashlib
from datetime import datetime, timedelta
from Crypto.Cipher import AES
from logger import logger
from lib.common import Common


class LicenseManager:
    """License管理器"""
    
    def __init__(self):
        self.license_file_path = self._get_license_path()
        self._cached_license = None
        self._cache_time = None
        self._cache_duration = 3600  # 缓存1小时
    
    def _get_license_path(self):
        """获取License文件路径（支持带机器码的文件名）"""
        # 获取当前机器码
        machine_code = Common.get_machine_code()
        
        # 1. 优先在软件同级目录查找（可执行文件所在目录）
        if getattr(sys, 'frozen', False):
            # 打包后的环境：可执行文件所在目录
            app_dir = os.path.dirname(sys.executable)
        else:
            # 开发环境：项目根目录
            app_dir = os.path.dirname(os.path.dirname(os.path.abspath(__file__)))
        
        # 优先查找带机器码的文件名（如 license_92BB5DF5.dat）
        machine_specific_file = os.path.join(app_dir, f'license_{machine_code}.dat')
        if os.path.exists(machine_specific_file):
            logger.info(f"✅ 找到License文件: {machine_specific_file}")
            return machine_specific_file
        
        # 查找固定文件名
        fixed_name_file = os.path.join(app_dir, 'license.dat')
        if os.path.exists(fixed_name_file):
            logger.info(f"✅ 找到License文件: {fixed_name_file}")
            return fixed_name_file
        
        # 2. 兼容旧版本：在 AppData 目录查找
        config_path = Common.getConfigPath()
        
        # 查找带机器码的文件名
        machine_specific_file_old = os.path.join(config_path, f'license_{machine_code}.dat')
        if os.path.exists(machine_specific_file_old):
            logger.info(f"✅ 找到License文件(旧位置): {machine_specific_file_old}")
            return machine_specific_file_old
        
        # 查找固定文件名
        fixed_name_file_old = os.path.join(config_path, 'license.dat')
        if os.path.exists(fixed_name_file_old):
            logger.info(f"✅ 找到License文件(旧位置): {fixed_name_file_old}")
            return fixed_name_file_old
        
        # 默认返回软件同级目录的路径（用于生成新License）
        logger.warning(f"⚠️ 未找到License文件，默认路径: {machine_specific_file}")
        return machine_specific_file
    
    def _get_keys(self):
        """从api.py获取密钥"""
        from lib.api import LICENSE_AES_KEY, LICENSE_AES_IV
        return LICENSE_AES_KEY, LICENSE_AES_IV
    
    def _encrypt_license(self, data: dict) -> str:
        """加密License数据"""
        try:
            LICENSE_AES_KEY, LICENSE_AES_IV = self._get_keys()
            
            # 转为JSON字符串并编码为UTF-8字节
            json_str = json.dumps(data, ensure_ascii=False)
            json_bytes = json_str.encode('utf-8')
            
            # PKCS7填充（字节级别）
            block_size = AES.block_size
            padding = block_size - len(json_bytes) % block_size
            padded_data = json_bytes + bytes([padding] * padding)
            
            # AES加密
            key = LICENSE_AES_KEY.encode('utf-8')[:32].ljust(32, b'\0')
            iv = LICENSE_AES_IV.encode('utf-8')[:16].ljust(16, b'\0')
            cipher = AES.new(key, AES.MODE_CBC, iv)
            encrypted = cipher.encrypt(padded_data)
            
            # Base64编码
            return base64.b64encode(encrypted).decode('utf-8')
        except Exception as e:
            logger.error(f"License加密失败: {str(e)}")
            return None
    
    def _decrypt_license(self, encrypted_data: str) -> dict:
        """解密License数据"""
        try:
            LICENSE_AES_KEY, LICENSE_AES_IV = self._get_keys()
            
            # Base64解码
            encrypted_bytes = base64.b64decode(encrypted_data)
            
            # AES解密
            key = LICENSE_AES_KEY.encode('utf-8')[:32].ljust(32, b'\0')
            iv = LICENSE_AES_IV.encode('utf-8')[:16].ljust(16, b'\0')
            cipher = AES.new(key, AES.MODE_CBC, iv)
            decrypted_bytes = cipher.decrypt(encrypted_bytes)
            
            # 移除PKCS7填充（字节级别）
            padding = decrypted_bytes[-1]
            json_bytes = decrypted_bytes[:-padding]
            
            # 解码为字符串
            json_str = json_bytes.decode('utf-8')
            
            # 解析JSON
            return json.loads(json_str)
        except Exception as e:
            logger.error(f"License解密失败: {str(e)}")
            return None
    
    def _calculate_signature(self, data: dict) -> str:
        """计算License签名（防篡改）"""
        # 按key排序后连接成字符串
        sorted_keys = sorted([k for k in data.keys() if k != 'signature'])
        content = ''.join([f"{k}:{data[k]}" for k in sorted_keys])
        
        # 加盐后计算SHA256
        salt = "FileNeatAI_Offline_Salt_2024"
        signature_content = content + salt
        return hashlib.sha256(signature_content.encode('utf-8')).hexdigest()
    
    def generate_license(self, machine_code: str, local_unlock: int, 
                        duration_days: int = 365, note: str = '') -> bool:
        """
        生成License文件
        
        Args:
            machine_code: 机器码（必须）
            local_unlock: 本地解锁权限 (0=无权限, 1=有权限)
            duration_days: 有效期天数（-1表示永久）
            note: 备注信息（可选，如客户名称等）
        """
        try:
            # 计算到期时间
            issue_time = datetime.now()
            if duration_days == -1:
                expire_time = datetime(2099, 12, 31, 23, 59, 59)
            else:
                expire_time = issue_time + timedelta(days=duration_days)
            
            # License数据（简化版，只包含核心字段）
            license_data = {
                'machine_code': machine_code,
                'local_unlock': local_unlock,  # 0或1
                'issue_time': issue_time.strftime('%Y-%m-%d %H:%M:%S'),
                'expire_time': expire_time.strftime('%Y-%m-%d %H:%M:%S'),
                'note': note,
                'version': '3.0'
            }
            
            # 计算签名（不包含signature字段本身）
            license_data['signature'] = self._calculate_signature(license_data)
            
            # 加密并保存
            encrypted = self._encrypt_license(license_data)
            if not encrypted:
                return False
            
            with open(self.license_file_path, 'w', encoding='utf-8') as f:
                f.write(encrypted)
            
            logger.info(f"License生成成功: local_unlock={local_unlock}, 到期时间={expire_time}")
            return True
            
        except Exception as e:
            logger.error(f"License生成失败: {str(e)}")
            return False
    
    def validate_license(self) -> tuple:
        """
        验证License
        
        Returns:
            (is_valid, error_message, license_info)
        """
        try:
            # 检查缓存
            if self._cached_license and self._cache_time:
                if (datetime.now() - self._cache_time).total_seconds() < self._cache_duration:
                    return True, "缓存验证通过", self._cached_license
            
            # 检查License文件是否存在
            if not os.path.exists(self.license_file_path):
                return False, "未找到离线License，请联系供应商获取授权", None
            
            # 读取并解密License
            with open(self.license_file_path, 'r', encoding='utf-8') as f:
                encrypted_data = f.read()
            
            license_data = self._decrypt_license(encrypted_data)
            if not license_data:
                return False, "License文件已损坏或无效", None
            
            # 验证签名
            saved_signature = license_data.pop('signature', '')
            calculated_signature = self._calculate_signature(license_data)
            if saved_signature != calculated_signature:
                return False, "License文件已被篡改，验证失败", None
            
            # 验证机器码
            current_machine_code = Common.get_machine_code()
            if license_data['machine_code'] != current_machine_code:
                return False, f"License与当前机器不匹配\n当前机器码: {current_machine_code}\nLicense机器码: {license_data['machine_code']}", None
            
            # 验证到期时间
            expire_time = datetime.strptime(license_data['expire_time'], '%Y-%m-%d %H:%M:%S')
            current_time = datetime.now()
            
            if current_time > expire_time:
                days_expired = (current_time - expire_time).days
                return False, f"License已过期{days_expired}天，请联系供应商续费", None
            
            # 计算剩余天数
            days_remaining = (expire_time - current_time).days
            
            # 获取在线模型限制（从License读取，如果没有则使用默认值）
            online_limit = license_data.get('online_limit', 35000)
            
            # 组装返回信息 - 构造完整的machine_data格式
            license_info = {
                'host': {
                    'id': 0,
                    'machine_code': license_data['machine_code'],
                    'filenums': online_limit,  # 使用License中的在线模型限制
                    'is_vip': 1 if license_data['local_unlock'] == 1 else 0,
                    'local_unlock': license_data['local_unlock'],
                    'osinfo': '',
                    'ip': '127.0.0.1',  # 离线模式
                    'c_time': license_data.get('issue_time', ''),
                    'expire_time': license_data['expire_time'],  # 添加到期时间
                },
                'version': {
                    'id': 0,
                    'version': license_data.get('version', '3.0'),
                    'force': 0,
                    'win_exe': '',
                    'mac_intel': '',
                    'mac_arm': '',
                    'ubuntu_linux': '',
                    'c_time': ''
                },
                'setting': {
                    'descripe_cn': '专业的跨平台AI文件整理工具---文件禅你的文件，AI来管理',
                    'descripe_en': 'Professional cross-platform AI file organization tool---FileNeatAI, your files, managed by AI',
                    'buy_link': 'https://fileneatai.com/pricing',
                    'trial_file_count': str(online_limit),  # 使用License中的限制
                    'vip_file_count': str(online_limit),    # 使用License中的限制
                    'website': 'https://fileneatai.com',
                    'guideurl': 'https://j42t9z347d.feishu.cn/docx/Tg7od5DpwoUhlPxMwbGcdIjXn5b',
                    'enguideurl': 'https://baidu.com'
                },
                # 添加额外信息供UI使用
                '_license_info': {
                    'issue_time': license_data.get('issue_time', ''),
                    'days_remaining': days_remaining,
                    'note': license_data.get('note', ''),
                    'online_limit': online_limit,  # 添加在线模型限制信息
                }
            }
            
            # 更新缓存
            self._cached_license = license_info
            self._cache_time = datetime.now()
            
            # 到期提醒
            if days_remaining <= 30:
                logger.warning(f"⚠️ License即将在{days_remaining}天后到期，请及时续费")
            
            return True, "验证通过", license_info
            
        except Exception as e:
            logger.error(f"License验证异常: {str(e)}")
            return False, f"License验证异常: {str(e)}", None
    
    def get_license_info(self) -> dict:
        """获取License信息（用于UI显示）"""
        is_valid, message, info = self.validate_license()
        
        if is_valid:
            return info
        else:
            # 返回默认信息（无权限）- 完整的machine_data格式
            machine_code = Common.get_machine_code()
            return {
                'host': {
                    'id': 0,
                    'machine_code': machine_code if machine_code else 'UNKNOWN',
                    'filenums': 100,  # 默认试用数量
                    'is_vip': 0,
                    'local_unlock': 0,
                    'osinfo': '',
                    'ip': '',
                    'c_time': '',
                    'expire_time': '未授权',
                },
                'version': {
                    'id': 0,
                    'version': '3.0',
                    'force': 0,
                    'win_exe': '',
                    'mac_intel': '',
                    'mac_arm': '',
                    'ubuntu_linux': '',
                    'c_time': ''
                },
                'setting': {
                    'descripe_cn': '专业的跨平台AI文件整理工具---文件禅你的文件，AI来管理',
                    'descripe_en': 'Professional cross-platform AI file organization tool---FileNeatAI, your files, managed by AI',
                    'buy_link': 'https://fileneatai.com/pricing',
                    'trial_file_count': '100',
                    'vip_file_count': '100',
                    'website': 'https://fileneatai.com',
                    'guideurl': 'https://j42t9z347d.feishu.cn/docx/Tg7od5DpwoUhlPxMwbGcdIjXn5b',
                    'enguideurl': 'https://baidu.com'
                },
                '_license_info': {
                    'issue_time': '',
                    'days_remaining': 0,
                    'note': '',
                    'error': message
                }
            }
    
    def revoke_license(self) -> bool:
        """撤销License（删除License文件）"""
        try:
            if os.path.exists(self.license_file_path):
                os.remove(self.license_file_path)
                self._cached_license = None
                self._cache_time = None
                logger.info("License已撤销")
                return True
            return False
        except Exception as e:
            logger.error(f"撤销License失败: {str(e)}")
            return False


# 全局License管理器实例
license_manager = LicenseManager()

