# -*- coding: utf-8 -*-
"""
多模态AI服务集成模块 - 支持多种在线AI图片识别服务

🔧 2026-01-22 修复：
- 添加 VIP 权限检查缓存，避免重复网络请求
- 添加线程安全保护，防止多线程并发导致的崩溃
- 添加网络请求超时和异常保护
"""

import os
import json
import time
import threading
from typing import Dict, List, Optional, Tuple, Union, Callable
from pathlib import Path
from logger import logger

# 🔧 线程安全：VIP 权限检查锁和缓存
_VIP_CHECK_LOCK = threading.Lock()
_VIP_CACHE = {
    'is_vip': None,           # 缓存的 VIP 状态
    'last_check_time': 0,     # 上次检查时间
    'cache_duration': 300,    # 缓存有效期（秒），5分钟
    'check_in_progress': False  # 是否正在检查中
}

# 🔧 网络请求超时设置
NETWORK_TIMEOUT = 15  # 网络请求超时（秒）
NETWORK_CONNECT_TIMEOUT = 5  # 连接超时（秒）

# 🔧 网络请求信号量 - 限制并发网络请求数量
_NETWORK_SEMAPHORE = threading.BoundedSemaphore(2)  # 最多同时 2 个网络请求


def _safe_requests_post(url: str, headers: dict = None, json_data: dict = None, 
                        timeout: int = NETWORK_TIMEOUT) -> Optional[dict]:
    """
    🔧 线程安全的 POST 请求
    
    - 使用信号量限制并发
    - 自动处理超时和异常
    - 返回 None 表示失败
    """
    acquired = _NETWORK_SEMAPHORE.acquire(timeout=timeout + 10)
    if not acquired:
        logger.warning("⚠️ 等待网络请求信号量超时")
        return None
    
    try:
        response = requests.post(
            url,
            headers=headers,
            json=json_data,
            timeout=(NETWORK_CONNECT_TIMEOUT, timeout)
        )
        if response.status_code == 200:
            return response.json()
        else:
            logger.warning(f"⚠️ HTTP 请求失败: {response.status_code}")
            return {'error': response.status_code, 'text': response.text}
    except requests.exceptions.Timeout:
        logger.warning(f"⚠️ 网络请求超时（{timeout}秒）")
        return None
    except requests.exceptions.ConnectionError as e:
        logger.warning(f"⚠️ 网络连接错误: {str(e)}")
        return None
    except Exception as e:
        logger.error(f"❌ 网络请求异常: {str(e)}")
        return None
    finally:
        _NETWORK_SEMAPHORE.release()


# 可选依赖导入
try:
    import requests
    HAS_REQUESTS = True
except ImportError:
    HAS_REQUESTS = False
    logger.warning("requests 未安装，多模态AI功能将受限")

try:
    import base64
    HAS_BASE64 = True
except ImportError:
    HAS_BASE64 = False

try:
    from PIL import Image
    HAS_PIL = True
except ImportError:
    HAS_PIL = False
    logger.warning("Pillow 未安装，图片处理功能将受限")


def vip_required(func):
    """VIP权限检查装饰器 - 使用缓存的 VIP 状态"""
    def wrapper(self, *args, **kwargs):
        if not self._check_vip_permission_cached():
            return False, "多模态AI功能仅限VIP用户使用，请升级为VIP用户", 0.0
        return func(self, *args, **kwargs)
    return wrapper


class MultimodalConfig:
    """多模态AI配置类"""
    
    def __init__(self):
        # 服务商配置
        self.providers = {
            'claude': {
                'name': 'Claude 3.5 Sonnet',
                'api_base': 'https://api.anthropic.com',
                'model': 'claude-3-5-sonnet-20241022',
                'max_image_size': 20 * 1024 * 1024,  # 20MB
                'supported_formats': ['.jpg', '.jpeg', '.png', '.gif', '.webp'],
                'cost_per_1k_tokens': 0.003,
                'cost_per_image': 0.015,  # 估算
                'quality': 'excellent',
                'speed': 'fast'
            },
            'gpt4_vision': {
                'name': 'GPT-4 Vision',
                'api_base': 'https://api.openai.com/v1',
                'model': 'gpt-4-vision-preview',
                'max_image_size': 20 * 1024 * 1024,
                'supported_formats': ['.jpg', '.jpeg', '.png', '.gif', '.webp'],
                'cost_per_1k_tokens': 0.01,
                'cost_per_image': 0.025,  # 估算
                'quality': 'excellent',
                'speed': 'medium'
            },
            'gemini_pro': {
                'name': 'Gemini Pro Vision',
                'api_base': 'https://generativelanguage.googleapis.com/v1beta',
                'model': 'gemini-pro-vision',
                'max_image_size': 20 * 1024 * 1024,
                'supported_formats': ['.jpg', '.jpeg', '.png', '.gif', '.webp'],
                'cost_per_1k_tokens': 0.002,
                'cost_per_image': 0.01,  # 估算
                'quality': 'good',
                'speed': 'fast'
            },
            'wenxin': {
                'name': '文心一言4.0多模态',
                'api_base': 'https://aip.baidubce.com/rpc/2.0/ai_custom/v1/wenxinworkshop',
                'model': 'ERNIE-Bot-4',
                'max_image_size': 10 * 1024 * 1024,
                'supported_formats': ['.jpg', '.jpeg', '.png', '.bmp'],
                'cost_per_1k_tokens': 0.001,  # 相对便宜
                'cost_per_image': 0.008,
                'quality': 'good',
                'speed': 'fast'
            },
            'doubao': {
                'name': '火山引擎豆包多模态',
                'api_base': 'https://ark.cn-beijing.volces.com/api/v3/chat/completions',
                'model': 'doubao-1-5-thinking-vision-pro-250428',
                'max_image_size': 20 * 1024 * 1024,
                'supported_formats': ['.jpg', '.jpeg', '.png', '.bmp', '.gif', '.webp'],
                'cost_per_1k_tokens': 0.0005,  # 估计价格
                'cost_per_image': 0.005,
                'quality': 'excellent',
                'speed': 'fast'
            }
        }
        
        # 默认配置
        self.default_provider = 'claude'
        self.fallback_providers = ['gpt4_vision', 'gemini_pro']
        self.max_retries = 3
        self.timeout = 30
        self.quality_threshold = 0.7


class CostController:
    """成本控制器"""
    
    def __init__(self, budget_limit: float = 50.0):
        self.budget_limit = budget_limit  # 月度预算限制（美元）
        self.current_usage = 0.0
        self.usage_history = []
        self.cost_alerts = []
        
    def estimate_cost(self, provider: str, image_count: int, config: MultimodalConfig) -> float:
        """估算处理成本"""
        provider_config = config.providers.get(provider, {})
        cost_per_image = provider_config.get('cost_per_image', 0.02)
        return image_count * cost_per_image
    
    def check_budget(self, estimated_cost: float) -> Tuple[bool, str]:
        """检查预算是否足够"""
        if self.current_usage + estimated_cost > self.budget_limit:
            remaining = self.budget_limit - self.current_usage
            return False, f"预算不足。剩余预算: ${remaining:.2f}, 需要: ${estimated_cost:.2f}"
        return True, ""
    
    def record_usage(self, provider: str, cost: float):
        """记录使用情况"""
        self.current_usage += cost
        self.usage_history.append({
            'timestamp': time.time(),
            'provider': provider,
            'cost': cost
        })
        
        # 检查预算警告
        usage_rate = self.current_usage / self.budget_limit
        if usage_rate >= 0.8 and '80%' not in [alert['type'] for alert in self.cost_alerts]:
            self.cost_alerts.append({
                'type': '80%',
                'message': f"预算使用已达到80%（${self.current_usage:.2f}/${self.budget_limit:.2f}）",
                'timestamp': time.time()
            })
        elif usage_rate >= 0.9 and '90%' not in [alert['type'] for alert in self.cost_alerts]:
            self.cost_alerts.append({
                'type': '90%',
                'message': f"预算使用已达到90%（${self.current_usage:.2f}/${self.budget_limit:.2f}）",
                'timestamp': time.time()
            })


class ImageProcessor:
    """图片处理器"""
    
    @staticmethod
    def optimize_image_for_ai(image_path: str, max_size: int = 2048) -> str:
        """优化图片以适合AI处理"""
        if not HAS_PIL:
            logger.warning("PIL未安装，跳过图片优化")
            return image_path
            
        try:
            import tempfile
            import uuid
            
            with Image.open(image_path) as img:
                # 转换为RGB模式
                if img.mode not in ('RGB', 'L'):
                    img = img.convert('RGB')
                
                # 调整尺寸
                if max(img.size) > max_size:
                    ratio = max_size / max(img.size)
                    new_size = tuple(int(dim * ratio) for dim in img.size)
                    img = img.resize(new_size, Image.Resampling.LANCZOS)
                
                # 保存优化后的图片
                temp_path = os.path.join(
                    tempfile.gettempdir(), 
                    f"optimized_{uuid.uuid4().hex[:8]}.jpg"
                )
                img.save(temp_path, 'JPEG', quality=85, optimize=True)
                
                logger.info(f"图片优化完成: {image_path} -> {temp_path}")
                return temp_path
                
        except Exception as e:
            logger.error(f"图片优化失败: {str(e)}")
            return image_path
    
    @staticmethod
    def encode_image_to_base64(image_path: str) -> str:
        """将图片编码为base64"""
        try:
            with open(image_path, "rb") as image_file:
                return base64.b64encode(image_file.read()).decode('utf-8')
        except Exception as e:
            logger.error(f"图片base64编码失败: {str(e)}")
            return ""


class MultimodalAIService:
    """多模态AI服务主类"""
    
    def __init__(self, config: MultimodalConfig = None, cost_controller: CostController = None):
        self.config = config or MultimodalConfig()
        self.cost_controller = cost_controller or CostController()
        self.processor = ImageProcessor()
        
        # API密钥存储
        self.api_keys = {}
        
        # 通用API配置
        self.api_base = ""
        self.model_name = ""
        
    def set_api_key(self, provider: str, api_key: str):
        """设置API密钥"""
        self.api_keys[provider] = api_key
        logger.info(f"已设置 {provider} API密钥")
    
    def reload_config(self):
        """重新加载配置文件"""
        try:
            import os
            import json
            config_file = os.path.join(os.path.expanduser('~'), '.fileneatai', 'multimodal_settings.json')
            if os.path.exists(config_file):
                with open(config_file, 'r', encoding='utf-8') as f:
                    settings = json.load(f)
                
                # 清除现有API密钥
                self.api_keys.clear()
                
                # 加载多模态AI配置
                multimodal_config = settings.get('multimodal', {})
                if multimodal_config.get('enabled', False) and multimodal_config.get('api_key'):
                    # 使用通用的'multimodal'作为provider名称
                    self.set_api_key('multimodal', multimodal_config['api_key'])
                    # 保存API地址和模型名称到服务实例
                    self.api_base = multimodal_config.get('api_base', '')
                    self.model_name = multimodal_config.get('model', '')
                
                # 加载成本控制设置
                budget = settings.get('budget', {})
                if budget.get('monthly_limit'):
                    self.cost_controller.budget_limit = budget['monthly_limit']
                    
                logger.info("多模态AI配置已重新加载")
                return True
        except Exception as e:
            logger.error(f"重新加载多模态AI配置失败: {str(e)}")
            return False
    
    def is_available(self) -> bool:
        """检查多模态AI服务是否可用"""
        logger.info("🔍 多模态AI服务可用性检查开始")
        
        # 🔧 使用缓存的 VIP 权限检查，避免重复网络请求
        if not self._check_vip_permission_cached():
            logger.warning("❌ VIP权限检查失败，多模态AI不可用")
            return False
        logger.info("✅ VIP权限检查通过")
        
        # 检查基本依赖
        if not HAS_REQUESTS or not HAS_BASE64 or not HAS_PIL:
            logger.warning(f"❌ 基本依赖缺失: requests={HAS_REQUESTS}, base64={HAS_BASE64}, PIL={HAS_PIL}")
            return False
        logger.info("✅ 基本依赖检查通过")
        
        # 检查是否有可用的API密钥
        if not self.api_keys:
            logger.warning("❌ 没有配置任何API密钥")
            return False
        logger.info(f"✅ API密钥已配置: {list(self.api_keys.keys())}")
        
        # 🚀 修复：优先检查新的统一配置 'multimodal'
        if 'multimodal' in self.api_keys:
            has_key = bool(self.api_keys.get('multimodal'))
            has_base = bool(self.api_base)
            has_model = bool(self.model_name)
            logger.info(f"📊 multimodal配置检查: API密钥={has_key}, API地址={has_base}, 模型={has_model}")
            logger.info(f"   - API地址: {self.api_base}")
            logger.info(f"   - 模型名称: {self.model_name}")
            
            if has_key and has_base and has_model:
                logger.info("✅ multimodal配置完整，服务可用")
                return True
            else:
                logger.warning(f"❌ multimodal配置不完整")
        
        # 检查是否有至少一个传统提供商可用
        for provider in self.config.providers:
            if provider in self.api_keys:
                logger.info(f"✅ 传统提供商 {provider} 可用")
                return True
        
        logger.warning("❌ 没有可用的多模态AI提供商")
        return False
    
    def _check_vip_permission_cached(self) -> bool:
        """
        🔧 线程安全的 VIP 权限检查（带缓存）
        
        - 使用缓存避免重复的网络请求
        - 缓存有效期为 5 分钟
        - 线程安全，避免多线程同时检查
        """
        global _VIP_CACHE
        
        current_time = time.time()
        
        # 🔧 快速路径：检查缓存是否有效
        with _VIP_CHECK_LOCK:
            if _VIP_CACHE['is_vip'] is not None:
                cache_age = current_time - _VIP_CACHE['last_check_time']
                if cache_age < _VIP_CACHE['cache_duration']:
                    # 缓存有效，直接返回
                    return _VIP_CACHE['is_vip']
            
            # 如果另一个线程正在检查，等待并使用其结果
            if _VIP_CACHE['check_in_progress']:
                logger.debug("🔄 VIP 检查正在进行中，等待结果...")
                # 释放锁，等待一小段时间后重试
        
        # 如果正在检查中，等待并返回缓存结果
        retry_count = 0
        while retry_count < 10:  # 最多等待 5 秒
            with _VIP_CHECK_LOCK:
                if not _VIP_CACHE['check_in_progress']:
                    if _VIP_CACHE['is_vip'] is not None:
                        return _VIP_CACHE['is_vip']
                    break
            time.sleep(0.5)
            retry_count += 1
        
        # 🔧 需要进行实际检查
        with _VIP_CHECK_LOCK:
            # 再次检查缓存（双重检查锁定模式）
            if _VIP_CACHE['is_vip'] is not None:
                cache_age = current_time - _VIP_CACHE['last_check_time']
                if cache_age < _VIP_CACHE['cache_duration']:
                    return _VIP_CACHE['is_vip']
            
            # 标记正在检查
            _VIP_CACHE['check_in_progress'] = True
        
        try:
            # 执行实际的 VIP 检查
            is_vip = self._check_vip_permission()
            
            # 更新缓存
            with _VIP_CHECK_LOCK:
                _VIP_CACHE['is_vip'] = is_vip
                _VIP_CACHE['last_check_time'] = time.time()
                _VIP_CACHE['check_in_progress'] = False
            
            return is_vip
            
        except Exception as e:
            logger.error(f"❌ VIP 权限检查异常: {str(e)}")
            with _VIP_CHECK_LOCK:
                _VIP_CACHE['check_in_progress'] = False
                # 发生异常时，如果有旧的缓存值，继续使用
                if _VIP_CACHE['is_vip'] is not None:
                    return _VIP_CACHE['is_vip']
        return False
    
    def _check_vip_permission(self) -> bool:
        """
        检查VIP权限（实际网络请求）
        
        🔧 注意：此方法会发起网络请求，应通过 _check_vip_permission_cached() 调用
        """
        try:
            from lib.common import Common
            from lib.api import getInfo
            
            # 获取机器码
            machine_code = Common.get_machine_code()
            if not machine_code:
                logger.warning("⚠️ 无法获取机器码，多模态AI功能不可用")
                return False
            logger.debug(f"✅ 机器码获取成功: {machine_code[:10]}...")
            
            # 🔧 使用超时保护的 VIP 检查
            flag, info = self._safe_get_info(machine_code)
            if not flag:
                logger.warning(f"⚠️ 无法验证用户信息，多模态AI功能不可用: {info}")
                return False
            
            is_vip = info.get('host', {}).get('is_vip', False)
            local_unlock = info.get('host', {}).get('local_unlock', 0)
            
            logger.info(f"📊 用户权限状态: is_vip={is_vip}, local_unlock={local_unlock}")
            
            if not is_vip:
                logger.warning("⚠️ 多模态AI功能仅限VIP用户使用，当前用户不是VIP")
                return False
            
            logger.info("✅ VIP权限验证通过")
            return True
            
        except Exception as e:
            logger.error(f"❌ VIP权限检查失败: {str(e)}")
            import traceback
            logger.error(f"详细错误: {traceback.format_exc()}")
            return False
    
    def _safe_get_info(self, machine_code: str) -> Tuple[bool, any]:
        """
        🔧 安全的用户信息获取（带超时保护）
        """
        try:
            from concurrent.futures import ThreadPoolExecutor, TimeoutError as FuturesTimeoutError
            from lib.api import getInfo
            
            with ThreadPoolExecutor(max_workers=1) as executor:
                future = executor.submit(getInfo, machine_code)
                try:
                    result = future.result(timeout=NETWORK_TIMEOUT)
                    return result
                except FuturesTimeoutError:
                    logger.warning(f"⚠️ 获取用户信息超时（{NETWORK_TIMEOUT}秒）")
                    return False, "请求超时"
        except Exception as e:
            logger.error(f"❌ 安全获取用户信息失败: {str(e)}")
            return False, str(e)
    
    @vip_required
    def recognize_image_with_claude(self, image_path: str, prompt: str = None) -> Tuple[bool, str, float]:
        """使用Claude 3.5 Sonnet识别图片"""
        try:
            api_key = self.api_keys.get('claude')
            if not api_key:
                return False, "未设置Claude API密钥", 0.0
            
            # 优化图片
            optimized_path = self.processor.optimize_image_for_ai(image_path)
            base64_image = self.processor.encode_image_to_base64(optimized_path)
            
            if not base64_image:
                return False, "图片编码失败", 0.0
            
            # 构建请求
            headers = {
                'Content-Type': 'application/json',
                'x-api-key': api_key,
                'anthropic-version': '2023-06-01'
            }
            
            # 默认提示词
            if not prompt:
                prompt = """请详细分析这张图片的内容，包括：
1. 图片中的文字内容（如果有）
2. 主要物体和场景
3. 图片的用途或类型
4. 任何重要的细节信息

请用中文回答，内容要准确且有条理。"""
            
            data = {
                "model": self.config.providers['claude']['model'],
                "max_tokens": 1000,
                "messages": [
                    {
                        "role": "user",
                        "content": [
                            {
                                "type": "text",
                                "text": prompt
                            },
                            {
                                "type": "image",
                                "source": {
                                    "type": "base64",
                                    "media_type": "image/jpeg",
                                    "data": base64_image
                                }
                            }
                        ]
                    }
                ]
            }
            
            # 发送请求
            response = requests.post(
                f"{self.config.providers['claude']['api_base']}/v1/messages",
                headers=headers,
                json=data,
                timeout=self.config.timeout
            )
            
            if response.status_code == 200:
                result = response.json()
                content = result['content'][0]['text']
                
                # 记录成本
                tokens_used = result.get('usage', {}).get('output_tokens', 500)
                cost = (tokens_used / 1000) * self.config.providers['claude']['cost_per_1k_tokens']
                cost += self.config.providers['claude']['cost_per_image']
                self.cost_controller.record_usage('claude', cost)
                
                # 清理临时文件
                if optimized_path != image_path:
                    try:
                        os.remove(optimized_path)
                    except:
                        pass
                
                return True, content, cost
            else:
                error_msg = f"Claude API错误: {response.status_code} - {response.text}"
                logger.error(error_msg)
                return False, error_msg, 0.0
                
        except Exception as e:
            logger.error(f"Claude图片识别失败: {str(e)}")
            return False, str(e), 0.0
    
    @vip_required
    def recognize_image_with_gpt4_vision(self, image_path: str, prompt: str = None) -> Tuple[bool, str, float]:
        """使用GPT-4 Vision识别图片"""
        try:
            api_key = self.api_keys.get('gpt4_vision')
            if not api_key:
                return False, "未设置OpenAI API密钥", 0.0
            
            # 优化图片
            optimized_path = self.processor.optimize_image_for_ai(image_path)
            base64_image = self.processor.encode_image_to_base64(optimized_path)
            
            if not base64_image:
                return False, "图片编码失败", 0.0
            
            # 构建请求
            headers = {
                'Content-Type': 'application/json',
                'Authorization': f'Bearer {api_key}'
            }
            
            # 默认提示词
            if not prompt:
                prompt = """请详细分析这张图片的内容，包括：
1. 图片中的文字内容（如果有）
2. 主要物体和场景
3. 图片的用途或类型
4. 任何重要的细节信息

请用中文回答，内容要准确且有条理。"""
            
            data = {
                "model": self.config.providers['gpt4_vision']['model'],
                "messages": [
                    {
                        "role": "user",
                        "content": [
                            {
                                "type": "text",
                                "text": prompt
                            },
                            {
                                "type": "image_url",
                                "image_url": {
                                    "url": f"data:image/jpeg;base64,{base64_image}",
                                    "detail": "high"
                                }
                            }
                        ]
                    }
                ],
                "max_tokens": 1000
            }
            
            # 发送请求
            response = requests.post(
                f"{self.config.providers['gpt4_vision']['api_base']}/chat/completions",
                headers=headers,
                json=data,
                timeout=self.config.timeout
            )
            
            if response.status_code == 200:
                result = response.json()
                content = result['choices'][0]['message']['content']
                
                # 记录成本
                tokens_used = result.get('usage', {}).get('total_tokens', 500)
                cost = (tokens_used / 1000) * self.config.providers['gpt4_vision']['cost_per_1k_tokens']
                cost += self.config.providers['gpt4_vision']['cost_per_image']
                self.cost_controller.record_usage('gpt4_vision', cost)
                
                # 清理临时文件
                if optimized_path != image_path:
                    try:
                        os.remove(optimized_path)
                    except:
                        pass
                
                return True, content, cost
            else:
                error_msg = f"OpenAI API错误: {response.status_code} - {response.text}"
                logger.error(error_msg)
                return False, error_msg, 0.0
                
        except Exception as e:
            logger.error(f"GPT-4 Vision图片识别失败: {str(e)}")
            return False, str(e), 0.0
    
    @vip_required
    def recognize_image_with_gemini(self, image_path: str, prompt: str = None) -> Tuple[bool, str, float]:
        """使用Gemini Pro Vision识别图片"""
        try:
            api_key = self.api_keys.get('gemini_pro')
            if not api_key:
                return False, "未设置Gemini API密钥", 0.0
            
            # 优化图片
            optimized_path = self.processor.optimize_image_for_ai(image_path)
            base64_image = self.processor.encode_image_to_base64(optimized_path)
            
            if not base64_image:
                return False, "图片编码失败", 0.0
            
            # 默认提示词
            if not prompt:
                prompt = """请详细分析这张图片的内容，包括：
1. 图片中的文字内容（如果有）
2. 主要物体和场景
3. 图片的用途或类型
4. 任何重要的细节信息

请用中文回答，内容要准确且有条理。"""
            
            # 构建请求数据
            data = {
                "contents": [
                    {
                        "parts": [
                            {"text": prompt},
                            {
                                "inline_data": {
                                    "mime_type": "image/jpeg",
                                    "data": base64_image
                                }
                            }
                        ]
                    }
                ],
                "generationConfig": {
                    "maxOutputTokens": 1000,
                    "temperature": 0.1
                }
            }
            
            # 发送请求
            url = f"{self.config.providers['gemini_pro']['api_base']}/models/{self.config.providers['gemini_pro']['model']}:generateContent?key={api_key}"
            response = requests.post(url, json=data, timeout=self.config.timeout)
            
            if response.status_code == 200:
                result = response.json()
                content = result['candidates'][0]['content']['parts'][0]['text']
                
                # 记录成本（Gemini的成本计算相对简单）
                cost = self.config.providers['gemini_pro']['cost_per_image']
                self.cost_controller.record_usage('gemini_pro', cost)
                
                # 清理临时文件
                if optimized_path != image_path:
                    try:
                        os.remove(optimized_path)
                    except:
                        pass
                
                return True, content, cost
            else:
                error_msg = f"Gemini API错误: {response.status_code} - {response.text}"
                logger.error(error_msg)
                return False, error_msg, 0.0
                
        except Exception as e:
            logger.error(f"Gemini图片识别失败: {str(e)}")
            return False, str(e), 0.0
    
    @vip_required
    def recognize_image_with_wenxin(self, image_path: str, prompt: str = None) -> Tuple[bool, str, float]:
        """使用文心一言4.0识别图片"""
        try:
            api_key = self.api_keys.get('wenxin')
            if not api_key:
                return False, "未设置文心一言API密钥", 0.0
            
            # 优化图片
            optimized_path = self.processor.optimize_image_for_ai(image_path)
            base64_image = self.processor.encode_image_to_base64(optimized_path)
            
            if not base64_image:
                return False, "图片编码失败", 0.0
            
            # 默认提示词
            if not prompt:
                prompt = """请详细分析这张图片的内容，包括：
1. 图片中的文字内容（如果有）
2. 主要物体和场景
3. 图片的用途或类型
4. 任何重要的细节信息

请用中文回答，内容要准确且有条理。"""
            
            # 文心一言需要先获取access_token
            access_token = self._get_wenxin_access_token()
            if not access_token:
                return False, "获取文心一言访问令牌失败", 0.0
            
            # 构建请求数据
            data = {
                "messages": [
                    {
                        "role": "user",
                        "content": [
                            {
                                "type": "text",
                                "text": prompt
                            },
                            {
                                "type": "image",
                                "image": base64_image
                            }
                        ]
                    }
                ]
            }
            
            # 发送请求
            url = f"{self.config.providers['wenxin']['api_base']}/chat/eb-instant?access_token={access_token}"
            response = requests.post(url, json=data, timeout=self.config.timeout)
            
            if response.status_code == 200:
                result = response.json()
                content = result.get('result', '')
                
                # 记录成本
                cost = self.config.providers['wenxin']['cost_per_image']
                self.cost_controller.record_usage('wenxin', cost)
                
                # 清理临时文件
                if optimized_path != image_path:
                    try:
                        os.remove(optimized_path)
                    except:
                        pass
                
                return True, content, cost
            else:
                error_msg = f"文心一言API错误: {response.status_code} - {response.text}"
                logger.error(error_msg)
                return False, error_msg, 0.0
                
        except Exception as e:
            logger.error(f"文心一言图片识别失败: {str(e)}")
            return False, str(e), 0.0
    
    @vip_required
    def recognize_image_with_doubao(self, image_path: str, prompt: str = None) -> Tuple[bool, str, float]:
        """使用火山引擎豆包识别图片"""
        try:
            api_key = self.api_keys.get('doubao')
            if not api_key:
                return False, "未设置火山引擎豆包API密钥", 0.0
            
            # 优化图片
            optimized_path = self.processor.optimize_image_for_ai(image_path)
            base64_image = self.processor.encode_image_to_base64(optimized_path)
            
            if not base64_image:
                return False, "图片编码失败", 0.0
            
            # 默认提示词
            if not prompt:
                prompt = """请详细分析这张图片的内容，包括：
1. 图片中的文字内容（如果有）
2. 主要物体和场景
3. 图片的用途或类型
4. 任何重要的细节信息

请用中文回答，内容要准确且有条理。"""
            
            # 获取文件扩展名来确定MIME类型
            from pathlib import Path
            file_extension = Path(optimized_path).suffix.lower()
            mime_type_map = {
                '.jpg': 'image/jpeg',
                '.jpeg': 'image/jpeg',
                '.png': 'image/png',
                '.gif': 'image/gif',
                '.bmp': 'image/bmp',
                '.webp': 'image/webp'
            }
            
            mime_type = mime_type_map.get(file_extension, 'image/jpeg')
            data_url = f"data:{mime_type};base64,{base64_image}"
            
            # 构建请求数据
            data = {
                "model": self.config.providers['doubao']['model'],
                "messages": [
                    {
                        "role": "user",
                        "content": [
                            {
                                "type": "image_url",
                                "image_url": {
                                    "url": data_url
                                }
                            },
                            {
                                "type": "text",
                                "text": prompt
                            }
                        ]
                    }
                ]
            }
            
            # 构建请求头
            headers = {
                "Content-Type": "application/json",
                "Authorization": f"Bearer {api_key}"
            }
            
            # 🔧 使用线程安全的请求方法
            result = _safe_requests_post(
                self.config.providers['doubao']['api_base'], 
                headers=headers, 
                json_data=data, 
                timeout=self.config.timeout
            )
            
            if result is None:
                logger.error("火山引擎豆包API请求失败或超时")
                return False, "API请求失败或超时", 0.0
            
            if 'error' in result:
                logger.error(f"火山引擎豆包API请求失败: {result.get('error')}")
                return False, f"HTTP错误: {result.get('error')}", 0.0
            
                if "choices" in result and len(result["choices"]) > 0:
                    content = result["choices"][0]["message"]["content"]
                    
                    # 记录成本
                    cost = self.config.providers['doubao']['cost_per_image']
                    self.cost_controller.record_usage('doubao', cost)
                    
                    logger.info("火山引擎豆包图片识别成功")
                    return True, content, cost
                else:
                    logger.error(f"火山引擎豆包API返回错误: {result}")
                    return False, f"API错误: {result}", 0.0
                
        except Exception as e:
            logger.error(f"火山引擎豆包图片识别失败: {str(e)}")
            return False, str(e), 0.0
    
    @vip_required
    def recognize_image_with_openai_compatible(self, image_path: str, prompt: str = None) -> Tuple[bool, str, float]:
        """使用OpenAI兼容API识别图片"""
        try:
            api_key = self.api_keys.get('multimodal')
            if not api_key:
                return False, "未设置多模态AI API密钥", 0.0
            
            if not self.api_base or not self.model_name:
                return False, "多模态AI配置不完整，请检查API地址和模型名称", 0.0
            
            # 优化图片
            optimized_path = self.processor.optimize_image_for_ai(image_path)
            base64_image = self.processor.encode_image_to_base64(optimized_path)
            
            if not base64_image:
                return False, "图片编码失败", 0.0
            
            # 默认提示词
            if not prompt:
                prompt = """请详细分析这张图片的内容，包括：
1. 图片中的文字内容（如果有）
2. 主要物体和场景
3. 图片的用途或类型
4. 任何重要的细节信息

请用中文回答，内容要准确且有条理。"""
            
            # 获取文件扩展名来确定MIME类型
            from pathlib import Path
            file_extension = Path(optimized_path).suffix.lower()
            mime_type_map = {
                '.jpg': 'image/jpeg',
                '.jpeg': 'image/jpeg',
                '.png': 'image/png',
                '.gif': 'image/gif',
                '.bmp': 'image/bmp',
                '.webp': 'image/webp'
            }
            
            mime_type = mime_type_map.get(file_extension, 'image/jpeg')
            data_url = f"data:{mime_type};base64,{base64_image}"
            
            # 构建OpenAI兼容的请求数据
            data = {
                "model": self.model_name,
                "messages": [
                    {
                        "role": "user",
                        "content": [
                            {
                                "type": "image_url",
                                "image_url": {
                                    "url": data_url
                                }
                            },
                            {
                                "type": "text",
                                "text": prompt
                            }
                        ]
                    }
                ]
            }
            
            # 构建请求头
            headers = {
                "Content-Type": "application/json",
                "Authorization": f"Bearer {api_key}"
            }
            
            # 构建完整的API URL
            if self.api_base.endswith('/'):
                api_url = f"{self.api_base}chat/completions"
            else:
                api_url = f"{self.api_base}/chat/completions"
            
            # 🔧 使用线程安全的请求方法
            result = _safe_requests_post(
                api_url, 
                headers=headers, 
                json_data=data, 
                timeout=self.config.timeout
            )
            
            if result is None:
                logger.error("多模态AI API请求失败或超时")
                return False, "API请求失败或超时", 0.0
            
            if 'error' in result:
                logger.error(f"多模态AI API请求失败: {result.get('error')}")
                return False, f"HTTP错误: {result.get('error')} - {result.get('text', '')}", 0.0
            
                if "choices" in result and len(result["choices"]) > 0:
                    content = result["choices"][0]["message"]["content"]
                    
                    # 记录成本（通用估算）
                    cost = 0.01  # 平均估算成本
                    self.cost_controller.record_usage('multimodal', cost)
                    
                    logger.info("多模态AI图片识别成功")
                    return True, content, cost
                else:
                    logger.error(f"多模态AI API返回错误: {result}")
                    return False, f"API错误: {result}", 0.0
                
        except Exception as e:
            logger.error(f"多模态AI图片识别失败: {str(e)}")
            return False, str(e), 0.0
    
    def _get_wenxin_access_token(self) -> str:
        """获取文心一言访问令牌"""
        try:
            api_key = self.api_keys.get('wenxin')
            if not api_key:
                logger.error("未设置文心一言API密钥")
                return ""
            
            # 文心一言API密钥格式应该是 "API_KEY:SECRET_KEY"
            if ':' not in api_key:
                logger.error("文心一言API密钥格式错误，应为 'API_KEY:SECRET_KEY'")
                return ""
            
            client_id, client_secret = api_key.split(':', 1)
            
            # 获取access_token
            url = "https://aip.baidubce.com/oauth/2.0/token"
            params = {
                "grant_type": "client_credentials",
                "client_id": client_id,
                "client_secret": client_secret
            }
            
            response = requests.post(url, params=params, timeout=30)
            if response.status_code == 200:
                result = response.json()
                if "access_token" in result:
                    logger.info("成功获取文心一言访问令牌")
                    return result["access_token"]
                else:
                    logger.error(f"获取文心一言访问令牌失败: {result}")
                    return ""
            else:
                logger.error(f"文心一言API请求失败: {response.status_code}")
                return ""
                
        except Exception as e:
            logger.error(f"获取文心一言访问令牌失败: {str(e)}")
            return ""
    
    def recognize_image_smart(self, image_path: str, prompt: str = None, 
                             preferred_provider: str = None) -> Tuple[bool, str, float, str]:
        """智能图片识别 - 自动选择最佳服务商"""
        
        # 🔧 使用缓存的 VIP 权限检查，避免重复网络请求
        if not self._check_vip_permission_cached():
            return False, "多模态AI功能仅限VIP用户使用，请升级为VIP用户", 0.0, "permission_denied"
        
        # 检查图片文件
        if not os.path.exists(image_path):
            return False, "图片文件不存在", 0.0, "file_error"
        
        # 确定使用的服务商
        providers_to_try = []
        
        # 优先使用新的统一配置
        if 'multimodal' in self.api_keys and self.api_base and self.model_name:
            providers_to_try.append('multimodal')
        
        # 如果有指定的服务商且在旧配置中
        if preferred_provider and preferred_provider in self.config.providers:
            providers_to_try.append(preferred_provider)
        elif self.config.default_provider not in providers_to_try:
            providers_to_try.append(self.config.default_provider)
        
        # 添加备用服务商
        for fallback in self.config.fallback_providers:
            if fallback not in providers_to_try:
                providers_to_try.append(fallback)
        
        # 逐个尝试服务商
        total_cost = 0.0
        last_error = ""
        
        for provider in providers_to_try:
            # 检查API密钥
            if provider not in self.api_keys:
                logger.warning(f"跳过 {provider}：未设置API密钥")
                continue
            
            # 估算成本并检查预算
            estimated_cost = self.cost_controller.estimate_cost(provider, 1, self.config)
            can_afford, budget_msg = self.cost_controller.check_budget(estimated_cost)
            
            if not can_afford:
                logger.warning(f"跳过 {provider}：{budget_msg}")
                continue
            
            logger.info(f"尝试使用 {provider} 识别图片: {os.path.basename(image_path)}")
            
            # 调用对应的识别方法
            success, content, cost = False, "", 0.0
            
            if provider == 'multimodal':
                success, content, cost = self.recognize_image_with_openai_compatible(image_path, prompt)
            elif provider == 'claude':
                success, content, cost = self.recognize_image_with_claude(image_path, prompt)
            elif provider == 'gpt4_vision':
                success, content, cost = self.recognize_image_with_gpt4_vision(image_path, prompt)
            elif provider == 'gemini_pro':
                success, content, cost = self.recognize_image_with_gemini(image_path, prompt)
            elif provider == 'wenxin':
                success, content, cost = self.recognize_image_with_wenxin(image_path, prompt)
            elif provider == 'doubao':
                success, content, cost = self.recognize_image_with_doubao(image_path, prompt)
            
            total_cost += cost
            
            if success and content.strip():
                logger.info(f"图片识别成功，使用服务商: {provider}，成本: ${cost:.4f}")
                logger.info(f"【图片识别内容】 {os.path.basename(image_path)}:\n{content}")
                return True, content, total_cost, provider
            else:
                last_error = content
                logger.warning(f"{provider} 识别失败: {content}")
        
        # 所有服务商都失败
        return False, f"所有多模态AI服务都失败，最后错误: {last_error}", total_cost, "all_failed"
    
    def batch_recognize_images(self, image_paths: List[str], prompt: str = None,
                              progress_callback: Callable = None) -> List[Dict]:
        """批量图片识别"""
        results = []
        total_images = len(image_paths)
        
        for i, image_path in enumerate(image_paths):
            try:
                success, content, cost, provider = self.recognize_image_smart(image_path, prompt)
                
                result = {
                    'image_path': image_path,
                    'success': success,
                    'content': content,
                    'cost': cost,
                    'provider': provider,
                    'timestamp': time.time()
                }
                results.append(result)
                
                # 更新进度
                if progress_callback:
                    progress_callback(i + 1, total_images, os.path.basename(image_path))
                
                # 适当延迟，避免API限流
                time.sleep(0.5)
                
            except Exception as e:
                logger.error(f"批量识别图片失败 {image_path}: {str(e)}")
                results.append({
                    'image_path': image_path,
                    'success': False,
                    'content': str(e),
                    'cost': 0.0,
                    'provider': 'error',
                    'timestamp': time.time()
                })
        
        return results
    
    def get_usage_statistics(self) -> Dict:
        """获取使用统计"""
        stats = self.cost_controller.usage_history
        
        # 按服务商统计
        provider_stats = {}
        total_cost = 0.0
        
        for usage in stats:
            provider = usage['provider']
            cost = usage['cost']
            
            if provider not in provider_stats:
                provider_stats[provider] = {
                    'count': 0,
                    'total_cost': 0.0,
                    'avg_cost': 0.0
                }
            
            provider_stats[provider]['count'] += 1
            provider_stats[provider]['total_cost'] += cost
            total_cost += cost
        
        # 计算平均成本
        for provider_data in provider_stats.values():
            provider_data['avg_cost'] = provider_data['total_cost'] / provider_data['count']
        
        return {
            'total_cost': total_cost,
            'budget_limit': self.cost_controller.budget_limit,
            'budget_remaining': self.cost_controller.budget_limit - total_cost,
            'usage_rate': total_cost / self.cost_controller.budget_limit,
            'provider_stats': provider_stats,
            'cost_alerts': self.cost_controller.cost_alerts
        }


class SmartImageRouter:
    """智能图片路由器 - 根据图片特征选择最佳识别方法"""
    
    def __init__(self):
        self.ocr_threshold = 0.8  # OCR置信度阈值
        
    def analyze_image_type(self, image_path: str) -> str:
        """分析图片类型"""
        try:
            with Image.open(image_path) as img:
                width, height = img.size
                aspect_ratio = width / height
                
                # 根据尺寸和比例判断图片类型
                if aspect_ratio > 3 or aspect_ratio < 0.3:
                    return "text_document"  # 可能是文档截图
                elif width * height > 2000 * 2000:
                    return "high_resolution"  # 高分辨率图片
                elif width < 800 and height < 600:
                    return "small_image"  # 小图片
                else:
                    return "general_image"  # 普通图片
                    
        except Exception as e:
            logger.error(f"分析图片类型失败: {str(e)}")
            return "unknown"
    
    def choose_recognition_method(self, image_path: str, available_methods: List[str]) -> str:
        """选择最佳识别方法"""
        image_type = self.analyze_image_type(image_path)
        
        # 根据图片类型选择方法
        if image_type == "text_document":
            # 文档类图片优先使用OCR
            if "tesseract" in available_methods:
                return "tesseract"
            elif "claude" in available_methods:
                return "claude"
        elif image_type == "high_resolution":
            # 高分辨率图片使用多模态AI
            if "claude" in available_methods:
                return "claude"
            elif "gpt4_vision" in available_methods:
                return "gpt4_vision"
        elif image_type == "small_image":
            # 小图片使用快速方法
            if "gemini_pro" in available_methods:
                return "gemini_pro"
            elif "tesseract" in available_methods:
                return "tesseract"
        
        # 默认选择
        for preferred in ["claude", "gpt4_vision", "gemini_pro", "tesseract"]:
            if preferred in available_methods:
                return preferred
        
        return available_methods[0] if available_methods else "tesseract"


# 创建全局实例并自动加载配置
def _create_multimodal_service():
    """创建并初始化多模态AI服务"""
    service = MultimodalAIService()
    
    # 尝试加载已保存的配置
    try:
        import os
        import json
        config_file = os.path.join(os.path.expanduser('~'), '.fileneatai', 'multimodal_settings.json')
        if os.path.exists(config_file):
            with open(config_file, 'r', encoding='utf-8') as f:
                settings = json.load(f)
            
            # 加载多模态AI配置
            multimodal_config = settings.get('multimodal', {})
            if multimodal_config.get('enabled', False) and multimodal_config.get('api_key'):
                # 使用通用的'multimodal'作为provider名称
                service.set_api_key('multimodal', multimodal_config['api_key'])
                # 保存API地址和模型名称到服务实例
                service.api_base = multimodal_config.get('api_base', '')
                service.model_name = multimodal_config.get('model', '')
                logger.info(f"已从配置文件加载多模态AI配置: {service.model_name}")
            
            # 加载成本控制设置
            budget = settings.get('budget', {})
            if budget.get('monthly_limit'):
                service.cost_controller.budget_limit = budget['monthly_limit']
                
    except Exception as e:
        logger.warning(f"加载多模态AI配置失败: {str(e)}")
    
    return service

multimodal_service = _create_multimodal_service()
image_router = SmartImageRouter()
