# -*- coding: utf-8 -*-
"""
PDF Moonshot处理器 - 针对扫描型PDF和图片型PDF的Moonshot API处理
"""

import os
import tempfile
import time
from pathlib import Path
from typing import Optional, Tuple, List
from logger import logger

try:
    import fitz  # PyMuPDF
    HAS_PYMUPDF = True
except ImportError:
    HAS_PYMUPDF = False
    logger.warning("PyMuPDF未安装，无法处理PDF文件")

try:
    from openai import OpenAI
    HAS_OPENAI = True
except ImportError:
    HAS_OPENAI = False
    logger.warning("OpenAI库未安装，无法使用Moonshot API")


class PDFMoonshotProcessor:
    """PDF和Word文档 Moonshot处理器"""
    
    def __init__(self, moonshot_api_key: str):
        """
        初始化PDF Moonshot处理器
        
        Args:
            moonshot_api_key: Moonshot API密钥
        """
        self.moonshot_api_key = moonshot_api_key
        self.client = None
        
        if HAS_OPENAI and moonshot_api_key:
            try:
                self.client = OpenAI(
                    base_url="https://api.moonshot.cn/v1",
                    api_key=moonshot_api_key,
                )
                logger.info("Moonshot API客户端初始化成功")
                
                # 🚀 初始化时自动清理旧的临时文件（超过1小时的）
                try:
                    self.cleanup_temp_files(max_age_minutes=60)
                except Exception as cleanup_e:
                    logger.debug(f"初始化时清理临时文件失败: {str(cleanup_e)}")
                    
            except Exception as e:
                logger.error(f"初始化Moonshot API客户端失败: {str(e)}")
                self.client = None
        else:
            logger.warning("无法初始化Moonshot API客户端：缺少依赖或API密钥")
    
    def is_available(self) -> bool:
        """检查处理器是否可用"""
        return HAS_PYMUPDF and HAS_OPENAI and self.client is not None
    
    def is_scanned_or_image_pdf(self, file_path: str) -> bool:
        """
        检测PDF是否为扫描型或图片型PDF
        
        Args:
            file_path: PDF文件路径
            
        Returns:
            bool: True表示是扫描型/图片型PDF
        """
        if not HAS_PYMUPDF:
            logger.warning("PyMuPDF未安装，无法检测PDF类型")
            return False
            
        try:
            pdf_doc = fitz.open(file_path)
            
            # 检查前3页
            text_content_length = 0
            for page_num in range(min(3, pdf_doc.page_count)):
                page = pdf_doc[page_num]
                page_text = page.get_text()
                text_content_length += len(page_text.strip())
            
            pdf_doc.close()
            
            # 如果前3页提取的文本内容很少，认为是扫描型/图片型PDF
            is_scanned = text_content_length < 100  # 少于100个字符认为是扫描型
            
            logger.info(f"PDF类型检测结果: {'扫描型/图片型' if is_scanned else '文本型'} PDF，文本长度: {text_content_length}")
            return is_scanned
            
        except Exception as e:
            logger.error(f"检测PDF类型失败: {str(e)}")
            return False
    
    def extract_first_n_pages_to_temp_pdf(self, file_path: str, max_pages: int = 10) -> Optional[str]:
        """
        提取PDF的前N页并创建临时PDF文件
        
        Args:
            file_path: 原始PDF文件路径
            max_pages: 最大页数，默认10页
            
        Returns:
            str: 临时PDF文件路径，如果失败返回None
        """
        if not HAS_PYMUPDF:
            logger.error("PyMuPDF未安装，无法提取PDF页面")
            return None
            
        try:
            # 打开原始PDF
            source_pdf = fitz.open(file_path)
            total_pages = source_pdf.page_count
            
            # 确定要提取的页数
            pages_to_extract = min(max_pages, total_pages)
            logger.info(f"从PDF中提取前{pages_to_extract}页（总页数: {total_pages}）")
            
            # 创建新的PDF文档
            new_pdf = fitz.open()
            
            # 复制前N页
            for page_num in range(pages_to_extract):
                page = source_pdf[page_num]
                new_pdf.insert_pdf(source_pdf, from_page=page_num, to_page=page_num)
                logger.debug(f"已复制第{page_num + 1}页")
            
            # 创建临时文件路径
            temp_dir = tempfile.gettempdir()
            import uuid
            temp_filename = f"pdf_extract_{uuid.uuid4().hex[:8]}.pdf"
            temp_pdf_path = os.path.join(temp_dir, temp_filename)
            
            # 保存临时PDF
            new_pdf.save(temp_pdf_path)
            new_pdf.close()
            source_pdf.close()
            
            logger.info(f"成功创建临时PDF文件: {temp_pdf_path}")
            return temp_pdf_path
            
        except Exception as e:
            logger.error(f"提取PDF页面失败: {str(e)}")
            return None
    
    def upload_file_to_moonshot(self, file_path: str) -> Optional[Tuple[str, str]]:
        """
        上传文件到Moonshot API
        
        Args:
            file_path: 要上传的文件路径
            
        Returns:
            Tuple[str, str]: (file_id, file_content) 如果成功，否则返回None
        """
        if not self.client:
            logger.error("Moonshot API客户端未初始化")
            return None
            
        try:
            logger.info(f"开始上传文件到Moonshot: {file_path}")
            
            # 上传文件
            file_object = self.client.files.create(
                file=Path(file_path), 
                purpose="file-extract"
            )
            
            logger.info(f"文件上传成功，file_id: {file_object.id}")
            
            # 获取文件内容
            file_content = self.client.files.content(file_id=file_object.id).text
            
            logger.info(f"文件内容提取成功，内容长度: {len(file_content)} 字符")
            
            # 🚀 重要：上传成功后，如果这是临时文件，立即删除本地副本以节省磁盘空间
            if file_path and 'pdf_extract_' in os.path.basename(file_path):
                try:
                    # 稍等一下确保文件上传完成
                    time.sleep(0.1)
                    if os.path.exists(file_path):
                        os.unlink(file_path)
                        logger.debug(f"上传后立即清理临时文件: {file_path}")
                except Exception as cleanup_e:
                    logger.warning(f"上传后清理临时文件失败: {str(cleanup_e)}")
            
            return file_object.id, file_content
            
        except Exception as e:
            logger.error(f"上传文件到Moonshot失败: {str(e)}")
            return None
    
    def delete_moonshot_file(self, file_id: str) -> bool:
        """
        删除Moonshot服务器上的文件
        
        Args:
            file_id: 要删除的文件ID
            
        Returns:
            bool: 删除是否成功
        """
        if not self.client:
            logger.error("Moonshot API客户端未初始化")
            return False
            
        try:
            logger.info(f"开始删除Moonshot服务器文件: {file_id}")
            
            # 删除文件
            delete_result = self.client.files.delete(file_id=file_id)
            
            logger.info(f"Moonshot服务器文件删除成功: {file_id}")
            return True
            
        except Exception as e:
            logger.error(f"删除Moonshot服务器文件失败 {file_id}: {str(e)}")
            return False
    
    def cleanup_moonshot_server_files(self) -> int:
        """
        清理Moonshot服务器上的所有文件
        
        Returns:
            int: 删除的文件数量
        """
        if not self.client:
            logger.error("Moonshot API客户端未初始化")
            return 0
            
        try:
            logger.info("开始清理Moonshot服务器上的文件")
            
            # 获取文件列表
            file_list = self.client.files.list()
            deleted_count = 0
            
            for file in file_list.data:
                try:
                    delete_result = self.client.files.delete(file_id=file.id)
                    deleted_count += 1
                    logger.debug(f"删除Moonshot服务器文件: {file.id}")
                except Exception as e:
                    logger.warning(f"删除Moonshot服务器文件失败 {file.id}: {str(e)}")
            
            if deleted_count > 0:
                logger.info(f"清理了 {deleted_count} 个Moonshot服务器文件")
            else:
                logger.info("Moonshot服务器上没有需要清理的文件")
            
            return deleted_count
            
        except Exception as e:
            logger.error(f"清理Moonshot服务器文件失败: {str(e)}")
            return 0
    
    def _is_valid_extracted_text(self, content: str) -> bool:
        """
        🚀 Token优化：检查内容是否已经是有效的提取文本
        
        Args:
            content: 待检查的内容
            
        Returns:
            bool: True表示是有效的文本内容，可以直接使用
        """
        if not content or len(content.strip()) < 20:
            return False
        
        # 检查是否是JSON格式（需要进一步解析）
        content_stripped = content.strip()
        if content_stripped.startswith('{') or content_stripped.startswith('['):
            try:
                import json
                json.loads(content_stripped)
                return False  # 是JSON，需要解析
            except:
                pass  # 不是有效JSON，当作普通文本
        
        # 如果内容中有足够多的中文或英文字符，认为是有效文本
        chinese_chars = sum(1 for c in content if '\u4e00' <= c <= '\u9fff')
        english_words = len(content.split())
        
        return chinese_chars > 10 or english_words > 20
    
    def _extract_text_from_json(self, content: str) -> Optional[str]:
        """
        🚀 Token优化：从JSON格式的内容中提取文本
        
        Args:
            content: JSON格式的内容
            
        Returns:
            str: 提取的文本，如果失败返回None
        """
        try:
            import json
            data = json.loads(content.strip())
            
            # 尝试常见的JSON结构
            if isinstance(data, dict):
                # 尝试 content 字段
                if 'content' in data:
                    return data['content']
                # 尝试 text 字段
                if 'text' in data:
                    return data['text']
                # 尝试 data 字段
                if 'data' in data and isinstance(data['data'], str):
                    return data['data']
            elif isinstance(data, str):
                return data
                
        except:
            pass
        
        return None
    
    def analyze_with_moonshot(self, file_content: str, custom_prompt: str = "", skip_if_valid: bool = True) -> Optional[str]:
        """
        使用Moonshot API分析文件内容
        🚀 Token优化版本：如果内容已经是有效文本，直接返回，避免重复API调用
        
        Args:
            file_content: 文件内容
            custom_prompt: 自定义提示词
            skip_if_valid: 是否在内容有效时跳过API调用（默认True，大幅节省token）
            
        Returns:
            str: 分析结果，如果失败返回None
        """
        if not self.client:
            logger.error("Moonshot API客户端未初始化")
            return None
        
        # 🚀 Token优化：首先尝试从JSON中提取文本
        json_text = self._extract_text_from_json(file_content)
        if json_text and len(json_text.strip()) > 20:
            logger.info(f"✅ 从JSON中直接提取文本成功，长度: {len(json_text)} 字符，跳过API调用节省token")
            return json_text
        
        # 🚀 Token优化：如果内容已经是有效的纯文本，直接返回
        if skip_if_valid and self._is_valid_extracted_text(file_content):
            logger.info(f"✅ 文件内容已是有效文本，长度: {len(file_content)} 字符，跳过API调用节省token")
            return file_content
            
        # 🚀 Token优化：只有在内容无效或需要特殊处理时才调用API
        logger.info("⚠️ 文件内容需要API分析（内容可能是二进制或格式特殊）")
        
        try:
            # 🚀 Token优化：大幅简化提示词，减少输入token
            # 不再将整个file_content作为system message发送
            content_preview = file_content[:2000] if len(file_content) > 2000 else file_content
            
            messages = [
                {
                    "role": "user",
                    "content": f"请从以下内容中提取关键信息用于文件分类（限500字）：\n\n{content_preview}"
                }
            ]
            
            logger.info("开始使用Moonshot API分析文件内容（已优化token消耗）")
            
            # 🚀 Token优化：使用8k模型而非32k，限制输出长度
            completion = self.client.chat.completions.create(
                model="moonshot-v1-8k",  # 使用更小的模型节省token
                messages=messages,
                max_tokens=800,  # 限制输出长度
                temperature=0.1,
            )
            
            result = completion.choices[0].message.content
            logger.info(f"Moonshot API分析完成，结果长度: {len(result)} 字符")
            
            return result
            
        except Exception as e:
            logger.error(f"使用Moonshot API分析失败: {str(e)}")
            return None
    
    def process_pdf_with_moonshot(self, file_path: str, custom_prompt: str = "", max_pages: int = 10) -> Optional[str]:
        """
        完整的PDF Moonshot处理流程
        
        Args:
            file_path: PDF文件路径
            custom_prompt: 自定义提示词
            max_pages: 最大处理页数
            
        Returns:
            str: 处理结果，如果失败返回None
        """
        if not self.is_available():
            logger.error("PDF Moonshot处理器不可用")
            return None
        
        temp_pdf_path = None
        
        try:
            logger.info(f"🌙 开始使用Moonshot处理PDF文件: {file_path}")
            
            # 检查PDF类型（仅用于日志记录，不影响处理决策）
            try:
                is_scanned = self.is_scanned_or_image_pdf(file_path)
                pdf_type = "扫描型/图片型" if is_scanned else "文本型"
                logger.info(f"📄 PDF类型: {pdf_type}，使用Moonshot API统一处理")
            except Exception as e:
                logger.debug(f"PDF类型检测失败，继续处理: {str(e)}")
            
            # 提取前N页到临时PDF
            temp_pdf_path = self.extract_first_n_pages_to_temp_pdf(file_path, max_pages)
            if not temp_pdf_path:
                logger.error("创建临时PDF失败")
                return None
            
            # 上传到Moonshot
            upload_result = self.upload_file_to_moonshot(temp_pdf_path)
            if not upload_result:
                logger.error("上传文件到Moonshot失败")
                return None
            
            file_id, file_content = upload_result
            
            try:
                # 使用Moonshot分析
                analysis_result = self.analyze_with_moonshot(file_content, custom_prompt)
                if not analysis_result:
                    logger.error("Moonshot分析失败")
                    return None
                
                logger.info("PDF Moonshot处理完成")
                return analysis_result
                
            finally:
                # 🚀 重要：分析完成后立即删除Moonshot服务器上的文件
                try:
                    self.delete_moonshot_file(file_id)
                except Exception as delete_e:
                    logger.warning(f"删除Moonshot服务器文件失败: {str(delete_e)}")
            
        except Exception as e:
            logger.error(f"PDF Moonshot处理过程中出错: {str(e)}")
            return None
        
        finally:
            # 清理临时文件（如果还存在的话）
            if temp_pdf_path and os.path.exists(temp_pdf_path):
                try:
                    os.unlink(temp_pdf_path)
                    logger.debug(f"最终清理临时PDF文件: {temp_pdf_path}")
                except Exception as cleanup_e:
                    logger.warning(f"最终清理临时PDF文件失败: {str(cleanup_e)}")
            elif temp_pdf_path:
                logger.debug(f"临时PDF文件已在上传后被清理: {temp_pdf_path}")
    
    def process_word_with_moonshot(self, file_path: str, custom_prompt: str = "") -> Optional[str]:
        """
        完整的Word文档 Moonshot处理流程
        
        Args:
            file_path: Word文件路径
            custom_prompt: 自定义提示词
            
        Returns:
            str: 处理结果，如果失败返回None
        """
        if not self.is_available():
            logger.error("Word Moonshot处理器不可用")
            return None
        
        try:
            logger.info(f"开始使用Moonshot处理Word文档: {file_path}")
            
            # 直接上传Word文档到Moonshot
            upload_result = self.upload_file_to_moonshot(file_path)
            if not upload_result:
                logger.error("上传Word文档到Moonshot失败")
                return None
            
            file_id, file_content = upload_result
            
            try:
                # 使用Moonshot分析
                analysis_result = self.analyze_with_moonshot(file_content, custom_prompt)
                if not analysis_result:
                    logger.error("Moonshot分析Word文档失败")
                    return None
                
                logger.info("Word文档 Moonshot处理完成")
                return analysis_result
                
            finally:
                # 🚀 重要：分析完成后立即删除Moonshot服务器上的文件
                try:
                    self.delete_moonshot_file(file_id)
                except Exception as delete_e:
                    logger.warning(f"删除Moonshot服务器文件失败: {str(delete_e)}")
            
        except Exception as e:
            logger.error(f"Word Moonshot处理过程中出错: {str(e)}")
            return None
    
    def process_document_with_moonshot(self, file_path: str, custom_prompt: str = "", max_pages: int = 10) -> Optional[str]:
        """
        通用文档 Moonshot处理流程（自动识别文件类型）
        
        Args:
            file_path: 文件路径
            custom_prompt: 自定义提示词
            max_pages: PDF最大处理页数
            
        Returns:
            str: 处理结果，如果失败返回None
        """
        file_extension = Path(file_path).suffix.lower()
        
        if file_extension == '.pdf':
            return self.process_pdf_with_moonshot(file_path, custom_prompt, max_pages)
        elif file_extension in ['.doc', '.docx']:
            return self.process_word_with_moonshot(file_path, custom_prompt)
        else:
            logger.warning(f"不支持的文件类型: {file_extension}")
            return None
    
    def cleanup_temp_files(self, temp_dir: Optional[str] = None, max_age_minutes: int = 60):
        """
        清理临时文件
        
        Args:
            temp_dir: 临时目录路径，如果为None则使用系统默认临时目录
            max_age_minutes: 文件最大保留时间（分钟），超过此时间的文件将被删除
        """
        if temp_dir is None:
            temp_dir = tempfile.gettempdir()
        
        try:
            cleaned_count = 0
            # 查找并删除以pdf_extract_开头的临时文件
            for filename in os.listdir(temp_dir):
                if filename.startswith('pdf_extract_') and filename.endswith('.pdf'):
                    temp_file_path = os.path.join(temp_dir, filename)
                    try:
                        # 检查文件是否较旧
                        file_age = time.time() - os.path.getctime(temp_file_path)
                        max_age_seconds = max_age_minutes * 60
                        
                        if file_age > max_age_seconds:
                            os.unlink(temp_file_path)
                            cleaned_count += 1
                            logger.debug(f"清理旧临时文件: {temp_file_path} (年龄: {file_age/60:.1f}分钟)")
                    except Exception as e:
                        logger.debug(f"清理临时文件失败 {temp_file_path}: {str(e)}")
            
            if cleaned_count > 0:
                logger.info(f"清理了 {cleaned_count} 个旧临时PDF文件")
        
        except Exception as e:
            logger.warning(f"清理临时文件目录失败: {str(e)}")
    
    def cleanup_all_temp_files(self, temp_dir: Optional[str] = None):
        """
        强制清理所有临时文件（不管年龄）
        
        Args:
            temp_dir: 临时目录路径，如果为None则使用系统默认临时目录
        """
        if temp_dir is None:
            temp_dir = tempfile.gettempdir()
        
        try:
            cleaned_count = 0
            # 查找并删除所有以pdf_extract_开头的临时文件
            for filename in os.listdir(temp_dir):
                if filename.startswith('pdf_extract_') and filename.endswith('.pdf'):
                    temp_file_path = os.path.join(temp_dir, filename)
                    try:
                        os.unlink(temp_file_path)
                        cleaned_count += 1
                        logger.debug(f"强制清理临时文件: {temp_file_path}")
                    except Exception as e:
                        logger.debug(f"强制清理临时文件失败 {temp_file_path}: {str(e)}")
            
            if cleaned_count > 0:
                logger.info(f"强制清理了 {cleaned_count} 个临时PDF文件")
        
        except Exception as e:
            logger.warning(f"强制清理临时文件目录失败: {str(e)}")


# 全局处理器实例
_pdf_moonshot_processor = None

def get_pdf_moonshot_processor(moonshot_api_key: str) -> PDFMoonshotProcessor:
    """
    获取PDF Moonshot处理器实例（单例模式）
    
    Args:
        moonshot_api_key: Moonshot API密钥
        
    Returns:
        PDFMoonshotProcessor: 处理器实例
    """
    global _pdf_moonshot_processor
    
    if _pdf_moonshot_processor is None or _pdf_moonshot_processor.moonshot_api_key != moonshot_api_key:
        _pdf_moonshot_processor = PDFMoonshotProcessor(moonshot_api_key)
    
    return _pdf_moonshot_processor


# 便捷函数
def process_pdf_with_moonshot_api(file_path: str, moonshot_api_key: str, custom_prompt: str = "", max_pages: int = 10) -> Optional[str]:
    """
    使用Moonshot API处理PDF文件的便捷函数
    
    Args:
        file_path: PDF文件路径
        moonshot_api_key: Moonshot API密钥
        custom_prompt: 自定义提示词
        max_pages: 最大处理页数
        
    Returns:
        str: 处理结果，如果失败返回None
    """
    processor = get_pdf_moonshot_processor(moonshot_api_key)
    return processor.process_pdf_with_moonshot(file_path, custom_prompt, max_pages)


def process_word_with_moonshot_api(file_path: str, moonshot_api_key: str, custom_prompt: str = "") -> Optional[str]:
    """
    使用Moonshot API处理Word文档的便捷函数
    
    Args:
        file_path: Word文件路径
        moonshot_api_key: Moonshot API密钥
        custom_prompt: 自定义提示词
        
    Returns:
        str: 处理结果，如果失败返回None
    """
    processor = get_pdf_moonshot_processor(moonshot_api_key)
    return processor.process_word_with_moonshot(file_path, custom_prompt)


def process_document_with_moonshot_api(file_path: str, moonshot_api_key: str, custom_prompt: str = "", max_pages: int = 10) -> Optional[str]:
    """
    使用Moonshot API处理文档的通用便捷函数（自动识别文件类型）
    
    Args:
        file_path: 文档文件路径
        moonshot_api_key: Moonshot API密钥
        custom_prompt: 自定义提示词
        max_pages: PDF最大处理页数
        
    Returns:
        str: 处理结果，如果失败返回None
    """
    processor = get_pdf_moonshot_processor(moonshot_api_key)
    return processor.process_document_with_moonshot(file_path, custom_prompt, max_pages)


def is_pdf_suitable_for_moonshot(file_path: str, moonshot_api_key: str) -> bool:
    """
    检查PDF是否适合使用Moonshot处理
    
    Args:
        file_path: PDF文件路径
        moonshot_api_key: Moonshot API密钥
        
    Returns:
        bool: True表示适合使用Moonshot处理
    """
    processor = get_pdf_moonshot_processor(moonshot_api_key)
    if not processor.is_available():
        return False
    
    return processor.is_scanned_or_image_pdf(file_path)


def cleanup_moonshot_temp_files(temp_dir: Optional[str] = None, max_age_minutes: int = 60) -> int:
    """
    清理Moonshot处理器产生的临时文件（全局函数）
    
    Args:
        temp_dir: 临时目录路径，如果为None则使用系统默认临时目录
        max_age_minutes: 文件最大保留时间（分钟），超过此时间的文件将被删除
        
    Returns:
        int: 清理的文件数量
    """
    if temp_dir is None:
        temp_dir = tempfile.gettempdir()
    
    try:
        cleaned_count = 0
        # 查找并删除以pdf_extract_开头的临时文件
        for filename in os.listdir(temp_dir):
            if filename.startswith('pdf_extract_') and filename.endswith('.pdf'):
                temp_file_path = os.path.join(temp_dir, filename)
                try:
                    # 检查文件是否较旧
                    file_age = time.time() - os.path.getctime(temp_file_path)
                    max_age_seconds = max_age_minutes * 60
                    
                    if file_age > max_age_seconds:
                        os.unlink(temp_file_path)
                        cleaned_count += 1
                        logger.debug(f"全局清理旧临时文件: {temp_file_path} (年龄: {file_age/60:.1f}分钟)")
                except Exception as e:
                    logger.debug(f"全局清理临时文件失败 {temp_file_path}: {str(e)}")
        
        if cleaned_count > 0:
            logger.info(f"全局清理了 {cleaned_count} 个旧临时PDF文件")
        
        return cleaned_count
    
    except Exception as e:
        logger.warning(f"全局清理临时文件目录失败: {str(e)}")
        return 0


def cleanup_all_moonshot_temp_files(temp_dir: Optional[str] = None) -> int:
    """
    强制清理所有Moonshot临时文件（全局函数）
    
    Args:
        temp_dir: 临时目录路径，如果为None则使用系统默认临时目录
        
    Returns:
        int: 清理的文件数量
    """
    if temp_dir is None:
        temp_dir = tempfile.gettempdir()
    
    try:
        cleaned_count = 0
        # 查找并删除所有以pdf_extract_开头的临时文件
        for filename in os.listdir(temp_dir):
            if filename.startswith('pdf_extract_') and filename.endswith('.pdf'):
                temp_file_path = os.path.join(temp_dir, filename)
                try:
                    os.unlink(temp_file_path)
                    cleaned_count += 1
                    logger.debug(f"全局强制清理临时文件: {temp_file_path}")
                except Exception as e:
                    logger.debug(f"全局强制清理临时文件失败 {temp_file_path}: {str(e)}")
        
        if cleaned_count > 0:
            logger.info(f"全局强制清理了 {cleaned_count} 个临时PDF文件")
        
        return cleaned_count
    
    except Exception as e:
        logger.warning(f"全局强制清理临时文件目录失败: {str(e)}")
        return 0


def cleanup_moonshot_server_files(moonshot_api_key: str) -> int:
    """
    清理Moonshot服务器上的所有文件（全局函数）
    
    Args:
        moonshot_api_key: Moonshot API密钥
        
    Returns:
        int: 删除的文件数量
    """
    try:
        processor = get_pdf_moonshot_processor(moonshot_api_key)
        if not processor.is_available():
            logger.error("Moonshot处理器不可用，无法清理服务器文件")
            return 0
        
        return processor.cleanup_moonshot_server_files()
        
    except Exception as e:
        logger.error(f"全局清理Moonshot服务器文件失败: {str(e)}")
        return 0
