# -*- coding: utf-8 -*-
"""
FFmpeg下载对话框
"""

import os
from PySide6.QtCore import Qt, Signal, QThread
from PySide6.QtWidgets import (QDialog, QVBoxLayout, QHBoxLayout, QLabel, 
                              QPushButton, QProgressBar, QTextEdit, QMessageBox)
from logger import logger


class FFmpegDownloadThread(QThread):
    """FFmpeg下载线程"""
    
    progress_updated = Signal(int, str)  # 进度更新信号 (百分比, 消息)
    download_finished = Signal(bool, str)  # 下载完成信号 (成功与否, 消息)
    
    def run(self):
        """执行下载"""
        try:
            from lib.video_processor import FFmpegManager
            
            manager = FFmpegManager()
            
            def progress_callback(percent, message):
                self.progress_updated.emit(percent, message)
            
            success, message = manager.download_ffmpeg(progress_callback)
            self.download_finished.emit(success, message)
            
        except Exception as e:
            error_msg = f"下载过程出错: {str(e)}"
            logger.error(error_msg)
            self.download_finished.emit(False, error_msg)


class FFmpegDownloadDialog(QDialog):
    """FFmpeg下载对话框"""
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("视频处理插件下载")
        self.setModal(True)
        self.setFixedSize(600, 400)
        self.download_thread = None
        self.setup_ui()
        self.check_installation_status()
        self.center_on_screen()
        
    def setup_ui(self):
        """设置UI"""
        self.setStyleSheet("""
            QDialog {
                background-color: #1e1e1e;
                color: #ffffff;
            }
            QLabel {
                color: #ffffff;
                font-size: 13px;
            }
            QPushButton {
                background-color: #19AF39;
                color: white;
                border: none;
                padding: 10px 20px;
                border-radius: 4px;
                font-size: 14px;
                font-weight: bold;
            }
            QPushButton:hover {
                background-color: #16973a;
            }
            QPushButton:disabled {
                background-color: #666666;
                color: #999999;
            }
            QProgressBar {
                border: 2px solid #3a3a3a;
                border-radius: 5px;
                background-color: #2a2a2a;
                text-align: center;
                color: #ffffff;
            }
            QProgressBar::chunk {
                background-color: #19AF39;
                border-radius: 3px;
            }
            QTextEdit {
                background-color: #2a2a2a;
                border: 1px solid #3a3a3a;
                border-radius: 4px;
                color: #ffffff;
                font-family: 'Consolas', 'Courier New', monospace;
                font-size: 12px;
            }
        """)
        
        layout = QVBoxLayout(self)
        layout.setSpacing(20)
        
        # 标题
        title_label = QLabel("🎬 视频分类插件 - FFmpeg")
        title_label.setStyleSheet("font-size: 18px; font-weight: bold; color: #19AF39;")
        title_label.setAlignment(Qt.AlignCenter)
        layout.addWidget(title_label)
        
        # 说明文字
        info_label = QLabel("""
FFmpeg是一个强大的开源视频处理工具，用于提取视频关键帧。

<b>功能说明：</b>
• 自动提取视频关键帧（最多5帧）
• 结合多模态AI识别视频内容
• 实现智能视频文件分类

<b>安装方式：</b>
• Windows：点击下载按钮自动安装
• macOS：使用 <code>brew install ffmpeg</code>
• Linux：使用 <code>sudo apt install ffmpeg</code>
        """)
        info_label.setWordWrap(True)
        info_label.setStyleSheet("color: #cccccc; padding: 10px; background-color: #2a2a2a; border-radius: 8px;")
        layout.addWidget(info_label)
        
        # 状态标签
        self.status_label = QLabel("正在检查安装状态...")
        self.status_label.setAlignment(Qt.AlignCenter)
        self.status_label.setStyleSheet("font-size: 14px; padding: 10px;")
        layout.addWidget(self.status_label)
        
        # 进度条
        self.progress_bar = QProgressBar()
        self.progress_bar.setMinimum(0)
        self.progress_bar.setMaximum(100)
        self.progress_bar.setValue(0)
        self.progress_bar.setVisible(False)
        layout.addWidget(self.progress_bar)
        
        # 日志输出
        self.log_text = QTextEdit()
        self.log_text.setReadOnly(True)
        self.log_text.setMaximumHeight(150)
        self.log_text.setVisible(False)
        layout.addWidget(self.log_text)
        
        # 按钮
        button_layout = QHBoxLayout()
        
        self.download_button = QPushButton("📥 下载安装 FFmpeg")
        self.download_button.clicked.connect(self.start_download)
        
        self.close_button = QPushButton("关闭")
        self.close_button.clicked.connect(self.accept)
        self.close_button.setStyleSheet("""
            QPushButton {
                background-color: #6c757d;
            }
            QPushButton:hover {
                background-color: #5a6268;
            }
        """)
        
        button_layout.addWidget(self.download_button)
        button_layout.addWidget(self.close_button)
        layout.addLayout(button_layout)
        
    def check_installation_status(self):
        """检查安装状态"""
        try:
            import os
            from lib.video_processor import FFmpegManager
            
            manager = FFmpegManager()
            
            if manager.is_installed():
                ffmpeg_path = manager.get_ffmpeg_executable()
                
                # 🔥 区分是应用专属安装还是系统安装
                is_app_install = os.path.exists(manager.ffmpeg_path)
                
                if is_app_install:
                    # 应用专属安装
                    self.status_label.setText(
                        f"✅ FFmpeg已安装（应用专属）\n"
                        f"路径: {ffmpeg_path}"
                    )
                else:
                    # 系统PATH中找到
                    self.status_label.setText(
                        f"✅ 检测到系统FFmpeg\n"
                        f"路径: {ffmpeg_path}\n"
                        f"来源: 系统环境（无需重复安装）"
                    )
                
                self.status_label.setStyleSheet("font-size: 13px; padding: 10px; color: #19AF39;")
                self.download_button.setEnabled(False)
                self.download_button.setText("已安装")
            else:
                self.status_label.setText("❌ FFmpeg未安装\n视频分类功能不可用")
                self.status_label.setStyleSheet("font-size: 14px; padding: 10px; color: #ff6b6b;")
                self.download_button.setEnabled(True)
                
        except Exception as e:
            logger.error(f"检查FFmpeg安装状态失败: {str(e)}")
            self.status_label.setText(f"检查状态失败: {str(e)}")
    
    def start_download(self):
        """开始下载"""
        import sys
        
        # macOS和Linux提示手动安装
        if sys.platform != 'win32':
            if sys.platform == 'darwin':
                msg = "macOS用户请使用Homebrew安装：\n\nbrew install ffmpeg"
            else:
                msg = "Linux用户请使用包管理器安装：\n\nsudo apt install ffmpeg"
            
            QMessageBox.information(self, "提示", msg)
            return
        
        # Windows自动下载
        self.download_button.setEnabled(False)
        self.progress_bar.setVisible(True)
        self.log_text.setVisible(True)
        self.log_text.append("开始下载FFmpeg...")
        
        # 创建下载线程
        self.download_thread = FFmpegDownloadThread()
        self.download_thread.progress_updated.connect(self.on_progress_updated)
        self.download_thread.download_finished.connect(self.on_download_finished)
        self.download_thread.start()
    
    def on_progress_updated(self, percent: int, message: str):
        """进度更新"""
        self.progress_bar.setValue(percent)
        self.log_text.append(message)
        # 自动滚动到底部
        self.log_text.verticalScrollBar().setValue(
            self.log_text.verticalScrollBar().maximum()
        )
    
    def on_download_finished(self, success: bool, message: str):
        """下载完成"""
        self.log_text.append(f"\n{'='*50}")
        self.log_text.append(message)
        
        if success:
            self.status_label.setText("✅ FFmpeg安装成功！")
            self.status_label.setStyleSheet("font-size: 14px; padding: 10px; color: #19AF39;")
            self.download_button.setText("已安装")
            QMessageBox.information(self, "成功", "FFmpeg安装成功！\n现在可以使用视频分类功能了。")
        else:
            self.status_label.setText(f"❌ 安装失败\n{message}")
            self.status_label.setStyleSheet("font-size: 14px; padding: 10px; color: #ff6b6b;")
            self.download_button.setEnabled(True)
            QMessageBox.warning(self, "失败", f"FFmpeg安装失败：\n{message}")
        
        self.download_thread = None
    
    def center_on_screen(self):
        """居中显示"""
        from PySide6.QtWidgets import QApplication
        screen = QApplication.primaryScreen()
        screen_geometry = screen.geometry()
        x = (screen_geometry.width() - self.width()) // 2
        y = (screen_geometry.height() - self.height()) // 2
        self.move(x, y)

