# -*- coding: utf-8 -*-
"""
🤖 本地模型管理对话框 - 下载、加载和管理本地大语言模型（无需Ollama）
支持文本模型（文件分类）和视觉模型（图片识别）
"""

import os
import sys
import threading
from PySide6.QtCore import Qt, Signal, QTimer, QThread
from PySide6.QtWidgets import (QDialog, QVBoxLayout, QHBoxLayout, QLabel, QPushButton,
                              QProgressBar, QFrame, QListWidget, QListWidgetItem,
                              QScrollArea, QWidget, QMessageBox, QSpinBox,
                              QAbstractItemView, QTabWidget)
from PySide6.QtGui import QIcon, QFont, QColor
from logger import logger


class DownloadThread(QThread):
    """下载线程（文本模型）"""
    progress = Signal(int, int, float)  # downloaded, total, speed
    finished = Signal(bool, str)  # success, message
    
    def __init__(self, model_id: str):
        super().__init__()
        self.model_id = model_id
        self._downloader = None
    
    def run(self):
        from lib.local_model import ModelDownloader
        
        def on_progress(downloaded, total, speed):
            self.progress.emit(downloaded, total, speed)
        
        def on_complete(success, message):
            self.finished.emit(success, message)
        
        self._downloader = ModelDownloader(
            self.model_id,
            progress_callback=on_progress,
            complete_callback=on_complete
        )
        self._downloader._download_thread()
    
    def stop(self):
        if self._downloader:
            self._downloader.stop()


class VisionDownloadThread(QThread):
    """下载线程（视觉模型 - 需下载两个文件）"""
    progress = Signal(int, int, float)
    finished = Signal(bool, str)
    
    def __init__(self, model_id: str):
        super().__init__()
        self.model_id = model_id
        self._downloader = None
    
    def run(self):
        from lib.local_model import VisionModelDownloader
        
        def on_progress(downloaded, total, speed):
            self.progress.emit(downloaded, total, speed)
        
        def on_complete(success, message):
            self.finished.emit(success, message)
        
        self._downloader = VisionModelDownloader(
            self.model_id,
            progress_callback=on_progress,
            complete_callback=on_complete
        )
        self._downloader._download_thread()
    
    def stop(self):
        if self._downloader:
            self._downloader.stop()


class LoadModelThread(QThread):
    """加载模型线程（文本模型）"""
    progress = Signal(str)
    finished = Signal(bool, str)
    
    def __init__(self, model_path: str, n_ctx: int = 4096, n_threads: int = 4, n_gpu_layers: int = 0):
        super().__init__()
        self.model_path = model_path
        self.n_ctx = n_ctx
        self.n_threads = n_threads
        self.n_gpu_layers = n_gpu_layers
    
    def run(self):
        from lib.local_model import get_local_model
        
        last_message = [""]  # 用列表以便在闭包中修改
        
        def on_progress(msg):
            last_message[0] = msg
            self.progress.emit(msg)
        
        try:
            model = get_local_model()
            success = model.load(
                self.model_path,
                n_ctx=self.n_ctx,
                n_gpu_layers=self.n_gpu_layers,
                progress_callback=on_progress,
                n_threads=self.n_threads,
            )
            if success:
                self.finished.emit(True, "模型加载成功！")
            else:
                # 传递具体的错误信息（由 progress_callback 记录）
                error_msg = last_message[0] if last_message[0] else "模型加载失败"
                self.finished.emit(False, error_msg)
        except Exception as e:
            self.finished.emit(False, f"加载失败: {str(e)}")


class LoadVisionModelThread(QThread):
    """加载视觉模型线程"""
    progress = Signal(str)
    finished = Signal(bool, str)
    
    def __init__(self, model_path: str, mmproj_path: str, handler_type: str = "llava15",
                 n_ctx: int = 4096, n_threads: int = 4, n_gpu_layers: int = 0):
        super().__init__()
        self.model_path = model_path
        self.mmproj_path = mmproj_path
        self.handler_type = handler_type
        self.n_ctx = n_ctx
        self.n_threads = n_threads
        self.n_gpu_layers = n_gpu_layers
    
    def run(self):
        from lib.local_model import get_vision_model
        
        def on_progress(msg):
            self.progress.emit(msg)
        
        try:
            model = get_vision_model()
            success = model.load(
                self.model_path,
                self.mmproj_path,
                handler_type=self.handler_type,
                n_ctx=self.n_ctx,
                n_gpu_layers=self.n_gpu_layers,
                n_threads=self.n_threads,
                progress_callback=on_progress
            )
            if success:
                self.finished.emit(True, "视觉模型加载成功！")
            else:
                self.finished.emit(False, "视觉模型加载失败")
        except Exception as e:
            self.finished.emit(False, f"加载失败: {str(e)}")


class TestModelThread(QThread):
    """测试模型线程"""
    finished = Signal(bool, str)
    
    def run(self):
        try:
            from lib.local_model import test_builtin_model
            success, message = test_builtin_model()
            self.finished.emit(success, message)
        except Exception as e:
            self.finished.emit(False, f"测试失败: {str(e)}")


class TestVisionModelThread(QThread):
    """测试视觉模型线程"""
    finished = Signal(bool, str)
    
    def run(self):
        try:
            from lib.local_model import test_builtin_vision_model
            success, message = test_builtin_vision_model()
            self.finished.emit(success, message)
        except Exception as e:
            self.finished.emit(False, f"测试失败: {str(e)}")


class LocalModelDialog(QDialog):
    """本地模型管理对话框"""
    
    model_loaded = Signal(str)  # text model path
    vision_model_loaded = Signal(str)  # vision model path
    
    def __init__(self, parent=None):
        super().__init__(parent)
        self.setWindowTitle("本地模型管理")
        self.setModal(True)
        self.setMinimumSize(620, 780)
        self.resize(660, 850)
        
        self._download_thread = None
        self._load_thread = None
        self._current_download_model_id = None
        self._current_download_is_vision = False
        
        self.setup_ui()
        try:
            self.refresh_text_models()
        except Exception as e:
            logger.warning(f"刷新文本模型列表失败: {e}")
        try:
            self.refresh_vision_models()
        except Exception as e:
            logger.warning(f"刷新视觉模型列表失败: {e}")
        self.check_llama_cpp()
        self.center_on_screen()
    
    def setup_ui(self):
        self.setStyleSheet("""
            QDialog { background-color: #1e1e1e; color: #ffffff; }
            QLabel { color: #ffffff; }
            QTabWidget::pane { border: 1px solid #3a3a3a; background-color: #1e1e1e; }
            QTabBar::tab {
                background-color: #2a2a2a; color: #cccccc; border: 1px solid #3a3a3a;
                padding: 6px 16px; margin-right: 2px; border-top-left-radius: 4px; border-top-right-radius: 4px;
            }
            QTabBar::tab:selected { background-color: #1e1e1e; color: #19AF39; border-bottom: none; font-weight: bold; }
            QTabBar::tab:hover { background-color: #333333; }
            QListWidget {
                background-color: #2a2a2a; border: 1px solid #3a3a3a; border-radius: 4px;
                color: #ffffff; font-size: 13px; padding: 2px;
            }
            QListWidget::item { padding: 8px 12px; border-bottom: 1px solid #333333; }
            QListWidget::item:selected { background-color: #3a3a3a; color: #ffffff; }
            QListWidget::item:hover { background-color: #333333; }
            QSpinBox {
                background-color: #2a2a2a; border: 1px solid #3a3a3a; border-radius: 4px;
                color: #ffffff; padding: 2px 6px; font-size: 12px; min-width: 70px;
            }
            QSpinBox::up-button, QSpinBox::down-button { background-color: #3a3a3a; border: none; width: 14px; }
            QScrollArea { border: none; background-color: #1e1e1e; }
            QScrollBar:vertical { background-color: #2a2a2a; width: 8px; border-radius: 4px; }
            QScrollBar::handle:vertical { background-color: #19AF39; border-radius: 4px; min-height: 20px; }
            QScrollBar::add-line:vertical, QScrollBar::sub-line:vertical { height: 0px; }
        """)
        
        outer_layout = QVBoxLayout(self)
        outer_layout.setContentsMargins(0, 0, 0, 0)
        outer_layout.setSpacing(0)
        
        scroll_area = QScrollArea()
        scroll_area.setWidgetResizable(True)
        scroll_area.setHorizontalScrollBarPolicy(Qt.ScrollBarAlwaysOff)
        
        scroll_widget = QWidget()
        scroll_widget.setStyleSheet("background-color: #1e1e1e;")
        layout = QVBoxLayout(scroll_widget)
        layout.setContentsMargins(24, 20, 24, 16)
        layout.setSpacing(6)
        
        # ═══ 标题 ═══
        title_label = QLabel("🤖 本地模型管理")
        title_label.setStyleSheet("font-size: 18px; font-weight: bold; color: #19AF39;")
        layout.addWidget(title_label)
        
        info_label = QLabel(
            "下载本地AI模型，无需网络即可进行文件分类和图片识别。"
            "首次使用需下载模型文件，下载后可离线使用。"
        )
        info_label.setWordWrap(True)
        info_label.setStyleSheet("color: #aaaaaa; font-size: 11px; margin-bottom: 4px;")
        layout.addWidget(info_label)
        
        # ═══ 当前状态 ═══
        section1_label = QLabel("🔌 当前状态")
        section1_label.setStyleSheet("font-size: 13px; font-weight: bold; color: #ffffff; margin-top: 6px;")
        layout.addWidget(section1_label)
        
        status_frame = QFrame()
        status_frame.setStyleSheet("""
            QFrame { background-color: #252525; border: 1px solid #3a3a3a; border-radius: 6px; padding: 8px; }
        """)
        status_inner = QVBoxLayout(status_frame)
        status_inner.setContentsMargins(12, 8, 12, 8)
        status_inner.setSpacing(4)
        
        # 文本模型状态
        text_status_row = QHBoxLayout()
        text_status_row.setSpacing(6)
        text_lbl = QLabel("📝 文本模型:")
        text_lbl.setStyleSheet("font-size: 12px; color: #cccccc;")
        text_lbl.setFixedWidth(85)
        text_status_row.addWidget(text_lbl)
        self.text_status_label = QLabel("⚠ 未加载")
        self.text_status_label.setStyleSheet("color: #ffcc00; font-size: 12px; font-weight: bold;")
        text_status_row.addWidget(self.text_status_label)
        text_status_row.addStretch()
        status_inner.addLayout(text_status_row)
        
        # 视觉模型状态
        vision_status_row = QHBoxLayout()
        vision_status_row.setSpacing(6)
        vision_lbl = QLabel("📷 视觉模型:")
        vision_lbl.setStyleSheet("font-size: 12px; color: #cccccc;")
        vision_lbl.setFixedWidth(85)
        vision_status_row.addWidget(vision_lbl)
        self.vision_status_label = QLabel("⚠ 未加载")
        self.vision_status_label.setStyleSheet("color: #ffcc00; font-size: 12px; font-weight: bold;")
        vision_status_row.addWidget(self.vision_status_label)
        vision_status_row.addStretch()
        status_inner.addLayout(vision_status_row)
        
        layout.addWidget(status_frame)
        
        # ═══ 模型标签页 ═══
        self.tab_widget = QTabWidget()
        self.tab_widget.setMinimumHeight(320)
        
        # --- Tab 1: 文本模型 ---
        text_tab = QWidget()
        text_tab.setStyleSheet("background-color: #1e1e1e;")
        text_layout = QVBoxLayout(text_tab)
        text_layout.setContentsMargins(8, 8, 8, 8)
        text_layout.setSpacing(6)
        
        self.text_model_list = QListWidget()
        self.text_model_list.setSelectionMode(QAbstractItemView.SingleSelection)
        self.text_model_list.setMinimumHeight(120)
        text_layout.addWidget(self.text_model_list)
        
        text_btn_row = QHBoxLayout()
        text_btn_row.setSpacing(6)
        
        self.text_download_btn = self._make_btn("⬇ 下载", "#19AF39", "#16973a")
        self.text_download_btn.clicked.connect(self._on_text_download)
        text_btn_row.addWidget(self.text_download_btn)
        
        self.text_load_btn = self._make_btn("🚀 加载", "#007bff", "#0069d9")
        self.text_load_btn.clicked.connect(self._on_text_load)
        text_btn_row.addWidget(self.text_load_btn)
        
        self.text_unload_btn = self._make_btn("📤 卸载", "#6c757d", "#5a6268")
        self.text_unload_btn.setEnabled(False)
        self.text_unload_btn.clicked.connect(self._on_text_unload)
        text_btn_row.addWidget(self.text_unload_btn)
        
        self.text_test_btn = self._make_btn("🔗 测试", "#17a2b8", "#138496")
        self.text_test_btn.setEnabled(False)
        self.text_test_btn.clicked.connect(self._on_text_test)
        text_btn_row.addWidget(self.text_test_btn)
        
        self.text_delete_btn = self._make_btn("🗑 删除", "#dc3545", "#c82333")
        self.text_delete_btn.clicked.connect(self._on_text_delete)
        text_btn_row.addWidget(self.text_delete_btn)
        
        text_btn_row.addStretch()
        text_layout.addLayout(text_btn_row)
        
        self.tab_widget.addTab(text_tab, "📝 文本模型")
        
        # --- Tab 2: 视觉模型 ---
        vision_tab = QWidget()
        vision_tab.setStyleSheet("background-color: #1e1e1e;")
        vision_layout = QVBoxLayout(vision_tab)
        vision_layout.setContentsMargins(8, 8, 8, 8)
        vision_layout.setSpacing(6)
        
        vision_info = QLabel("📷 视觉模型用于识别图片中的文字，加载后遇到图片时将自动使用。")
        vision_info.setWordWrap(True)
        vision_info.setStyleSheet("color: #aaaaaa; font-size: 11px;")
        vision_layout.addWidget(vision_info)
        
        self.vision_model_list = QListWidget()
        self.vision_model_list.setSelectionMode(QAbstractItemView.SingleSelection)
        self.vision_model_list.setMinimumHeight(120)
        vision_layout.addWidget(self.vision_model_list)
        
        vision_btn_row = QHBoxLayout()
        vision_btn_row.setSpacing(6)
        
        self.vision_download_btn = self._make_btn("⬇ 下载", "#19AF39", "#16973a")
        self.vision_download_btn.clicked.connect(self._on_vision_download)
        vision_btn_row.addWidget(self.vision_download_btn)
        
        self.vision_load_btn = self._make_btn("🚀 加载", "#007bff", "#0069d9")
        self.vision_load_btn.clicked.connect(self._on_vision_load)
        vision_btn_row.addWidget(self.vision_load_btn)
        
        self.vision_unload_btn = self._make_btn("📤 卸载", "#6c757d", "#5a6268")
        self.vision_unload_btn.setEnabled(False)
        self.vision_unload_btn.clicked.connect(self._on_vision_unload)
        vision_btn_row.addWidget(self.vision_unload_btn)
        
        self.vision_test_btn = self._make_btn("🔗 测试", "#17a2b8", "#138496")
        self.vision_test_btn.setEnabled(False)
        self.vision_test_btn.clicked.connect(self._on_vision_test)
        vision_btn_row.addWidget(self.vision_test_btn)
        
        self.vision_delete_btn = self._make_btn("🗑 删除", "#dc3545", "#c82333")
        self.vision_delete_btn.clicked.connect(self._on_vision_delete)
        vision_btn_row.addWidget(self.vision_delete_btn)
        
        vision_btn_row.addStretch()
        vision_layout.addLayout(vision_btn_row)
        
        self.tab_widget.addTab(vision_tab, "📷 视觉模型")
        
        layout.addWidget(self.tab_widget)
        
        # ═══ 下载进度 ═══
        self.progress_frame = QFrame()
        self.progress_frame.setStyleSheet("""
            QFrame { background-color: #252525; border: 1px solid #3a3a3a; border-radius: 6px; }
        """)
        progress_layout = QVBoxLayout(self.progress_frame)
        progress_layout.setContentsMargins(12, 8, 12, 8)
        progress_layout.setSpacing(4)
        
        self.progress_label = QLabel("等待下载...")
        self.progress_label.setStyleSheet("color: #aaaaaa; font-size: 11px;")
        progress_layout.addWidget(self.progress_label)
        
        self.progress_bar = QProgressBar()
        self.progress_bar.setFixedHeight(16)
        self.progress_bar.setStyleSheet("""
            QProgressBar { border: 1px solid #3a3a3a; border-radius: 3px; background-color: #2a2a2a;
                          text-align: center; color: white; font-size: 10px; }
            QProgressBar::chunk { background-color: #19AF39; border-radius: 2px; }
        """)
        self.progress_bar.setValue(0)
        progress_layout.addWidget(self.progress_bar)
        
        self.cancel_btn = QPushButton("✕ 取消下载")
        self.cancel_btn.setEnabled(False)
        self.cancel_btn.setStyleSheet("""
            QPushButton { background-color: #dc3545; color: white; border: none;
                         padding: 4px 12px; border-radius: 4px; font-size: 11px; }
            QPushButton:hover { background-color: #c82333; }
            QPushButton:disabled { background-color: #444444; color: #666666; }
        """)
        self.cancel_btn.clicked.connect(self._cancel_download)
        progress_layout.addWidget(self.cancel_btn)
        
        self.progress_frame.hide()
        layout.addWidget(self.progress_frame)
        
        # ═══ 高级设置 ═══
        section3_label = QLabel("⚙ 高级设置")
        section3_label.setStyleSheet("font-size: 13px; font-weight: bold; color: #ffffff; margin-top: 6px;")
        layout.addWidget(section3_label)
        
        settings_frame = QFrame()
        settings_frame.setStyleSheet("""
            QFrame { background-color: #252525; border: 1px solid #3a3a3a; border-radius: 6px; }
        """)
        settings_inner = QVBoxLayout(settings_frame)
        settings_inner.setContentsMargins(12, 8, 12, 8)
        settings_inner.setSpacing(4)
        
        settings_row = QHBoxLayout()
        settings_row.setSpacing(12)
        
        ctx_label = QLabel("上下文长度:")
        ctx_label.setStyleSheet("font-size: 12px;")
        settings_row.addWidget(ctx_label)
        self.spin_n_ctx = QSpinBox()
        self.spin_n_ctx.setRange(512, 32768)
        self.spin_n_ctx.setValue(4096)
        self.spin_n_ctx.setSingleStep(512)
        self.spin_n_ctx.setFixedWidth(80)
        settings_row.addWidget(self.spin_n_ctx)
        
        settings_row.addSpacing(8)
        threads_label = QLabel("CPU线程数:")
        threads_label.setStyleSheet("font-size: 12px;")
        settings_row.addWidget(threads_label)
        self.spin_n_threads = QSpinBox()
        self.spin_n_threads.setRange(1, 32)
        self.spin_n_threads.setValue(4)
        self.spin_n_threads.setFixedWidth(60)
        settings_row.addWidget(self.spin_n_threads)
        
        settings_row.addSpacing(8)
        gpu_label = QLabel("GPU层数:")
        gpu_label.setStyleSheet("font-size: 12px;")
        settings_row.addWidget(gpu_label)
        self.spin_n_gpu = QSpinBox()
        self.spin_n_gpu.setRange(-1, 100)
        self.spin_n_gpu.setToolTip("-1=全部用GPU（自动），0=纯CPU，1-100=指定GPU层数")
        self.spin_n_gpu.setFixedWidth(60)
        
        # 🎮 自动检测 GPU，设置默认值
        try:
            from lib.local_model import get_gpu_info
            gpu_info = get_gpu_info()
            self.spin_n_gpu.setValue(gpu_info["recommended_gpu_layers"])
            logger.info(f"🎮 GPU 自动检测: {gpu_info['message']}")
        except Exception:
            self.spin_n_gpu.setValue(0)
        
        settings_row.addWidget(self.spin_n_gpu)
        
        # GPU 状态指示
        self.gpu_status_label = QLabel("")
        self.gpu_status_label.setStyleSheet("font-size: 11px;")
        try:
            from lib.local_model import get_gpu_info
            gpu_info = get_gpu_info()
            if gpu_info["has_nvidia_gpu"] and gpu_info["recommended_gpu_layers"] != 0:
                self.gpu_status_label.setText(f"🎮 {gpu_info['gpu_name']}")
                self.gpu_status_label.setStyleSheet("color: #28a745; font-size: 11px;")
            elif gpu_info["has_nvidia_gpu"]:
                self.gpu_status_label.setText(f"⚠ {gpu_info['gpu_name']}(需CUDA)")
                self.gpu_status_label.setStyleSheet("color: #ffc107; font-size: 11px;")
            else:
                self.gpu_status_label.setText("CPU模式")
                self.gpu_status_label.setStyleSheet("color: #6c757d; font-size: 11px;")
        except Exception:
            self.gpu_status_label.setText("")
        settings_row.addWidget(self.gpu_status_label)
        
        settings_row.addStretch()
        
        save_btn = self._make_btn("💾 保存", "#007bff", "#0069d9")
        save_btn.clicked.connect(self._save_settings)
        settings_row.addWidget(save_btn)
        
        settings_inner.addLayout(settings_row)
        layout.addWidget(settings_frame)
        
        # ═══ 使用说明 ═══
        help_text = QLabel(
            "📝 文本模型：用于文件分类和重命名（推荐 Qwen2.5-1.5B）\n"
            "📷 视觉模型：用于图片文字识别，加载后遇到图片时自动使用（推荐 MiniCPM-V 2.6）\n"
            "⚠ 同时加载两个模型需要较大内存，请根据电脑配置选择"
        )
        help_text.setWordWrap(True)
        help_text.setStyleSheet("color: #999999; font-size: 11px; padding: 4px 8px; margin-top: 4px;")
        layout.addWidget(help_text)
        
        layout.addStretch(1)
        scroll_area.setWidget(scroll_widget)
        outer_layout.addWidget(scroll_area)
        
        # ═══ 底部按钮 ═══
        bottom_bar = QFrame()
        bottom_bar.setFixedHeight(48)
        bottom_bar.setStyleSheet("QFrame { background-color: #1e1e1e; border-top: 1px solid #3a3a3a; }")
        bottom_layout = QHBoxLayout(bottom_bar)
        bottom_layout.setContentsMargins(24, 8, 24, 8)
        bottom_layout.addStretch()
        
        close_btn = QPushButton("关闭")
        close_btn.setStyleSheet("""
            QPushButton { background-color: #6c757d; color: white; border: none;
                         padding: 6px 20px; border-radius: 4px; font-size: 12px; }
            QPushButton:hover { background-color: #5a6268; }
        """)
        close_btn.clicked.connect(self.close)
        bottom_layout.addWidget(close_btn)
        
        outer_layout.addWidget(bottom_bar)
        self._load_settings()
    
    def _make_btn(self, text, bg_color, hover_color):
        """创建统一风格的按钮"""
        btn = QPushButton(text)
        btn.setStyleSheet(f"""
            QPushButton {{
                background-color: {bg_color}; color: white; border: none;
                padding: 5px 10px; border-radius: 4px; font-size: 11px; font-weight: bold;
            }}
            QPushButton:hover {{ background-color: {hover_color}; }}
            QPushButton:disabled {{ background-color: #444444; color: #666666; }}
        """)
        return btn
    
    # ═══════════════════════════════════════════
    # 文本模型操作
    # ═══════════════════════════════════════════
    
    def refresh_text_models(self):
        """刷新文本模型列表"""
        from lib.local_model import get_available_models, get_local_model
        
        self.text_model_list.clear()
        self._text_models_data = {}
        
        for model_info in get_available_models():
            model_id = model_info["id"]
            display = model_info["name"]
            if model_info.get("recommended"):
                display += " ⭐"
            if model_info.get("downloaded"):
                display += "  ✅已下载"
            else:
                display += f"  ({model_info['size_mb']}MB)"
            
            item = QListWidgetItem(display)
            item.setData(Qt.UserRole, model_id)
            item.setForeground(Qt.white if model_info.get("downloaded") else QColor("#cccccc"))
            self.text_model_list.addItem(item)
            self._text_models_data[model_id] = model_info
        
        if self.text_model_list.count() > 0:
            self.text_model_list.setCurrentRow(0)
        
        # 更新文本模型状态
        model = get_local_model()
        if model.is_loaded():
            model_path = model.get_model_path()
            name = os.path.basename(model_path) if model_path else "未知"
            self.text_status_label.setText(f"✅ {name}")
            self.text_status_label.setStyleSheet("color: #19AF39; font-size: 12px; font-weight: bold;")
            self.text_unload_btn.setEnabled(True)
            self.text_test_btn.setEnabled(True)
        else:
            self.text_status_label.setText("⚠ 未加载")
            self.text_status_label.setStyleSheet("color: #ffcc00; font-size: 12px; font-weight: bold;")
            self.text_unload_btn.setEnabled(False)
            self.text_test_btn.setEnabled(False)
    
    def refresh_vision_models(self):
        """刷新视觉模型列表"""
        from lib.local_model import get_available_vision_models, get_vision_model
        
        self.vision_model_list.clear()
        self._vision_models_data = {}
        
        for model_info in get_available_vision_models():
            model_id = model_info["id"]
            display = model_info["name"]
            if model_info.get("recommended"):
                display += " ⭐"
            if model_info.get("downloaded"):
                display += "  ✅已下载"
            else:
                display += f"  ({model_info['size_mb']}MB)"
            
            item = QListWidgetItem(display)
            item.setData(Qt.UserRole, model_id)
            item.setForeground(Qt.white if model_info.get("downloaded") else QColor("#cccccc"))
            self.vision_model_list.addItem(item)
            self._vision_models_data[model_id] = model_info
        
        if self.vision_model_list.count() > 0:
            self.vision_model_list.setCurrentRow(0)
        
        # 更新视觉模型状态
        vision_model = get_vision_model()
        if vision_model.is_loaded():
            model_path = vision_model.get_model_path()
            name = os.path.basename(model_path) if model_path else "未知"
            self.vision_status_label.setText(f"✅ {name}")
            self.vision_status_label.setStyleSheet("color: #19AF39; font-size: 12px; font-weight: bold;")
            self.vision_unload_btn.setEnabled(True)
            self.vision_test_btn.setEnabled(True)
        else:
            self.vision_status_label.setText("⚠ 未加载")
            self.vision_status_label.setStyleSheet("color: #ffcc00; font-size: 12px; font-weight: bold;")
            self.vision_unload_btn.setEnabled(False)
            self.vision_test_btn.setEnabled(False)
    
    def _get_selected_text_model(self):
        item = self.text_model_list.currentItem()
        if not item:
            return None, None
        mid = item.data(Qt.UserRole)
        return mid, self._text_models_data.get(mid)
    
    def _get_selected_vision_model(self):
        item = self.vision_model_list.currentItem()
        if not item:
            return None, None
        mid = item.data(Qt.UserRole)
        return mid, self._vision_models_data.get(mid)
    
    def _on_text_download(self):
        mid, info = self._get_selected_text_model()
        if not info:
            QMessageBox.warning(self, "提示", "请先选择一个文本模型")
            return
        if info.get("downloaded"):
            QMessageBox.information(self, "提示", "该模型已下载，可直接加载。")
            return
        self._start_text_download(mid)
    
    def _on_text_load(self):
        mid, info = self._get_selected_text_model()
        if not info:
            QMessageBox.warning(self, "提示", "请先选择一个文本模型")
            return
        if not info.get("downloaded"):
            QMessageBox.warning(self, "提示", "请先下载该模型")
            return
        path = info.get("path", "")
        if not path or not os.path.exists(path):
            QMessageBox.warning(self, "错误", f"模型文件不存在: {path}")
            return
        self._load_text_model(mid, path)
    
    def _on_text_unload(self):
        from lib.local_model import get_local_model
        reply = QMessageBox.question(self, "确认卸载", "确定要卸载文本模型吗？", QMessageBox.Yes | QMessageBox.No)
        if reply == QMessageBox.Yes:
            get_local_model().unload()
            self.text_status_label.setText("⚠ 未加载")
            self.text_status_label.setStyleSheet("color: #ffcc00; font-size: 12px; font-weight: bold;")
            self.text_unload_btn.setEnabled(False)
            self.text_test_btn.setEnabled(False)
    
    def _on_text_test(self):
        self.text_test_btn.setEnabled(False)
        self.text_test_btn.setText("测试中...")
        self._test_thread = TestModelThread()
        self._test_thread.finished.connect(self._on_text_test_finished)
        self._test_thread.start()
    
    def _on_text_test_finished(self, success, message):
        self.text_test_btn.setEnabled(True)
        self.text_test_btn.setText("🔗 测试")
        if success:
            QMessageBox.information(self, "测试成功", message)
        else:
            QMessageBox.warning(self, "测试失败", message)
    
    def _on_text_delete(self):
        mid, info = self._get_selected_text_model()
        if not info or not info.get("downloaded"):
            QMessageBox.warning(self, "提示", "该模型未下载")
            return
        path = info.get("path", "")
        reply = QMessageBox.question(self, "确认删除",
                                     f"确定要删除 {os.path.basename(path)} 吗？\n此操作不可恢复！",
                                     QMessageBox.Yes | QMessageBox.No)
        if reply == QMessageBox.Yes:
            try:
                from lib.local_model import get_local_model
                model = get_local_model()
                if model.is_loaded() and model.get_model_path() == path:
                    model.unload()
                os.remove(path)
                QMessageBox.information(self, "成功", "模型文件已删除。")
                self.refresh_text_models()
            except Exception as e:
                QMessageBox.warning(self, "失败", f"删除失败: {str(e)}")
    
    def _start_text_download(self, model_id):
        if self._download_thread and self._download_thread.isRunning():
            QMessageBox.warning(self, "下载中", "已有下载任务正在进行")
            return
        
        from lib.local_model import AVAILABLE_MODELS
        info = AVAILABLE_MODELS.get(model_id, {})
        reply = QMessageBox.question(self, "确认下载",
                                     f"下载 {info.get('name', model_id)}？\n大小约 {info.get('size_mb', 0)}MB",
                                     QMessageBox.Yes | QMessageBox.No)
        if reply != QMessageBox.Yes:
            return
        
        self.progress_frame.show()
        self.progress_bar.setValue(0)
        self.progress_label.setText(f"正在下载 {info.get('name', model_id)}...")
        self.cancel_btn.setEnabled(True)
        self._current_download_model_id = model_id
        self._current_download_is_vision = False
        
        self._download_thread = DownloadThread(model_id)
        self._download_thread.progress.connect(self._on_download_progress)
        self._download_thread.finished.connect(self._on_text_download_finished)
        self._download_thread.start()
    
    def _on_text_download_finished(self, success, message):
        self.cancel_btn.setEnabled(False)
        mid = self._current_download_model_id
        self._current_download_model_id = None
        
        if success:
            self.progress_label.setText("✅ 下载完成！正在自动加载...")
            self.progress_bar.setValue(100)
            self.refresh_text_models()
            if mid and message and os.path.exists(message):
                QTimer.singleShot(1000, lambda: self._load_text_model(mid, message))
            else:
                QMessageBox.information(self, "完成", "下载成功！点击「加载」开始使用。")
        else:
            self.progress_label.setText(f"❌ {message}")
            if "取消" not in message:
                QMessageBox.warning(self, "下载失败", message)
        
        QTimer.singleShot(5000, lambda: self.progress_frame.hide())
    
    def _load_text_model(self, model_id, model_path):
        from lib.local_model import is_llama_cpp_available
        if not is_llama_cpp_available():
            QMessageBox.warning(self, "依赖缺失", "llama-cpp-python 未安装")
            return
        if self._load_thread and self._load_thread.isRunning():
            QMessageBox.warning(self, "加载中", "模型正在加载中...")
            return
        
        self.text_status_label.setText("⏳ 加载中...")
        self.text_status_label.setStyleSheet("color: #ffcc00; font-size: 12px; font-weight: bold;")
        self.text_load_btn.setEnabled(False)
        
        self._load_thread = LoadModelThread(model_path, self.spin_n_ctx.value(),
                                            self.spin_n_threads.value(), self.spin_n_gpu.value())
        self._load_thread.progress.connect(lambda msg: self.text_status_label.setText(f"⏳ {msg}"))
        self._load_thread.finished.connect(self._on_text_load_finished)
        self._load_thread.start()
    
    def _on_text_load_finished(self, success, message):
        self.text_load_btn.setEnabled(True)
        if success:
            from lib.local_model import get_local_model
            model = get_local_model()
            path = model.get_model_path()
            name = os.path.basename(path) if path else "未知"
            self.text_status_label.setText(f"✅ {name}")
            self.text_status_label.setStyleSheet("color: #19AF39; font-size: 12px; font-weight: bold;")
            self.text_unload_btn.setEnabled(True)
            self.text_test_btn.setEnabled(True)
            QMessageBox.information(self, "加载成功", "文本模型加载成功！\n在「模型选择」中选择「本地模型」即可使用。")
            self.model_loaded.emit(path)
        else:
            # 状态栏只显示简短错误，弹窗显示完整信息
            short_msg = message.split('\n')[0] if '\n' in message else message
            # 去掉重复的 "加载失败: " 前缀用于状态栏显示
            label_msg = short_msg.replace("加载失败: ", "") if short_msg.startswith("加载失败: ") else short_msg
            self.text_status_label.setText(f"❌ {label_msg}")
            self.text_status_label.setStyleSheet("color: #ff6b6b; font-size: 12px; font-weight: bold;")
            QMessageBox.warning(self, "加载失败", message)
    
    # ═══════════════════════════════════════════
    # 视觉模型操作
    # ═══════════════════════════════════════════
    
    def _on_vision_download(self):
        mid, info = self._get_selected_vision_model()
        if not info:
            QMessageBox.warning(self, "提示", "请先选择一个视觉模型")
            return
        if info.get("downloaded"):
            QMessageBox.information(self, "提示", "该视觉模型已下载，可直接加载。")
            return
        self._start_vision_download(mid)
    
    def _on_vision_load(self):
        mid, info = self._get_selected_vision_model()
        if not info:
            QMessageBox.warning(self, "提示", "请先选择一个视觉模型")
            return
        if not info.get("downloaded"):
            QMessageBox.warning(self, "提示", "请先下载该视觉模型")
            return
        
        model_path = info.get("model_path", "")
        mmproj_path = info.get("mmproj_path", "")
        handler_type = info.get("chat_handler_type", "llava15")
        
        if not model_path or not os.path.exists(model_path):
            QMessageBox.warning(self, "错误", f"模型文件不存在: {model_path}")
            return
        if not mmproj_path or not os.path.exists(mmproj_path):
            QMessageBox.warning(self, "错误", f"视觉投影器文件不存在: {mmproj_path}")
            return
        
        # 🔔 视觉模型注意事项弹窗
        mem_mb = info.get("memory_required_mb", 4096)
        reply = QMessageBox.information(
            self, "📷 视觉模型注意事项",
            f"即将加载视觉模型：{info.get('name', mid)}\n\n"
            f"⚠ 注意事项：\n"
            f"  1. 视觉模型需要约 {mem_mb}MB 内存\n"
            f"  2. 如果同时加载文本模型，总内存需求会更大\n"
            f"  3. 加载过程可能需要 30秒-2分钟\n"
            f"  4. 加载后遇到图片将自动使用本地视觉模型识别\n"
            f"  5. 无需网络连接，完全离线运行\n\n"
            f"确定要加载吗？",
            QMessageBox.Ok | QMessageBox.Cancel
        )
        if reply != QMessageBox.Ok:
            return
        
        self._load_vision_model(model_path, mmproj_path, handler_type)
    
    def _on_vision_unload(self):
        from lib.local_model import get_vision_model
        reply = QMessageBox.question(self, "确认卸载", "确定要卸载视觉模型吗？", QMessageBox.Yes | QMessageBox.No)
        if reply == QMessageBox.Yes:
            get_vision_model().unload()
            self.vision_status_label.setText("⚠ 未加载")
            self.vision_status_label.setStyleSheet("color: #ffcc00; font-size: 12px; font-weight: bold;")
            self.vision_unload_btn.setEnabled(False)
            self.vision_test_btn.setEnabled(False)
    
    def _on_vision_test(self):
        self.vision_test_btn.setEnabled(False)
        self.vision_test_btn.setText("测试中...")
        self._vision_test_thread = TestVisionModelThread()
        self._vision_test_thread.finished.connect(self._on_vision_test_finished)
        self._vision_test_thread.start()
    
    def _on_vision_test_finished(self, success, message):
        self.vision_test_btn.setEnabled(True)
        self.vision_test_btn.setText("🔗 测试")
        if success:
            QMessageBox.information(self, "测试成功", message)
        else:
            QMessageBox.warning(self, "测试失败", message)
    
    def _on_vision_delete(self):
        mid, info = self._get_selected_vision_model()
        if not info or not info.get("downloaded"):
            QMessageBox.warning(self, "提示", "该模型未下载")
            return
        
        from lib.local_model import AVAILABLE_VISION_MODELS
        vm_info = AVAILABLE_VISION_MODELS.get(mid, {})
        model_file = info.get("model_path", "")
        mmproj_file = info.get("mmproj_path", "")
        
        reply = QMessageBox.question(self, "确认删除",
                                     f"确定要删除视觉模型 {info.get('name', mid)} 吗？\n"
                                     f"将删除主模型和视觉投影器两个文件。\n此操作不可恢复！",
                                     QMessageBox.Yes | QMessageBox.No)
        if reply == QMessageBox.Yes:
            try:
                from lib.local_model import get_vision_model
                vm = get_vision_model()
                if vm.is_loaded() and vm.get_model_path() == model_file:
                    vm.unload()
                    self.vision_status_label.setText("⚠ 未加载")
                    self.vision_status_label.setStyleSheet("color: #ffcc00; font-size: 12px; font-weight: bold;")
                    self.vision_unload_btn.setEnabled(False)
                    self.vision_test_btn.setEnabled(False)
                
                if model_file and os.path.exists(model_file):
                    os.remove(model_file)
                if mmproj_file and os.path.exists(mmproj_file):
                    os.remove(mmproj_file)
                
                QMessageBox.information(self, "成功", "视觉模型文件已删除。")
                self.refresh_vision_models()
            except Exception as e:
                QMessageBox.warning(self, "失败", f"删除失败: {str(e)}")
    
    def _start_vision_download(self, model_id):
        if self._download_thread and self._download_thread.isRunning():
            QMessageBox.warning(self, "下载中", "已有下载任务正在进行")
            return
        
        from lib.local_model import AVAILABLE_VISION_MODELS
        info = AVAILABLE_VISION_MODELS.get(model_id, {})
        
        reply = QMessageBox.question(
            self, "确认下载",
            f"下载视觉模型 {info.get('name', model_id)}？\n\n"
            f"将下载两个文件（主模型 + 视觉投影器），总大小约 {info.get('size_mb', 0)}MB\n"
            f"下载可能需要较长时间，请耐心等待。",
            QMessageBox.Yes | QMessageBox.No
        )
        if reply != QMessageBox.Yes:
            return
        
        self.progress_frame.show()
        self.progress_bar.setValue(0)
        self.progress_label.setText(f"正在下载视觉模型 {info.get('name', model_id)}...")
        self.cancel_btn.setEnabled(True)
        self._current_download_model_id = model_id
        self._current_download_is_vision = True
        
        self._download_thread = VisionDownloadThread(model_id)
        self._download_thread.progress.connect(self._on_download_progress)
        self._download_thread.finished.connect(self._on_vision_download_finished)
        self._download_thread.start()
    
    def _on_vision_download_finished(self, success, message):
        self.cancel_btn.setEnabled(False)
        mid = self._current_download_model_id
        self._current_download_model_id = None
        
        if success:
            self.progress_label.setText("✅ 视觉模型下载完成！")
            self.progress_bar.setValue(100)
            self.refresh_vision_models()
            
            # 不自动加载视觉模型，让用户手动点击加载（因为需要看注意事项弹窗）
            QMessageBox.information(self, "下载完成",
                                   "视觉模型下载成功！\n\n"
                                   "请点击「加载」按钮加载视觉模型。\n"
                                   "加载后遇到图片时将自动使用本地视觉模型识别。")
        else:
            self.progress_label.setText(f"❌ {message}")
            if "取消" not in message:
                QMessageBox.warning(self, "下载失败", message)
        
        QTimer.singleShot(5000, lambda: self.progress_frame.hide())
    
    def _load_vision_model(self, model_path, mmproj_path, handler_type):
        from lib.local_model import is_llama_cpp_available
        if not is_llama_cpp_available():
            QMessageBox.warning(self, "依赖缺失", "llama-cpp-python 未安装")
            return
        if self._load_thread and self._load_thread.isRunning():
            QMessageBox.warning(self, "加载中", "模型正在加载中...")
            return
        
        self.vision_status_label.setText("⏳ 加载中...")
        self.vision_status_label.setStyleSheet("color: #ffcc00; font-size: 12px; font-weight: bold;")
        self.vision_load_btn.setEnabled(False)
        
        self._load_thread = LoadVisionModelThread(
            model_path, mmproj_path, handler_type,
            self.spin_n_ctx.value(), self.spin_n_threads.value(), self.spin_n_gpu.value()
        )
        self._load_thread.progress.connect(lambda msg: self.vision_status_label.setText(f"⏳ {msg}"))
        self._load_thread.finished.connect(self._on_vision_load_finished)
        self._load_thread.start()
    
    def _on_vision_load_finished(self, success, message):
        self.vision_load_btn.setEnabled(True)
        if success:
            from lib.local_model import get_vision_model
            vm = get_vision_model()
            path = vm.get_model_path()
            name = os.path.basename(path) if path else "未知"
            self.vision_status_label.setText(f"✅ {name}")
            self.vision_status_label.setStyleSheet("color: #19AF39; font-size: 12px; font-weight: bold;")
            self.vision_unload_btn.setEnabled(True)
            self.vision_test_btn.setEnabled(True)
            QMessageBox.information(self, "加载成功",
                                   "📷 视觉模型加载成功！\n\n"
                                   "遇到图片文件时将自动使用本地视觉模型识别，无需网络连接。")
            self.vision_model_loaded.emit(path)
        else:
            self.vision_status_label.setText(f"❌ {message}")
            self.vision_status_label.setStyleSheet("color: #ff6b6b; font-size: 12px; font-weight: bold;")
            QMessageBox.warning(self, "加载失败", message)
    
    # ═══════════════════════════════════════════
    # 共用方法
    # ═══════════════════════════════════════════
    
    def _on_download_progress(self, downloaded, total, speed):
        if total > 0:
            percent = int(downloaded * 100 / total)
            self.progress_bar.setValue(percent)
            downloaded_mb = downloaded / 1024 / 1024
            total_mb = total / 1024 / 1024
            self.progress_label.setText(
                f"已下载: {downloaded_mb:.1f}MB / {total_mb:.1f}MB ({percent}%) - {speed:.1f}MB/s"
            )
    
    def _cancel_download(self):
        if self._download_thread and self._download_thread.isRunning():
            self._download_thread.stop()
            self.progress_label.setText("正在取消下载...")
    
    def check_llama_cpp(self):
        try:
            from lib.local_model import is_llama_cpp_available
            available = is_llama_cpp_available()
        except Exception as e:
            logger.warning(f"检查 llama-cpp-python 失败: {e}")
            available = False
        
        if not available:
            self.text_status_label.setText("⚠ llama-cpp-python 未安装或 DLL 缺失")
            self.text_status_label.setStyleSheet("color: #ff6b6b; font-size: 12px; font-weight: bold;")
            self.vision_status_label.setText("⚠ llama-cpp-python 未安装或 DLL 缺失")
            self.vision_status_label.setStyleSheet("color: #ff6b6b; font-size: 12px; font-weight: bold;")
            self.text_load_btn.setEnabled(False)
            self.vision_load_btn.setEnabled(False)
    
    def _save_settings(self):
        try:
            import json
            from lib.local_model import CONFIG_FILE
            config = {}
            if CONFIG_FILE.exists():
                with open(CONFIG_FILE, "r", encoding="utf-8") as f:
                    config = json.load(f)
            config["n_ctx"] = self.spin_n_ctx.value()
            config["n_threads"] = self.spin_n_threads.value()
            config["n_gpu_layers"] = self.spin_n_gpu.value()
            with open(CONFIG_FILE, "w", encoding="utf-8") as f:
                json.dump(config, f, ensure_ascii=False, indent=2)
            QMessageBox.information(self, "保存成功", "设置已保存！下次加载模型时生效。")
        except Exception as e:
            QMessageBox.warning(self, "保存失败", f"保存设置失败: {str(e)}")
    
    def _load_settings(self):
        try:
            from lib.local_model import LocalModel
            config = LocalModel.load_config()
            if config:
                if "n_ctx" in config:
                    self.spin_n_ctx.setValue(config["n_ctx"])
                if "n_threads" in config:
                    self.spin_n_threads.setValue(config["n_threads"])
                if "n_gpu_layers" in config:
                    self.spin_n_gpu.setValue(config["n_gpu_layers"])
        except Exception:
            pass
    
    def center_on_screen(self):
        from PySide6.QtWidgets import QApplication
        screen = QApplication.primaryScreen()
        if screen:
            sg = screen.geometry()
            self.move((sg.width() - self.width()) // 2, (sg.height() - self.height()) // 2)
    
    def closeEvent(self, event):
        if self._download_thread and self._download_thread.isRunning():
            reply = QMessageBox.question(self, "正在下载",
                                        "下载进行中，关闭将取消下载。确定？",
                                        QMessageBox.Yes | QMessageBox.No)
            if reply == QMessageBox.Yes:
                self._download_thread.stop()
                self._download_thread.wait(5000)
                event.accept()
            else:
                event.ignore()
                return
        event.accept()
